% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/count.R
\name{count}
\alias{count}
\alias{count_R}
\alias{count_IR}
\alias{count_I}
\alias{count_SI}
\alias{count_S}
\alias{count_df}
\title{Count isolates}
\source{
Wickham H. \strong{Tidy Data.} The Journal of Statistical Software, vol. 59, 2014. \url{http://vita.had.co.nz/papers/tidy-data.html}
}
\usage{
count_R(...)

count_IR(...)

count_I(...)

count_SI(...)

count_S(...)

count_df(data, translate_ab = getOption("get_antibiotic_names",
  "official"))
}
\arguments{
\item{...}{one or more vectors (or columns) with antibiotic interpretations. They will be transformed internally with \code{\link{as.rsi}} if needed. Use multiple columns to calculate (the lack of) co-resistance: the probability where one of two drugs have a resistant or susceptible result. See Examples.}

\item{data}{a \code{data.frame} containing columns with class \code{rsi} (see \code{\link{as.rsi}})}

\item{translate_ab}{a column name of the \code{\link{antibiotics}} data set to translate the antibiotic abbreviations to, using \code{\link{abname}}. This can be set with \code{\link{getOption}("get_antibiotic_names")}.}
}
\value{
Integer
}
\description{
These functions can be used to count resistant/susceptible microbial isolates. All functions support quasiquotation with pipes, can be used in \code{dplyr}s \code{\link[dplyr]{summarise}} and support grouped variables, see \emph{Examples}.

\code{count_R} and \code{count_IR} can be used to count resistant isolates, \code{count_S} and \code{count_SI} can be used to count susceptible isolates.\cr
}
\details{
\strong{Remember that you should filter your table to let it contain only first isolates!} Use \code{\link{first_isolate}} to determine them in your data set.

These functions are meant to count isolates. Use the \code{\link{portion}_*} functions to calculate microbial resistance.

\code{count_df} takes any variable from \code{data} that has an \code{"rsi"} class (created with \code{\link{as.rsi}}) and counts the amounts of R, I and S. The resulting \emph{tidy data} (see Source) \code{data.frame} will have three rows (S/I/R) and a column for each variable with class \code{"rsi"}.
}
\examples{
# septic_patients is a data set available in the AMR package. It is true, genuine data.
?septic_patients

# Count resistant isolates
count_R(septic_patients$amox)
count_IR(septic_patients$amox)

# Or susceptibile isolates
count_S(septic_patients$amox)
count_SI(septic_patients$amox)

# Since n_rsi counts available isolates, you can
# calculate back to count e.g. non-susceptible isolates.
# This results in the same:
count_IR(septic_patients$amox)
portion_IR(septic_patients$amox) * n_rsi(septic_patients$amox)

library(dplyr)
septic_patients \%>\%
  group_by(hospital_id) \%>\%
  summarise(R = count_R(cipr),
            I = count_I(cipr),
            S = count_S(cipr),
            n = n_rsi(cipr), # the actual total; sum of all three
            total = n())     # NOT the amount of tested isolates!

# Count co-resistance between amoxicillin/clav acid and gentamicin,
# so we can see that combination therapy does a lot more than mono therapy.
# Please mind that `portion_S` calculates percentages right away instead.
count_S(septic_patients$amcl)   # S = 1056 (67.3\%)
n_rsi(septic_patients$amcl)     # n = 1570

count_S(septic_patients$gent)   # S = 1363 (74.0\%)
n_rsi(septic_patients$gent)     # n = 1842

with(septic_patients,
     count_S(amcl, gent))       # S = 1385 (92.1\%)
with(septic_patients,           # n = 1504
     n_rsi(amcl, gent))

# Get portions S/I/R immediately of all rsi columns
septic_patients \%>\%
  select(amox, cipr) \%>\%
  count_df(translate = FALSE)

# It also supports grouping variables
septic_patients \%>\%
  select(hospital_id, amox, cipr) \%>\%
  group_by(hospital_id) \%>\%
  count_df(translate = FALSE)

}
\seealso{
\code{\link{portion}_*} to calculate microbial resistance and susceptibility.\cr
\code{\link{n_rsi}} to count all cases where antimicrobial results are available.
}
\keyword{antibiotics}
\keyword{isolate}
\keyword{isolates}
\keyword{resistance}
\keyword{rsi}
\keyword{susceptibility}
