% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/coint_eq.R
\name{coint_eq}
\alias{coint_eq}
\alias{coint_eq.recm}
\alias{coint_eq.default}
\title{Cointegrating equation (long-run level relationship)}
\usage{
coint_eq(object, case)

\method{coint_eq}{recm}(object, ...)

\method{coint_eq}{default}(object, case)
}
\arguments{
\item{object}{An object of \code{\link[base]{class}} 'ardl', 'uecm' or
'recm'.}

\item{case}{An integer from 1-5 or a character string specifying whether the
'intercept' and/or the 'trend' have to participate in the long-run level
relationship (cointegrating equation) (see section 'Cases' below). If the
input object is of class 'recm', \code{case} is not needed as the model is
already under a certain case.}

\item{...}{Currently unused argument.}
}
\value{
\code{coint_eq} returns an numeric vector containing the
  cointegrating equation.
}
\description{
Creates the cointegrating equation (long-run level relationship) providing an
'ardl', 'uecm' or 'recm' model.
}
\section{Cases}{
 According to \cite{Pesaran et al. (2001)}, we distinguish the
long-run relationship (cointegrating equation) (and thus the bounds-test and
the Restricted ECMs) between 5 different cases. These differ in terms of
whether the 'intercept' and/or the 'trend' are restricted to participate in
the long-run relationship or they are unrestricted and so they participate in
the short-run relationship.

\describe{
  \item{Case 1:}{\itemize{
    \item No \emph{intercept} and no \emph{trend}.
    \item \code{case} inputs: 1 or "n" where "n" stands for none.}}
  \item{Case 2:}{\itemize{
    \item Restricted \emph{intercept} and no \emph{trend}.
    \item \code{case} inputs: 2 or "rc" where "rc" stands for restricted
     constant.}}
  \item{Case 3:}{\itemize{
    \item Unrestricted \emph{intercept} and no \emph{trend}.
    \item \code{case} inputs: 3 or "uc" where "uc" stands for unrestricted
    constant.}}
  \item{Case 4:}{\itemize{
    \item Unrestricted \emph{intercept} and restricted \emph{trend}.
    \item \code{case} inputs: 4 or "ucrt" where "ucrt" stands for
    unrestricted constant and restricted trend.}}
  \item{Case 5:}{\itemize{
    \item Unrestricted \emph{intercept} and unrestricted \emph{trend}.
    \item \code{case} inputs: 5 or "ucut" where "ucut" stands for
     unrestricted constant and unrestricted trend.}}
}

Note that you can't restrict (or leave unrestricted) a parameter that doesn't
exist in the input model. For example, you can't compute \code{recm(object,
case=3)} if the object is an ARDL (or UECM) model with no intercept. The same
way, you can't compute \code{bounds_f_test(object, case=5)} if the object is
an ARDL (or UECM) model with no linear trend.
}

\section{References}{
 Pesaran, M. H., Shin, Y., & Smith, R. J. (2001). Bounds
  testing approaches to the analysis of level relationships. \emph{Journal of
  Applied Econometrics}, 16(3), 289-326
}

\examples{
data(denmark)
library(zoo) # for cbind.zoo()

## Estimate the Cointegrating Equation of an ARDL(3,1,3,2) model -------

# From an ARDL model (under case 2, restricted constant)
ardl_3132 <- ardl(LRM ~ LRY + IBO + IDE, data = denmark, order = c(3,1,3,2))
ce2_ardl <- coint_eq(ardl_3132, case = 2)

# From an UECM (under case 2, restricted constant)
uecm_3132 <- uecm(ardl_3132)
ce2_uecm <- coint_eq(uecm_3132, case = 2)

# From a RECM (under case 2, restricted constant)
# Notice that if a RECM has already been estimated under a certain case,
# the 'coint_eq()' can't be under different case, so no 'case' argument needed.
recm_3132 <- recm(uecm_3132, case = 2)
# The RECM is already under case 2, so the 'case' argument is no needed
ce2_recm <- coint_eq(recm_3132)

identical(ce2_ardl, ce2_uecm, ce2_recm)

## Check for a degenerate level relationship ---------------------------

# The bounds F-test under both cases reject the Null Hypothesis of no level relationship.
bounds_f_test(ardl_3132, case = 2)
bounds_f_test(ardl_3132, case = 3)

# The bounds t-test also rejects the NUll Hypothesis of no level relationship.
bounds_t_test(ardl_3132, case = 3)

# But when the constant enters the long-run equation (case 3)
# this becomes a degenerate relationship.
ce3_ardl <- coint_eq(ardl_3132, case = 3)

plot_lr(ardl_3132, coint_eq = ce2_ardl, show.legend = TRUE)

plot_lr(ardl_3132, coint_eq = ce3_ardl, show.legend = TRUE)
plot_lr(ardl_3132, coint_eq = ce3_ardl, facets = TRUE, show.legend = TRUE)
}
\seealso{
\code{\link{plot_lr}} \code{\link{ardl}} \code{\link{uecm}} \code{\link{recm}}
  \code{\link{bounds_f_test}} \code{\link{bounds_t_test}}
}
\author{
Kleanthis Natsiopoulos, \email{klnatsio@gmail.com}
}
\keyword{ts}
