% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/generic_support.r
\name{rasterToBitmap}
\alias{rasterToBitmap}
\title{Convert a grDevices \code{raster} object into binary bitmap data}
\usage{
rasterToBitmap(x, depth = 3, interleaved = T, indexing = index.colours)
}
\arguments{
\item{x}{A raster object created with \code{\link[grDevices]{as.raster}} which
needs to be converted into bitmap data. It is also posible to let \code{x} be
a matrix of \code{character}s, representing colours.}

\item{depth}{The colour depth of the bitmap image. The image will be composed
of \code{2^depth} indexed colours.}

\item{interleaved}{A \code{logical} value, indicating whether the bitmap needs to be
interleaved. An interleaved bitmap image stores each consecutive bitmap layer per
horizontal scanline.}

\item{indexing}{A function that accepts two arguments: \code{x} (a grDevices
\code{raster} object); \code{length.out}, a numeric value indicating the
desired size of the palette (i.e., the number of colours). It should return
a matrix with numeric palette indices (ranging from 1 up to the number of
colours in the palette). The result should have an attribute named `palette' that
contains the colours that correspond with the index numbers. The result should
also carry an attribute with the name `transparent', with a single numeric value
representing which colour in the palette should be treated as transparent (or
\code{NA} when no transparency is required). By default the
function \code{\link{index.colours}} is used. You are free to provide
a customised version of this function (see examples).}
}
\value{
The bitmap is returned as a \code{vector} of \code{logical} values.
The \code{logical} values reflect the bits for each bitplane. The palette used
for the bitmap is returned as attribute to the \code{vector}. There will also be
an attribute called `transparent'. This will hold a numeric index corresponding
with the colour in the palette that will be treated as transparent. It will be
\code{NA} when transparency is not used.
}
\description{
Converts an image represented by a grDevices \code{raster} object into binary
(Amiga) bitmap data.
}
\details{
Images represented by grDevices \code{raster} objects are virtually true colour (24 bit
colour depth) and an alpha layer (transparency). On the early Amiga's the chipset
(in combination with memory restrictions) only allowed images with indexed
palettes. The colour depth was 12 bit with the original chipset and the number
of colours allowed in a palette also depended on the chipset. This function
will allow you to convert a \code{raster} object into binary bitmap data with
an indexed palette. This means that the image is converted in a lossy way
(information will be lost). So don't expect the result to have the same quality as
the original image.
}
\examples{
\dontrun{
## first: Let's make a raster out of the 'volcano' data, which we can use in the example:
volcano.raster <- as.raster(t(matrix(terrain.colors(1 + diff(range(volcano)))[volcano -
  min(volcano) + 1], nrow(volcano))))

## convert the raster into binary (logical) bitmap data:
volcano.bm <- rasterToBitmap(volcano.raster)

## The palette for the indexed colours of the generated bitmap is returned as
## attribute. There is no transparency is the image:
attributes(volcano.bm)

## We can also include a custom function for colour quantisation. Let's include
## some dithering:
volcano.dither <- rasterToBitmap(volcano.raster,
                                 indexing = function(x, length.out) {
                                   index.colours(x, length.out,
                                                 dither = "floyd-steinberg")
                                 })

## You can also use a custom indexing function to force a specified palette,
## in this case black and white:
volcano.bw <- rasterToBitmap(volcano.raster,
                             indexing = function(x, length.out) {
                               index.colours(x, length.out,
                                             palette = c("black", "white"),
                                             dither = "floyd-steinberg")
                             })

}
}
\seealso{
Other raster.operations: \code{\link{as.raster}},
  \code{\link{bitmapToRaster}}, \code{\link{dither}},
  \code{\link{index.colours}}, \code{\link{rasterToIFF}}
}
\author{
Pepijn de Vries
}
