#' Bayesian analysis for age estimation of 14C age of various samples
#'
#' This function calibrate 14C age of samples to get an age (years).
#' The user can choose one of the following radiocarbon calibration curve:
#' Northern or Sourthen Hemisphere or marine atmospheric. It must be the same curve for all samples.
#'
#' @param Data_C14Cal numeric vector: corresponding to 14C age estimate.
#' @param Data_SigmaC14Cal numeric vector: correponding to the error of 14C age estimates.
#' @param SampleNames character vector: names of sample. The length of this vector is equal to \code{Nb_sample}.
#' @param Nb_sample integer: number of samples.
#' @param PriorAge numeric vector (with default): lower and upper bounds for age parameter of each sample in years (not in ka).
#' Note that, \code{length(PriorAge)=2*Nb_sample}
#' and \code{PriorAge[2i-1,2i]} correponds to the lower and upper bounds of sample whose number ID is equal to \code{i}.
#' @param SavePdf boolean (with default): if TRUE save graphs in pdf file named \code{OutputFileName} in folder \code{OutputFilePath}.
#' @param OutputFileName character (with default): name of the pdf file that will be generated by the function if \code{SavePdf}=TRUE,
#' \code{length(OutputFileName)=3}, see \bold{PLOT OUTPUT} in \bold{Value} section for more informations.
#' @param OutputFilePath character (with default): path to the pdf file that will be generated by the function if \code{SavePdf}=TRUE.
#' If it is not equal to "", it must be terminated by "/".
#' @param SaveEstimates boolean (with default): if TRUE save Bayes estimates, credible interval at level 68\% and 95\% and
#' the result of the gelman en Rudin test of convergency, in a csv table named \code{OutputFileName} in folder \code{OutputFilePath}.
#' @param OutputTableName character (with default): name of the table that will be generated by the function if \code{SaveEstimates}=TRUE.
#' @param OutputTablePath character (with default): path to the table that will be generated by the function if \code{SaveEstimates}=TRUE.
#' If it is not equal to "", it must be terminated by "/".
#' @param StratiConstraints numeric matrix or character(with default): input object for the statigraphic relation between samples.
#' If there is stratigraphic relation between samples see the details section for instructions regarding how to correctly fill \code{StratiConstraints};
#' the user can refer to a matrix (numeric matrix) or to a csv file (character).
#' If there is no stratigraphic relation default value is suitable.
#' @param sepSC character (with default): if \code{StratiConstraints} is character,
#' indicate column separator in \code{StratiConstraints} csv file.
#' @param Model character (with default): if \bold{"full"}, error on estimate calibration curve is taken account.
#' If \bold{"naive"} this error is not taken account in the age estimate.
#' @param CalibrationCurve character (with default): calibration curve choosen. Allowed inputs are
#' \itemize{
#'   \item \bold{"AtmosphericNorth"} for Northern Hemisphere atmospheric radiocarbon calibration curve,
#'   \item \bold{"Marine"} for Marine radiocarbon calibration curve,
#'   \item \bold{"AtmosphericSouth"} for Southern Hemisphere atmospheric radiocarbon calibration curve
#'   \item \bold{a csv file, with tree columns, the first column is dedicated to "Cal.BP", the second to "X14C.age", the third to "Error".
#'   The decimal of this file must be a dot, and the separator must be a comma. }
#' }
#' @param Iter integer (with default): number of iterations for the MCMC computation (for more information see \code{\link{jags.model}}).
#' @param t integer (with default): 1 every \code{t} iterations of the MCMC is considered for sampling the posterior distribution
#' (for more information see \code{\link{jags.model}}).
#' @param Nb_chaines integer (with default): number of independent chains for the model (for more information see \code{\link{jags.model}}).
#'
#' @details
#' \bold{** How to fill} \code{StratiConstraints} \bold{? **}\cr
#'
#' If there is stratigraphic relations between samples, \bold{14C age in Data_C14Cal must be ordered by order of increasing ages}.
#'
#' The user can fill the \code{StratiConstraints} matrix as follow.
#' \enumerate{
#'  \item \bold{Size of the matrix}: row number of \code{StratiConstraints} matrix is equal to \code{Nb_sample+1},
#' and column number is equal to \code{Nb_sample}.
#'  \item \bold{First line of the matrix}:
#' for all \code{i in {1,...,Nb_Sample}}, \code{StratiConstraints[1,i]=1} that means the lower bound of the sample age (given in \code{PriorAge[2i-1]})
#' for the sample whose number ID is equal to \code{i}, is taken into account.
#'  \item \bold{Sample relations}: for all  \code{j in {2,...,Nb_Sample+1}} and all \code{i in {j,...,Nb_Sample}},
#' \code{StratiConstraints[j,i]=1} if sample age whose number ID is equal to \code{j-1} is lower than sample age whose number ID is equal to \code{i}.
#' Otherwise, \code{StratiConstraints[j,i]=0}.
#' }
#' Note that \code{StratiConstraints_{2:Nb_sample+1,1:Nb_sample}} is a upper triangular matrix.
#'
#' The user can also use \code{\link{SCMatrix}} or \code{\link{SC_Ordered}} (if all samples are ordered) functions
#' to construct the \code{StratiConstraints} matrix.
#'
#' The user can also refer to a .csv file that containts the relation between samples as defined above.
#' The user must take care about the separator used in the csv file using the argument \code{sepSC}.\cr
#'
#' \bold{** More precision on \code{Model} **}\cr
#'
#' We propose two models "full" or "naive". If \code{Model='full'} that means measurement error and error on calibration curve are taken account in
#' the Bayesian model; if \code{Model="naive"} that means only error on measurement are taken account in the mode.
#'
#' More precisely, the model considered here, as the one developped by Christen, JA (1994), assume multiplicative effect of errors to address the
#' problem of outliers. In addition, to not penalyse variables that are not outliers and damage theirs estimation,
#' we introduce a structure of mixture, that means only variable that are considered as outlier have in addition a multiplicative error.
#'
#' @return
#' \bold{NUMERICAL OUTPUT}
#'
#' \enumerate{
#' \item \bold{A list containing the following objects:}
#'  \itemize{
#'   \item \bold{Sampling}: that corresponds to a sample of the posterior distributions of the age parameters;
#'   \item \bold{Outlier}: stating the names of samples that are considered as outliers;
#'   \item \bold{Model}: stating which model was chosen (\code{"full"} or \code{"naive"});
#'   \item \bold{CalibrationCurve}: stating which radiocarbon calibration curve was chosen;
#'   \item \bold{PriorAge}: stating the priors used for the age parameter;
#'   \item \bold{StratiConstraints}: stating the stratigraphic relations between samples considered in the model.
#'  }
#' \item\bold{The Gelman and Rudin test of convergency}: print the result of the Gelman and Rudin test of convergency for the age estimate for each sample.
#' A result close to one is expected.\cr
#' In addition, the user must visually assess the convergency of the trajectories by looking at the graph
#' generated by the function (see \bold{PLOT OUTPUT} for more informations).\cr
#' If both convergencies (Gelman and Rudin test and plot checking) are satisfactory,
#' the user can consider the estimates as valid.
#' Otherwise, the user may try increasing the number of MCMC interations (\code{Iter})
#' or being more precise if it is possible on the \code{PriorAge} parameter to reach convergency.
#' \item \bold{Credible intervals and Bayes estimates}: prints the Bayes estimates, the credible intervals at 95\% and 68\% for
#' the age parameters for each sample.
#' }
#'
#' \bold{PLOT OUTPUT}
#'
#' \enumerate{
#'  \item\bold{MCMC trajectories}: A graph with the MCMC trajectories and posterior distributions of the age parameter is displayed. \cr
#' On each line, the plot on the left represents the MCMC trajectories, and the one on the right the posterior distribution of the parameter.
#'  \item \bold{Summary of sample age estimates}: plot credible intervals and Bayes estimate of each sample age on one graph.
#' }
#'
#' To give the results in a publication, we recommend to give the Bayes estimate of the parameters as well as the credible interval at 95\% or 68\%.
#'
#' @author Claire Christophe, Anne Philippe, Guillaume Guerin
#'
#' @seealso
#' \code{\link{rjags}}, \code{\link{MCMC_plot}}, \code{\link{SCMatrix}}
#'
#' @references
#' Christen, JA (1994). Summarizing a set of radiocarbon determinations: a robust approach. Applied Statistics, 489-503.
#'
#' Reimer PJ, Bard E, Bayliss A, Beck JW, Blackwell PC, Bronl Ramsey C, Buck CE, Cheng H, Edwards RL, Friedrich M,
#' Grootes PM, Guilderson TP, Haflidason H, Hajdas I, Hatte C, Heaton TJ, Hoffmann DL, Hogg AG, Hughen KA, Kaiser KF, Kromer B,
#' Manning SW, Niu M, Reimer RW, Richards DA, Scott EM, Southon JR, Staff RA, Turney CSM, van der Plicht J. 2013.
#' IntCal13 ans Marine13 radiocarbon age calibration curves 0-50000 years cal BP. Radiocarbon 55(4)=1869-1887.
#'
#' Hogg AG, Hua Q, Blackwell PG, Niu M, Buck CE, Guilderson TP, Heaton TJ, Palmer JG, Reimer PJ, Reimer RW, Turney CSM, Zimmerman SRH.
#' 2013. SHCal13 Southern Hemisphere calibration, 0-50000 years cal BP. Radiocarbon 55(4):1889-1903
#'
#'
#' @examples
#' ## Load data
#' data(DATA_C14,envir = environment())
#' C14Cal=DATA_C14$C14[,1]
#' SigmaC14Cal=DATA_C14$C14[,2]
#' Names=DATA_C14$Names
#' nb_sample=length(Names)
#' ## Age computation of samples without stratigraphic relations
#'
#' Age=AgeC14_Computation(Data_C14Cal=C14Cal,Data_SigmaC14Cal=SigmaC14Cal,
#'    SampleNames=Names,Nb_sample=nb_sample,PriorAge=rep(c(20000,60000),nb_sample),Iter=500)
#'
#' @export
AgeC14_Computation<-function(Data_C14Cal,Data_SigmaC14Cal,SampleNames,Nb_sample,
                             PriorAge=rep(c(10000,50000),Nb_sample),
                             SavePdf=FALSE,
                             OutputFileName=c('MCMCplot',"HPD_Cal14CCurve","summary"),OutputFilePath=c(""),
                             SaveEstimates=FALSE,OutputTableName=c("DATA"),OutputTablePath=c(''),
                             StratiConstraints=c(),sepSC=c(','),
                             Model=c("full"),CalibrationCurve=c("AtmosphericNorth"),
                             Iter=50000,t=5,Nb_chaines=3){

   #--- BUG file selection
   Model_AgeC14<-0
   data(Model_AgeC14,envir = environment())

   #--- Calibration curve
   TableauCalib=c()
   if(CalibrationCurve=="AtmosphericNorth"){
     AtmosphericNorth_CalC14<-0
     data(AtmosphericNorth_CalC14,envir = environment())
     TableauCalib=AtmosphericNorth_CalC14
   }else{if(CalibrationCurve=="AtmosphericSouth"){
     AtmosphericSouth_CalC14<-0
     data(AtmosphericSouth_CalC14,envir = environment())
     TableauCalib=AtmosphericSouth_CalC14
   }else{if(CalibrationCurve=="Marine"){
     Marine_CalC14<-0
     data(Marine_CalC14,envir = environment())
     TableauCalib=Marine_CalC14
   }else{
     TableauCalib=read.csv(file=CalibrationCurve,sep=",",dec=".")
   }}}
   AgeBP=rev(TableauCalib[,1])
   CalC14=rev(TableauCalib[,2])
   SigmaCalC14=rev(TableauCalib[,3])

   # #--- Calibration curve
   # TableauCalib=read.csv(file=paste("inst/extdata/",CalibrationCurve,"_CalC14.csv",sep=""),sep=",",dec=".")
   # AgeBP=rev(TableauCalib[,1])
   # CalC14=rev(TableauCalib[,2])
   # SigmaCalC14=rev(TableauCalib[,3])

   #--- StratiConstraints matrix
   if(length(StratiConstraints)==0){
     StratiConstraints=matrix(data=c(rep(1,Nb_sample),rep(0,Nb_sample*Nb_sample)),ncol=Nb_sample,nrow = (Nb_sample+1),byrow = T)
   }else{
     if(is(StratiConstraints)[1]=="character"){
       SCMatrix=read.csv(StratiConstraints,sep=sepSC)
       StratiConstraints=as.matrix(SCMatrix)
     }
   }

   if(Model=="full"){
    dataList = list('X'=Data_C14Cal,"sigma"=Data_SigmaC14Cal,'N'= Nb_sample,
                   "xTableauCalib"=AgeBP,"yTableauCalib"=CalC14,"zTableauCalib"=SigmaCalC14,
                   "xbound"=PriorAge,"StratiConstraints"=StratiConstraints)
   }else{
     dataList = list('X'=Data_C14Cal,"sigma"=Data_SigmaC14Cal,'N'= Nb_sample,
                     "xTableauCalib"=AgeBP,"yTableauCalib"=CalC14,
                     "xbound"=PriorAge,"StratiConstraints"=StratiConstraints)
   }
   jags <- rjags::jags.model(textConnection(Model_AgeC14[[Model]]),data = dataList,n.chains = Nb_chaines,n.adapt= Iter)
   update(jags,Iter)
   echantillon = rjags::coda.samples(jags,c('Age','Z'),min(Iter,10000),thin=t)
   U=summary(echantillon)

   Sample=echantillon[[1]]
   for(i in 2:Nb_chaines){
     Sample=rbind(Sample,echantillon[[i]])
   }

   nom=c()
   for(i in 1:Nb_sample){
     nom=c(nom,paste("A_",SampleNames[i],sep=""))
   }
   r=Nb_sample%%6
   q=Nb_sample%/%6
   if(q>0){
     for (l in 1:q){
       MCMC_plot(Sample[,(6*(l-1)+1):(6*l)],
                 length(echantillon[[1]][,1]),
                 SampleNames=c(""),
                 Nb_sample=1,
                 Nb_chaines=Nb_chaines,
                 value=seq(0,5,1),
                 param=nom[(6*(l-1)+1):(6*l)])
     }
   }
   if(r>0){
   MCMC_plot(as.matrix(Sample[,(6*q+1):Nb_sample]),
             length(echantillon[[1]][,1]),
             SampleNames=c(""),
             Nb_sample=1,
             Nb_chaines=Nb_chaines,
             value=seq(0,(r-1),1),
             param=nom[(6*q+1):Nb_sample])
   }
   if(SavePdf==TRUE){
     dev.print(pdf,file=paste(OutputFilePath,OutputFileName[1],'.pdf',sep=""),width=8,height=10)
   }

   Outlier=SampleNames[which(U$statistics[(Nb_sample+1):(2*Nb_sample),1]<1.5)]

   ##- Gelman and rudin test of convergency of the MCMC
   CV=gelman.diag(echantillon,multivariate=FALSE)
   cat(paste("\n\n>> Convergencies of MCMC of Age parameter <<\n"))
   cat("----------------------------------------------\n")
   cat(paste("Sample name ", " Bayes estimate ", " Uppers credible interval\n"))
   for(i in 1:Nb_sample){
     #cat(paste(" Sample name: ", SampleNames[i],"\n"))
     #cat("---------------------\n")
     cat(paste(paste("A_",SampleNames[i],sep=""),"\t",round(CV$psrf[i,1],2),"\t\t",round(CV$psrf[i,2],2),"\n"))
   }

   cat("\n\n________________________________________________________________________________\n")
   cat(" *** WARNING: following informations are only valid if MCMC chains converged  ***\n")
   cat("________________________________________________________________________________\n")

   # Matrix of results
   rnames=c()
   for(i in 1:Nb_sample){
     rnames=c(rnames,paste("A_",SampleNames[i],sep=""))
   }
   R=matrix(data=NA,ncol=8,nrow=Nb_sample,
            dimnames=list(rnames,c("lower bound at 95%","lower bound at 68%","Bayes estimate",
                                   "upper bound at 68%","upper bound at 95%","",
                                   "Convergencies: Bayes estimate","Convergencies: uppers credible interval")))

   ##- Bayes estimate and credible interval
   cat(paste("\n\n>> Bayes estimates of Age for each sample and credible interval <<\n"))
   AgePlot95=matrix(data=NA,nrow=Nb_sample,ncol=3)
   AgePlot68=matrix(data=NA,nrow=Nb_sample,ncol=3)
   AgePlotMoy=rep(0,Nb_sample)
   for(i in 1:Nb_sample){
     cat("------------------------------------------------------\n")
     #cat(paste(" Sample name: ", SampleNames[i],"\n"))
     #cat("---------------------\n")

     cat(paste("Sample name", "\t","Bayes estimate"," Credible interval: \n"))
     cat(paste(paste("A_",SampleNames[i],sep=""),"\t",round(mean(Sample[,i])),'\n'))
     cat("\t\t\t\t\t\t lower bound \t upper bound\n")
     HPD_95=ArchaeoPhases::CredibleInterval(Sample[,i],0.95)
     HPD_68=ArchaeoPhases::CredibleInterval(Sample[,i],0.68)
     cat("\t\t\t\t at level 95% \t",round(c(HPD_95[2])),"\t\t",round(c(HPD_95[3])),"\n")
     cat("\t\t\t\t at level 68% \t",round(c(HPD_68[2])),"\t\t",round(c(HPD_68[3])),"\n")
     AgePlot95[i,]=HPD_95
     AgePlot68[i,]=HPD_68
     AgePlotMoy[i]=round(mean(Sample[,i]))

     R[i,3]=round(mean(Sample[,i]))
     R[i,c(1,5)]=round(HPD_95[2:3])
     R[i,c(2,4)]=round(HPD_68[2:3])
     R[i,6]=c('')
     R[i,7]=round(CV$psrf[i,1],2)
     R[i,8]=round(CV$psrf[i,2],2)

   }

   cat("\n------------------------------------------------------\n")

   # Representation graphique des resultats
   #        des HPD sur la courbe de calibration
   couleur=rainbow(Nb_sample)
   par(mfrow=c(1,1),las = 0,mar=c(5,5,2,2))
   xl=c(min(PriorAge[seq(1,(2*Nb_sample-1),2)]),max(PriorAge[seq(2,(2*Nb_sample),2)]))
   plot(xl,xl,col="white",xlab=c("Age"),ylab=c("cal C14"),xaxt="n",yaxt="n",cex.lab=1.8)
   axis(2,cex.axis=2)
   axis(1,cex.axis=2)
   polygon(c(TableauCalib[,1],rev(TableauCalib[,1])),c(TableauCalib[,2]+2*TableauCalib[,3],rev(TableauCalib[,2]-2*TableauCalib[,3])),col="gray",border="black")
   for(i in 1:Nb_sample){
     lines(c(AgePlot95[i,2:3]),rep(Data_C14Cal[i],2),col=couleur[i],lwd=4)
     lines(AgePlotMoy[i],Data_C14Cal[i],col="black",lwd=2,type='p')
   }
   legend("topleft",SampleNames,lty=rep(1,Nb_sample),lwd=rep(2,Nb_sample),cex=1,col=couleur)
   if(SavePdf==TRUE){
     dev.print(pdf,file=paste(OutputFilePath,OutputFileName[2],'.pdf',sep=""),width=8,height=10)
   }


   #       HPD sur un meme graphe
   par(mfrow=c(1,1),las = 1,oma=c(2,5,0.5,0.5))
   plot(c(AgePlot95[1,2:3]),c(Nb_sample,Nb_sample),
       ylim=c(0.5,Nb_sample+1),xlim=c((min(AgePlot95[,2])-1),(max(AgePlot95[,3])+1)),
       type="l",lwd=5,yaxt="n",xlab="Age (ka)",ylab="",cex.lab=1.5,col="midnightblue",
       main="",cex.main=2)
   grid(ny=NA, lwd = 3)
   lines(c(AgePlot68[1,2:3]),c(Nb_sample,Nb_sample),col="darkslategray3",lwd=5)
   lines(AgePlotMoy[1],Nb_sample,col="firebrick",lwd=5,type='o')
   uu=seq(Nb_sample,1,-1)
   axis(2,uu,labels=SampleNames,cex.axis=1.5)
   legend("topright",c("Bayes estimator","68% credible interval","95% credible interval"),
         pch=c(1,NA,NA),lty=c(NA,1,1),lwd=c(5,5,5),col=c("firebrick","darkslategray3","midnightblue"),bty="o")
   if(Nb_sample>1){
     for(i in 2:Nb_sample){
       lines(c(AgePlot95[i,2:3]),c(Nb_sample-(i-1),Nb_sample-(i-1)),lwd=5,col="midnightblue")
       lines(c(AgePlot68[i,2:3]),c(Nb_sample-(i-1),Nb_sample-(i-1)),lwd=5,col='darkslategray3')
       lines(AgePlotMoy[i],(Nb_sample-(i-1)),col="firebrick",lwd=5,type="o")
     }
   }
   if(SavePdf==TRUE){
     dev.print(pdf,file=paste(OutputFilePath,OutputFileName[3],'.pdf',sep=""),width=8,height=10)
   }

   Info=list("Sampling"=echantillon,
           "Outlier"=Outlier,
           "Model"=Model, "CalibrationCurve"=CalibrationCurve,
           "PriorAge"=PriorAge,
           "StratiConstraints"=StratiConstraints)
  return(Info)
}




