% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/BayesCumulativeProbitHSD.r
\name{BayesCumulativeProbitHSD}
\alias{BayesCumulativeProbitHSD}
\title{Perform MCMC algorithm to generate the posterior samples for longitudinal ordinal data}
\usage{
BayesCumulativeProbitHSD(
  fixed,
  data,
  random,
  Robustness,
  subset,
  na.action,
  HS.model,
  hyper.params,
  num.of.iter
)
}
\arguments{
\item{fixed}{a two-sided linear formula object to describe fixed-effects with the response on the left of
a \samp{~} operator and the terms separated by \samp{+} or \samp{*} operators, on the right.
The specification \code{first*second} indicates the cross of \code{first} and \code{second}.
This is the same as \code{first + second + first:second}.}

\item{data}{an optional data frame containing the variables named in \samp{fixed} and \samp{random}.
It requires an ``integer'' variable named by \samp{id} to denote the identifications of subjects.}

\item{random}{a one-sided linear formula object to describe random-effects with the terms separated by
\samp{+} or \samp{*} operators on the right of a \samp{~} operator.}

\item{Robustness}{logical. If 'TRUE' the distribution of random effects is assumed to be t-distribution;
otherwise normal distribution.}

\item{subset}{an optional expression indicating the subset of the rows of \samp{data} that should be used in the fit.
This can be a logical vector, or a numeric vector indicating which observation numbers are to be included,
or a character vector of the row names to be included.  All observations are included by default.}

\item{na.action}{a function that indicates what should happen when the data contain NA’s.
The default action (\samp{na.omit}, inherited from the \samp{factory fresh} value of
\samp{getOption("na.action")}) strips any observations with any missing values in any variables.}

\item{HS.model}{a specification of the correlation structure in HSD model: \code{HS.model = ~0} denotes
independence, that is, \eqn{R_i} is an identity matrix,
\code{HS.model = ~IndTime+}\eqn{\cdots}\code{+IndTimer} denotes AR(r) correlation structure,
\code{HS.model = ~DiffTime1+}\eqn{\cdots}\code{+DiffTimer} denotes correlation structure related to \eqn{r}th order
of time difference.}

\item{hyper.params}{specify the values in hyperparameters in priors.}

\item{num.of.iter}{an integer to specify the total number of iterations; default is 20000.}
}
\value{
a list of posterior samples, parameters estimates, AIC, BIC, CIC, DIC, MPL, RJR, predicted values,
and the acceptance rates in MH are returned.
}
\description{
This function is used to generate the posterior samples using MCMC algorithm from the
cumulative probit model with the hypersphere decomposition applied to model the correlation structure
in the serial dependence of repeated responses.
}
\note{
Only a model either HSD (\samp{HS.model}) or ARMA (\samp{arma.order}) model should be specified in the function.
We'll provide the reference for details of the model and the algorithm for performing
model estimation whenever the manuscript is accepted.
}
\examples{
\dontrun{
library(BayesRGMM)
library(mvtnorm)
library(dplyr)
library(abind)
library(msm)
library(reshape)

rm(list=ls(all=TRUE))

Fixed.Effs = c(-0.1, 0.1, -0.1) #c(-0.2,-0.8, 1.0, -1.2)
P = length(Fixed.Effs) 
q = 1 #number of random effects
T = 4 #time points
N = 300 #number of subjects
Num.of.Cats = 3
num.of.iter = 5000 #number of iterations

HSD.para = c(-0.9, -0.6) #the parameters in HSD model
a = length(HSD.para)
w = array(runif(T*T*a), c(T, T, a)) #design matrix in HSD model
 
for(time.diff in 1:a)
w[, , time.diff] = 1*(as.matrix(dist(1:T, 1:T, method="manhattan")) ==time.diff)


x = array(0, c(T, P, N))
for(i in 1:N){
    #x[,, i] = t(rmvnorm(P, rep(0, T), AR1.cor(T, Cor.in.DesignMat)))
    x[, 1, i] = 1:T
    x[, 2, i] = rbinom(1, 1, 0.5)
    x[, 3, i] = x[, 1, i]*x[, 2, i]
}

DesignMat = x


CPREM.sim.data = SimulatedDataGenerator.CumulativeProbit(Num.of.Obs = N, 
    Num.of.TimePoints = T, Num.of.Cats = Num.of.Cats, Fixed.Effs = Fixed.Effs, 
    Random.Effs = list(Sigma = 0.5*diag(1), df=3), DesignMat = DesignMat, 
    Missing = list(Missing.Mechanism = 1, Probs = (1:T)^2), 
    HSD.DesignMat.para = list(HSD.para = HSD.para, DesignMat = w))

print(table(CPREM.sim.data$sim.data$y))
print(CPREM.sim.data$classes)

BCP.output = BayesCumulativeProbitHSD(
             fixed = as.formula(paste("y~-1+", paste0("x", 1:P, collapse="+"))), 
             data=CPREM.sim.data$sim.data, random = ~ 1, Roubstness = TRUE, 
             subset = NULL, na.action='na.exclude', HS.model = ~IndTime1, 
             hyper.params=NULL, num.of.iter=num.of.iter)

BayesRobustProbitSummary(BCP.output)

alpha.post.est = BCP.output$Posterior.Samples$PosteriorEstimates$alpha.mean
alpha.est = cbind(CPREM.sim.data$classes, alpha.post.est)
dimnames(alpha.est) = NULL
colnames(alpha.est) = c("True", "Estimate")

print(alpha.est)


y.pred = BCP.output$Posterior.Samples$PosteriorEstimates$Y.Pred

cat("Prediction matrix")
print(table(y.pred+1, CPREM.sim.data$sim.data$y))

beta.est = BCP.output$Posterior.Samples$PosteriorEstimates$beta.mean
PRB = sum(abs((beta.est - Fixed.Effs)/Fixed.Effs))/P

cat("PRB = ", PRB, "\n")

} 
}
\references{
{
\insertRef{Lee:etal:2021}{BayesRGMM}

\insertRef{Lee:etal:2020}{BayesRGMM}

}
}
\author{
Kuo-Jung Lee \href{mailto:kuojunglee@ncku.edu.tw}{kuojunglee@ncku.edu.tw}
}
