requireNamespace("dplyr")



testData <- dplyr::tribble( # even step coordinates (different for lat/long)
  ~database_id, ~datasetName,  ~fieldNotes,                             ~id, ~institutionCode, ~collectionCode,         ~ownerInstitutionCode,      ~basisOfRecord,                  ~occurrenceID,      ~catalogNumber, ~otherCatalogNumbers,   ~kingdom,      ~phylum,    ~class,        ~order,  ~family,  ~scientificName, ~taxonID, ~scientificNameAuthorship, ~genus, ~specificEpithet,        ~recordedBy, ~eventDate, ~year,       ~month, ~day,  ~verbatimEventDate,        ~country, ~stateProvince,  ~locality,                  ~locationRemarks, ~decimalLatitude, ~decimalLongitude, ~minimumElevationInMeters,                                             ~rights,                                     ~rightsHolder, ~accessRights,                                       ~recordId,                                                                        ~references,
  "fake SCAN1", "fakeDataset", "",                                      13775122L,            "CAS",        "ANTWEB",        "UCDC, Davis, CA, USA", "PreservedSpecimen",     "CAS:ANTWEB:casent0106100",     "casent0106100",                   NA, "Animalia", "Arthropoda", "Insecta", "Hymenoptera", "apidae", "apis mellifera",  235783L,          "Linnaeus, 1758", "Apis",      "mellifera",        "P.S. Ward",  "",        NA,        NA,  NA, "", "United States",   "California",    "Davis", "coordinates obtained from Label; 6/28/05",           38.541,        -121.75667,                       15L, "http://creativecommons.org/publicdomain/zero/1.0/", "The California Academy of Sciences - AntWeb.org",            NA, "urn:uuid:46a46727-6535-4e70-88e7-a42c98f806ed", "https://scan-bugs.org:443/portal/collections/individual/index.php?occid=13775122",
  "fake SCAN2", "fakeDataset", "",                                      13775123L,            "CAS",        "ANTWEB",        "UCDC, Davis, CA, USA", "PreservedSpecimen", "CAS:ANTWEB:casent0106100-d01", "casent0106100-d01",                   NA, "Animalia", "Arthropoda", "Insecta", "Hymenoptera", "apidae", "apis mellifera",  235783L,          "Linnaeus, 1758", "Apis",      "mellifera",        "P.S. Ward",  "",        NA,        NA,  NA, "", "United States",   "California",    "Davis", "coordinates obtained from Label: 3 march 2022",           38.542,        -121.75668,                       15L, "http://creativecommons.org/publicdomain/zero/1.0/", "The California Academy of Sciences - AntWeb.org",            NA, "urn:uuid:d6ff3ddb-4695-4aaf-ab89-c251ab2fc7e6", "https://scan-bugs.org:443/portal/collections/individual/index.php?occid=13775123",
  "fake SCAN3", "fakeDataset", "28 Jun 2005",               13775124L,            "CAS",        "ANTWEB",        "UCDC, Davis, CA, USA", "PreservedSpecimen", "CAS:ANTWEB:casent0106100-d11", "casent0106100-d11",                   NA, "Animalia", "Arthropoda", "Insecta", "Hymenoptera", "apidae", "apis mellifera",  235783L,          "Linnaeus, 1758", "Apis",      "mellifera",        "P.S. Ward",  "",        NA,        NA,  NA, "", "United States",   "California",    "Davis", "coordinates obtained from Label",           38.543,        -121.75669,                       15L, "http://creativecommons.org/publicdomain/zero/1.0/", "The California Academy of Sciences - AntWeb.org",            NA, "urn:uuid:4504a6e6-ab1c-4de7-ac43-a3588a63ae90", "https://scan-bugs.org:443/portal/collections/individual/index.php?occid=13775124",
  "fake SCAN4", "fakeDataset", "",                                      NA,                   "CAS",        "ANTWEB",        "UCDC, Davis, CA, USA", "PreservedSpecimen",                             NA,                  NA,                   NA, "Animalia", "Arthropoda", "Insecta", "Hymenoptera", "apidae", "apis mellifera",  235783L,          "Linnaeus, 1758", "Apis",      "mellifera",        "P.S. Ward",  "",        2022,        2,  1, "", "United States",   "California",    "Davis", "coordinates obtained from Label",           38.544,        -121.75670,                       15L, "http://creativecommons.org/publicdomain/zero/1.0/", "The California Academy of Sciences - AntWeb.org",            NA,                                              NA,                                                                                 NA,
  "fake SCAN5", "fakeDataset", "",                                      NA,                   "CAS",        "ANTWEB",        "UCDC, Davis, CA, USA", "PreservedSpecimen",                             NA,                  NA,                   NA, "Animalia", "Arthropoda", "Insecta", "Hymenoptera", "apidae", "apis mellifera",  235783L,          "Linnaeus, 1758", "Apis",      "mellifera",        "P.S. Ward",  "",        NA,     NA,  NA, "28 Jun 2005/29 Jun 2005", "United States",   "California",    "Davis", "coordinates obtained from Label",           38.545,        -121.75671,                       15L, "http://creativecommons.org/publicdomain/zero/1.0/", "The California Academy of Sciences - AntWeb.org",            NA,                                              NA,                                                                                 NA,
  "fake SCAN6", "fakeDataset", "",                                      NA,                   "CAS",        "ANTWEB",        "UCDC, Davis, CA, USA", "PreservedSpecimen",                             NA,                  NA,                   NA, "Animalia", "Arthropoda", "Insecta", "Hymenoptera", "apidae", "apis mellifera",  235783L,          "Linnaeus, 1758", "Apis",      "mellifera",        "P.S. Ward",  "", 2005L,     6L,  28L, "28 Jun 2005/29 Jun 2005", "United States",   "California",    "Davis", "coordinates obtained from Label",           38.546,        -121.75672,                       15L, "http://creativecommons.org/publicdomain/zero/1.0/", "The California Academy of Sciences - AntWeb.org",            NA,                                              NA,                                                                                 NA,
  "fake SCAN7", "fakeDataset", "",                                      NA,                   "CAS",        "ANTWEB",        "UCDC, Davis, CA, USA", "PreservedSpecimen",                             NA,                  NA,                   NA, "Animalia", "Arthropoda", "Insecta", "Hymenoptera", "apidae", "apis mellifera",  235783L,          "Linnaeus, 1758", "Apis",      "mellifera",        "P.S. Ward",  "", NA,     NA,  NA, "28 IX 2005", "United States",   "California",    "Davis", "coordinates obtained from Label",           38.546,        -121.75672,                       15L, "http://creativecommons.org/publicdomain/zero/1.0/", "The California Academy of Sciences - AntWeb.org",            NA,                                              NA,                                                                                 NA,
  "fake SCAN8", "fakeDataset", "",                                      NA,                   "CAS",        "ANTWEB",        "UCDC, Davis, CA, USA", "PreservedSpecimen",                             NA,                  NA,                   NA, "Animalia", "Arthropoda", "Insecta", "Hymenoptera", "apidae", "apis mellifera",  235783L,          "Linnaeus, 1758", "Apis",      "mellifera",        "P.S. Ward",  "", NA,     NA,  NA, "I 2022", "United States",   "California",    "Davis", "coordinates obtained from Label",           38.546,        -121.75672,                       15L, "http://creativecommons.org/publicdomain/zero/1.0/", "The California Academy of Sciences - AntWeb.org",            NA,                                              NA,                                                                                 NA,
  "fake SCAN10", "fakeDataset", "",                                      NA,                   "CAS",        "ANTWEB",        "UCDC, Davis, CA, USA", "PreservedSpecimen",                             NA,                  NA,                   NA, "Animalia", "Arthropoda", "Insecta", "Hymenoptera", "apidae", "apis mellifera",  235783L,          "Linnaeus, 1758", "Apis",      "mellifera",        "P.S. Ward",  "", NA,     NA,  NA, "2022", "United States",   "California",    "Davis", "coordinates obtained from Label",           38.546,        -121.75672,                       15L, "http://creativecommons.org/publicdomain/zero/1.0/", "The California Academy of Sciences - AntWeb.org",            NA,                                              NA,                                                                                 NA,
  "fake SCAN11", "fakeDataset", "",                                      NA,                   "CAS",        "ANTWEB",        "UCDC, Davis, CA, USA", "PreservedSpecimen",                             NA,                  NA,                   NA, "Animalia", "Arthropoda", "Insecta", "Hymenoptera", "apidae", "apis mellifera",  235783L,          "Linnaeus, 1758", "Apis",      "mellifera",        "P.S. Ward",  "2005-05-05", NA,     NA,  NA, "2022", "United States",   "California",    "Davis", "coordinates obtained from Label",           38.546,        -121.75672,                       15L, "http://creativecommons.org/publicdomain/zero/1.0/", "The California Academy of Sciences - AntWeb.org",            NA,                                              NA,                                                                                 NA
  
  )


testOut <- BeeBDC::dateFindR(data = testData,
                        # Years above this are removed (from the recovered dates only)
                        maxYear = lubridate::year(Sys.Date()),
                        # Years below this are removed (from the recovered dates only)
                        minYear = 1700) %>%
  dplyr::select(database_id, eventDate, fieldNotes, locationRemarks, verbatimEventDate, 
                day, month, year) %>%
  dplyr::arrange(desc(database_id))


# Test the expected results
testthat::test_that("dateFindR results successfuly matched", {
  testthat::expect_equal(sum(complete.cases(testOut$eventDate) ), 9)
})
testthat::test_that("dateFindR results unsuccessfuly matched", {
  testthat::expect_equal(sum(is.na(testOut$eventDate) ), 1)
})

testthat::test_that("dateFindR output dates match", {
  testthat::expect_equal(testOut$eventDate %>% as.character(),
                         c( "2022-01-01", "2005-09-28","2005-06-28", "2005-06-28",
                            "2022-02-01", "2005-06-28", "2022-03-03",  "2005-05-05",
                            NA, "2005-06-28"))
})

  # Test classes
testthat::test_that("dateFindR expected class", {
  testthat::expect_type(testOut, "list")
})
testthat::test_that("dateFindR expected class", {
  testthat::expect_equal(attributes(testOut)$class, c("tbl_df","tbl" ,"data.frame"))
})




troubleRows <- dplyr::tribble(
           ~database_id,                              ~scientificName,      ~family,    ~subfamily,           ~genus, ~subgenus,                    ~species, ~specificEpithet, ~infraspecificEpithet, ~taxonRank, ~scientificNameAuthorship, ~identificationQualifier,                                        ~higherClassification, ~typeStatus, ~previousIdentifications, ~verbatimIdentification,                   ~identifiedBy,       ~dateIdentified, ~decimalLatitude, ~decimalLongitude, ~verbatimLatitude, ~verbatimLongitude, ~verbatimElevation,   ~stateProvince,                   ~country,      ~continent,                                                                                                       ~locality,    ~county,    ~municipality, ~countryCode, ~level0Gid,     ~level0Name, ~level1Gid,      ~level1Name,       ~license,                                                                                                                                                                                                                      ~issue,            ~eventDate, ~eventTime, ~startDayOfYear, ~endDayOfYear, ~day, ~month, ~year,       ~basisOfRecord, ~type, ~occurrenceStatus, ~recordNumber,                                                     ~recordedBy, ~eventID, ~samplingProtocol, ~samplingEffort, ~individualCount, ~coordinatePrecision, ~coordinateUncertaintyInMeters,      ~catalogNumber,     ~gbifID,                                      ~datasetID,                                                                ~institutionCode,                                                                                             ~datasetName, ~otherCatalogNumbers,                                                                     ~occurrenceID, ~taxonKey,                                ~coreid,                                       ~recordId,                                   ~collectionID, ~associatedSequences,                      ~verbatimScientificName,   ~verbatimEventDate,         ~associatedTaxa, ~associatedOrganisms,                                                                                                                                         ~fieldNotes,           ~sex,                        ~rightsHolder, ~accessRights, ~associatedReferences, ~bibliographicCitation,                                                                        ~references, ~informationWithheld, ~hasCoordinate, ~hasGeospatialIssues,       ~id, ~associatedOccurrences, ~locationRemarks,          ~dataSource, ~.scientificName_empty, ~.coordinates_empty, ~.coordinates_outOfRange, ~.basisOfRecords_notStandard, ~.basisOfRecord_notSpecimen, ~verbatimCounty, ~.occurrenceAbsent, ~rights, ~.unLicensed, ~.GBIFflags, ~.uncer_terms, ~.invalidName, ~.rou, ~.val, ~.equ, ~.zer, ~.cap, ~.cen, ~.gbf, ~.inst, ~.summary, ~.sequential, ~.lonFlag, ~.latFlag, ~.gridSummary, ~.uncertaintyThreshold,
   "NC_checklist_32750",                  "Dialictus Robertson, 1902", "Halictidae",            NA,      "Dialictus",        NA,                          NA,               NA,                    NA,    "GENUS",                        NA,                       NA,                                                           NA,          NA,                       NA,                      NA,   "Russell, P. J. (Phillip J.)", "2010-01-01T00:00:00",          35.1675,          -83.3044,                NA,                 NA,                 NA, "North Carolina",                         NA, "NORTH_AMERICA",          "\"GSMNP Oconaluftee residential area, 35°10'3\"\"N 83°18'16\"\"W, 2039ft, Old Field, Swain Co., NC\"",    "Swain",               NA,         "US",      "USA", "United States", "USA.34_1", "North Carolina",    "CC_BY_4_0",                                                                                   "GEODETIC_DATUM_ASSUMED_WGS84;CONTINENT_DERIVED_FROM_COORDINATES;OCCURRENCE_STATUS_INFERRED_FROM_INDIVIDUAL_COUNT;INSTITUTION_MATCH_NONE", "2004-07-14T00:00:00",         NA,              NA,            NA,  14L,     7L, 2004L, "PRESERVED_SPECIMEN",    NA,         "PRESENT",            NA,                                     "Steck, G., Sutton & Super",       NA,    "Malaise trap",              NA,               1L,                   NA,                             NA,    "UCFC 0 260 365", 1800769439L,                                              NA,                  "University of Central Florida Collection of Arthopods (UCFC)",                                                                                                       NA,                   NA,               "urn:lsid:biosci.ohio-state.edu:osuc_occurrences:UCFC__0__260__365",  1352488L,                                     NA,                                              NA,                                              NA,                   NA,                   "Lasioglossum (Dialictus)",          "14-Jul-04",                      NA,                   NA,                                                                                                                "lot_id: 6994; preparation: pointed",             NA,                                   NA,            NA,                    NA,                     NA,                                                                                 NA,                   NA,           TRUE,                FALSE,        NA,                     NA,               NA,    "GBIF_Halictidae",                   TRUE,                TRUE,                     TRUE,                         TRUE,                        TRUE,         "Swain",               TRUE,      NA,         TRUE,        TRUE,          TRUE,         FALSE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,   TRUE,        NA,         TRUE,      TRUE,      TRUE,          TRUE,                     NA,
    "NC_checklist_3666",              "Andrena accepta Viereck, 1916", "Andrenidae",  "Andreninae",        "Andrena",        NA,           "Andrena accepta",        "accepta",                    NA,    "GENUS",           "Viereck, 1916",                       NA,                                                           NA,          NA,                       NA,                      NA, "Mitchell, T. B. (Theodore B.)", "1962-01-01T00:00:00",        34.476556,        -77.969157,                NA,                 NA,                 NA, "North Carolina",                         NA, "NORTH_AMERICA",                                        "Savanna, Burgaw, Pender Co., NC, Pender, North Carolina, United States",   "Pender",               NA,         "US",      "USA", "United States", "USA.34_1", "North Carolina",    "CC_BY_4_0", "COORDINATE_ROUNDED;GEODETIC_DATUM_ASSUMED_WGS84;CONTINENT_DERIVED_FROM_COORDINATES;TAXON_MATCH_SCIENTIFIC_NAME_ID_IGNORED;OCCURRENCE_STATUS_INFERRED_FROM_INDIVIDUAL_COUNT;INSTITUTION_MATCH_FUZZY;COLLECTION_MATCH_FUZZY",                    NA,         NA,              NA,            NA,   NA,     NA,    NA, "PRESERVED_SPECIMEN",    NA,         "PRESENT",            NA,                                 "Mitchell, T. B. (Theodore B.)",       NA,  "none specified",              NA,               1L,                   NA,                             NA,     "FSCA 00096225",          NA,                                              NA,                                                                          "FSCA",                                                                                                       NA,                   NA, "http://mbd-db.osu.edu/hol/collecting_units/403a7c17-66f1-4bfe-9047-d23cb4669bf3",  1345710L,                                     NA,                                              NA,                                              NA,                   NA,                            "Andrena accepta",                   NA,                      NA,                   NA,  "[Savanna at Burgaw, N. C.] [T. B. Mitchell Oct. 23, 1957] [Hellanthus] [Andrena accepta Vier. ♀ det. Mitch. '62] [Andrena (Pterandrena) accepta]",       "FEMALE",                                   NA,            NA,                    NA,                     NA,  "http://mbd-db.osu.edu/hol/collecting_units/403a7c17-66f1-4bfe-9047-d23cb4669bf3",                   NA,           TRUE,                FALSE,        NA,                     NA,               NA,    "GBIF_Andrenidae",                   TRUE,                TRUE,                     TRUE,                         TRUE,                        TRUE,        "Pender",               TRUE,      NA,         TRUE,        TRUE,          TRUE,          TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,   TRUE,        NA,         TRUE,      TRUE,      TRUE,          TRUE,                     NA,
   "NC_checklist_45856",                                 "Halictidae", "Halictidae",            NA,               NA,        NA,                          NA,               NA,                    NA,         NA,                        NA,                       NA,                                                           NA,          NA,                       NA,                      NA,                     "Dasch, C.",                "2001",          35.7547,          -82.2556,                NA,                 NA,                 NA, "North Carolina",            "United States",              NA,                                                                        "Mount Mitchell, 5500ft, Yancey Co., NC",   "Yancey",               NA,           NA,         NA,              NA,         NA,               NA,             NA,                                                                                                                                                                                                                          NA,           "8/17/1970",         NA,              NA,            NA,   NA,     NA,    NA,  "PreservedSpecimen",    NA,                NA,            NA,                                                     "Dasch, C.",       NA,            "trap",              NA,               1L,                   NA,                             NA,       "OSUC 102262",          NA,                                              NA, "C.A. Triplehorn Insect Collection, Ohio State University, Columbus, OH (OSUC)",                                                                                                       NA,                   NA,                    "urn:lsid:biosci.ohio-state.edu:osuc_occurrences:OSUC__102262",        NA, "57ba61b4-c249-411b-b46f-cb15d7f75913",                                              NA,                                              NA,                   NA,                                           NA,                   NA,                      NA,                   NA,                                             "C. Dasch Collection [Mt. Mitchell, N.C. 5500 ft.; added 12-JUL-2002] [Aug 17 1970; added 12-JUL-2002]", "undetermined",                                   NA,            NA,                    NA,                     NA,                                                                                 NA,                   NA,             NA,                   NA,        NA,                     NA,               NA, "iDigBio_Halictidae",                   TRUE,                TRUE,                     TRUE,                         TRUE,                        TRUE,        "Yancey",               TRUE,      NA,         TRUE,        TRUE,          TRUE,         FALSE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,   TRUE,        NA,         TRUE,      TRUE,      TRUE,          TRUE,                     NA,
   "NC_checklist_45975",                                 "Halictidae", "Halictidae",            NA,               NA,        NA,                          NA,               NA,                    NA,         NA,                        NA,                       NA,                                                           NA,          NA,                       NA,                      NA,                              NA,                "2002",          35.2247,          -75.5303,                NA,                 NA,                 NA, "North Carolina",            "United States",              NA,                                                                          "Cape Hatteras, Coccino, Dare Co., NC",     "Dare",               NA,           NA,         NA,              NA,         NA,               NA,             NA,                                                                                                                                                                                                                          NA,           "8/24/1971",         NA,              NA,            NA,   NA,     NA,    NA,  "PreservedSpecimen",    NA,                NA,            NA,                                                     "Dasch, C.",       NA,  "none specified",              NA,               1L,                   NA,                             NA,       "OSUC 102376",          NA,                                              NA, "C.A. Triplehorn Insect Collection, Ohio State University, Columbus, OH (OSUC)",                                                                                                       NA,                   NA,                    "urn:lsid:biosci.ohio-state.edu:osuc_occurrences:OSUC__102376",        NA, "364cca84-19f9-4883-b777-f28587802925",                                              NA,                                              NA,                   NA,                                           NA,                   NA,                      NA,                   NA,                                             "C. Dasch Collection [Coccino Cape Hatteras NC.; added 12-JUL-2002] [Aug. 24, 1971; added 12-JUL-2002]", "undetermined",                                   NA,            NA,                    NA,                     NA,                                                                                 NA,                   NA,             NA,                   NA,        NA,                     NA,               NA, "iDigBio_Halictidae",                   TRUE,                TRUE,                     TRUE,                         TRUE,                        TRUE,          "Dare",               TRUE,      NA,         TRUE,        TRUE,          TRUE,         FALSE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,   TRUE,        NA,         TRUE,      TRUE,      TRUE,          TRUE,                     NA,
   "NC_checklist_45007",               "Bombus terricola Kirby, 1837",     "Apidae",      "Apinae",         "Bombus",        NA,          "Bombus terricola",      "terricola",                    NA,         NA,             "Kirby, 1837",                       NA,                                                           NA,          NA,                       NA,                      NA,                              NA,                "2001",          35.7547,          -82.2556,                NA,                 NA,                 NA, "North Carolina",            "United States",              NA,                                                                        "Mount Mitchell, 5500ft, Yancey Co., NC",   "Yancey",               NA,           NA,         NA,              NA,         NA,               NA,             NA,                                                                                                                                                                                                                          NA,           "8/13/1970",         NA,              NA,            NA,   NA,     NA,    NA,  "PreservedSpecimen",    NA,                NA,            NA,                                                     "Dasch, C.",       NA,  "none specified",              NA,               1L,                   NA,                             NA,       "OSUC 100900",          NA,                                              NA, "C.A. Triplehorn Insect Collection, Ohio State University, Columbus, OH (OSUC)",                                                                                                       NA,                   NA,                    "urn:lsid:biosci.ohio-state.edu:osuc_occurrences:OSUC__100900",        NA, "0047d1b4-94ef-4b45-a74f-913867ac965b",                                              NA,                                              NA,                   NA,                           "Bombus terricola",                   NA,                      NA,                   NA,                                                                                                "[Mt. Mitchell N.C.5500 ft. Aug. 13, 1970 C. Dasch]",       "female",                                   NA,            NA,                    NA,                     NA,                                                                                 NA,                   NA,             NA,                   NA,        NA,                     NA,               NA,     "iDigBio_Apidae",                   TRUE,                TRUE,                     TRUE,                         TRUE,                        TRUE,        "Yancey",               TRUE,      NA,         TRUE,        TRUE,          TRUE,          TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,   TRUE,        NA,         TRUE,      TRUE,      TRUE,          TRUE,                     NA,
   "NC_checklist_87482",              "Nomada articulata Smith, 1854",     "Apidae",   "Nomadinae",         "Nomada",        NA,         "Nomada articulata",     "articulata",                    NA,         NA,             "Smith, 1854",                       NA,                                                           NA,          NA,                       NA,                      NA,                      "Snelling",                    NA,         35.17173,         -79.39167,                NA,                 NA,                 NA, "North Carolina",            "United States",              NA,                                                                                                "Southern Pines",         NA, "Southern Pines",           NA,         NA,              NA,         NA,               NA,             NA,                                                                                                                                                                                                                          NA,           "4/14/2010",         NA,            104L,            NA,  14L,     4L, 1910L,  "PreservedSpecimen",    NA,                NA,            NA,                                                   "A. H. Manoo",       NA,                NA,              NA,               NA,                   NA,                          2000L,              "9949",          NA,                                              NA,                                                                          "UNSM",                                                                                                       NA,                   NA,                                            "7942d7fb-d0fb-436a-aa65-f367143443ce",        NA,                                     NA, "urn:uuid:7942d7fb-d0fb-436a-aa65-f367143443ce",                                              NA,                   NA,                          "Nomada articulata",          "4/14/2010",                      NA,                   NA,                                                                                                                                                  NA,            "M",                                   NA,            NA,                    NA,                     NA, "https://scan-bugs.org:443/portal/collections/individual/index.php?occid=64923665",                   NA,             NA,                   NA, 64923665L,                     NA,               NA,        "SCAN_Apidae",                   TRUE,                TRUE,                     TRUE,                         TRUE,                        TRUE,              NA,               TRUE,      NA,         TRUE,        TRUE,          TRUE,          TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,   TRUE,        NA,         TRUE,      TRUE,      TRUE,          TRUE,                   TRUE,
   "NC_checklist_49805",                "Andrena integra Smith, 1853", "Andrenidae",  "Andreninae",        "Andrena",        NA,           "Andrena integra",        "integra",                    NA,         NA,             "Smith, 1853",                       NA,                                                           NA,          NA,                       NA,                      NA,                              NA,                "2001",          35.4267,          -83.0853,                NA,                 NA,                 NA, "North Carolina",            "United States",              NA,                                                                                       "Balsam, Jackson Co., NC",  "Jackson",               NA,           NA,         NA,              NA,         NA,               NA,             NA,                                                                                                                                                                                                                          NA,            "6/1/1956",         NA,              NA,            NA,   NA,     NA,    NA,  "PreservedSpecimen",    NA,                NA,            NA, "Knull, D. J. (Dorothy Johnson) & Knull, J. N. (Josef Nissley)",       NA,  "none specified",              NA,               1L,                   NA,                             NA,       "OSUC 125900",          NA,                                              NA, "C.A. Triplehorn Insect Collection, Ohio State University, Columbus, OH (OSUC)",                                                                                                       NA,                   NA,                    "urn:lsid:biosci.ohio-state.edu:osuc_occurrences:OSUC__125900",        NA, "fd50033e-f274-4e47-9225-e546aa140562",                                              NA,                                              NA,                   NA,                            "Andrena integra",                   NA,                      NA,                   NA,                                                                                    "[Balsam, N.C.; added 12-JUL-2002] [VI-1-56; added 12-JUL-2002]",       "female",                                   NA,            NA,                    NA,                     NA,                                                                                 NA,                   NA,             NA,                   NA,        NA,                     NA,               NA, "iDigBio_Andrenidae",                   TRUE,                TRUE,                     TRUE,                         TRUE,                        TRUE,       "Jackson",               TRUE,      NA,         TRUE,        TRUE,          TRUE,          TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,   TRUE,        NA,         TRUE,      TRUE,      TRUE,          TRUE,                     NA,
   "NC_checklist_49033",                                 "Halictidae", "Halictidae",            NA,               NA,        NA,                          NA,               NA,                    NA,         NA,                        NA,                       NA,                                                           NA,          NA,                       NA,                      NA,                     "Dasch, C.",                "2001",          35.7547,          -82.2556,                NA,                 NA,                 NA, "North Carolina",            "United States",              NA,                                                                        "Mount Mitchell, 5500ft, Yancey Co., NC",   "Yancey",               NA,           NA,         NA,              NA,         NA,               NA,             NA,                                                                                                                                                                                                                          NA,           "8/13/1970",         NA,              NA,            NA,   NA,     NA,    NA,  "PreservedSpecimen",    NA,                NA,            NA,                                 "Dasch, B. (Betty) & Dasch, C.",       NA,  "none specified",              NA,               1L,                   NA,                             NA,       "OSUC 102252",          NA,                                              NA, "C.A. Triplehorn Insect Collection, Ohio State University, Columbus, OH (OSUC)",                                                                                                       NA,                   NA,                    "urn:lsid:biosci.ohio-state.edu:osuc_occurrences:OSUC__102252",        NA, "ab5ccc45-f82b-450b-8634-b3be0b426804",                                              NA,                                              NA,                   NA,                                           NA,                   NA,                      NA,                   NA,                                             "C. Dasch Collection [Mt. Mitchell, N.C. 5500 ft.; added 12-JUL-2002] [Aug 13 1970; added 12-JUL-2002]", "undetermined",                                   NA,            NA,                    NA,                     NA,                                                                                 NA,                   NA,             NA,                   NA,        NA,                     NA,               NA, "iDigBio_Halictidae",                   TRUE,                TRUE,                     TRUE,                         TRUE,                        TRUE,        "Yancey",               TRUE,      NA,         TRUE,        TRUE,          TRUE,         FALSE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,   TRUE,        NA,         TRUE,      TRUE,      TRUE,          TRUE,                     NA,
   "NC_checklist_46230",             "Triepeolus lunatus (Say, 1824)",     "Apidae",   "Nomadinae",     "Triepeolus",        NA,        "Triepeolus lunatus",        "lunatus",                    NA,         NA,             "(Say, 1824)",                       NA,              "Animalia;Arthropoda;Apidae;Nomadinae;Epeolini",      "None",                       NA,                      NA,               "M. G. Rightmyer",                "2006",         35.17389,          -79.3925,                NA,                 NA,            "159 m", "North Carolina",            "UNITED STATES",              NA,                                                                                                "Southern Pines",    "Moore",               NA,           NA,         NA,              NA,         NA,               NA,             NA,                                                                                                                                                                                                                          NA,                    NA,         NA,              NA,            NA,   NA,     NA,    NA,  "PreservedSpecimen",    NA,                NA,            NA,                                                       "unknown",       NA,         "Netting",              NA,               1L,                   NA,                             NA, "AMNH_BEE 00233319",          NA, "urn:uuid:13674fa4-8611-11e4-8259-0026552be7ea",                                                                          "AMNH", "Collaborative databasing of North American bee collections within a global informatics network project",                   NA,                                   "urn:uuid:87faf44e-d8e1-11e2-99a2-0026552be7ea",        NA, "2cd38ed1-e14c-4d7d-86e9-d59ec87e19a5",                                              NA,                                              NA,                   NA,                         "Triepeolus lunatus",          "31-Jul-18", "associated with:Aster",                   NA,                                                                                                                                                  NA,         "Male",                                   NA,            NA,                    NA,                     NA,                                                                                 NA,                   NA,             NA,                   NA,        NA,                     NA,               NA,     "iDigBio_Apidae",                   TRUE,                TRUE,                     TRUE,                         TRUE,                        TRUE,         "Moore",               TRUE,      NA,         TRUE,        TRUE,          TRUE,          TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,   TRUE,        NA,         TRUE,      TRUE,      TRUE,          TRUE,                     NA,
   "NC_checklist_44059",                     "Bombus Latreille, 1802",     "Apidae",            NA,         "Bombus",        NA,                          NA,               NA,                    NA,    "GENUS",                        NA,                       NA, "Animalia, Arthropoda, Insecta, Hymenoptera, Apidae, Apinae",          NA,                       NA,                      NA,                              NA,                    NA,          35.5561,          -83.3114,                NA,                 NA,                 NA,      "Tennessee",                         NA, "NORTH_AMERICA",                                                                "Smokemont, Great Smoky Mountains National Park",         NA,               NA,         "US",      "USA", "United States", "USA.34_1", "North Carolina",      "CC0_1_0",                                                                                                          "GEODETIC_DATUM_ASSUMED_WGS84;CONTINENT_DERIVED_FROM_COORDINATES;OCCURRENCE_STATUS_INFERRED_FROM_INDIVIDUAL_COUNT", "1835-08-01T00:00:00",         NA,            243L,          243L,   NA,     8L, 1835L, "PRESERVED_SPECIMEN",    NA,         "PRESENT",            NA,                                                     "H. Allard",       NA,                NA,              NA,               1L,                   NA,                             NA,   "USNMENT01053032", 1321292009L,                                              NA,                                                                          "USNM",                                                                                    "NMNH Extant Biology",                   NA,                 "http://n2t.net/ark:/65665/3b4dcb978-d77c-4e2e-94ef-e051c2db9206",  1340278L,                                     NA,                                              NA, "urn:uuid:18e3cd08-a962-4f0a-b72c-9a0b3600c5ad",                   NA,                                 "Bombus sp.",           "Aug. '35",                      NA,                   NA,                                                                                                                                                  NA,             NA,                                   NA,            NA,                    NA,                     NA,                                                                                 NA,                   NA,           TRUE,                FALSE,        NA,                     NA,               NA,        "GBIF_Apidae",                   TRUE,                TRUE,                     TRUE,                         TRUE,                        TRUE,        "Sevier",               TRUE,      NA,         TRUE,        TRUE,          TRUE,         FALSE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,   TRUE,        NA,         TRUE,      TRUE,      TRUE,          TRUE,                     NA,
  "NC_checklist_100075", "Protandrena compositarum (Robertson, 1893)", "Andrenidae",  "Panurginae",    "Protandrena",        NA,  "Protandrena compositarum",   "compositarum",                    NA,         NA,       "(Robertson, 1893)",                       NA,                                                           NA,          NA,                       NA,                      NA,                      "Mitchell",                    NA,               NA,                NA,                NA,                 NA,                 NA, "North Carolina",            "United States",              NA,                                                                                                      "Savannah",   "Pender",         "Burgaw",           NA,         NA,              NA,         NA,               NA,             NA,                                                                                                                                                                                                                          NA,          "1954-00-00",         NA,              NA,            NA,  10L,    13L, 1954L,  "PreservedSpecimen",    NA,                NA,            NA,                                                "T. B. Mitchell",       NA,                NA,              NA,               NA,                   NA,                             NA, "NCSU_ENT 00302466",          NA,                                              NA,                                                                          "NCSU",                                                                                                       NA,                   NA,                                            "960e4015-413d-4e46-a458-a78b11f57e52",        NA,                                     NA, "urn:uuid:960e4015-413d-4e46-a458-a78b11f57e52",                                              NA,                   NA,                "Pseudopanurgus compositarum",         "1954-13-10",                 "Aster",                   NA,                                                                                                                                                  NA,             NA,                                   NA,            NA,                    NA,                     NA, "https://scan-bugs.org:443/portal/collections/individual/index.php?occid=65350841",                   NA,             NA,                   NA, 65350841L,                     NA,               NA,    "SCAN_Andrenidae",                   TRUE,               FALSE,                     TRUE,                         TRUE,                        TRUE,        "Pender",               TRUE,      NA,         TRUE,        TRUE,          TRUE,          TRUE, FALSE,    NA,    NA,    NA,    NA,    NA,    NA,     NA,     FALSE,         TRUE,      TRUE,      TRUE,          TRUE,                     NA,
   "NC_checklist_28295",    "Lasioglossum perpunctatum (Ellis, 1913)", "Halictidae",  "Halictinae",   "Lasioglossum",        NA, "Lasioglossum perpunctatum",   "perpunctatum",                    NA,  "SPECIES",           "(Ellis, 1913)",                       NA,                                                           NA,          NA,                       NA,                      NA,                              NA,                    NA,        35.792278,        -82.310111,                NA,                 NA,                 NA, "North Carolina",                         NA, "NORTH_AMERICA",                                                                                     "Valley of Black Mountains",   "Yancey",               NA,         "US",      "USA", "United States", "USA.34_1", "North Carolina",    "CC_BY_4_0",                                                            "GEODETIC_DATUM_ASSUMED_WGS84;CONTINENT_DERIVED_FROM_COORDINATES;OCCURRENCE_STATUS_INFERRED_FROM_INDIVIDUAL_COUNT;COLLECTION_MATCH_NONE;INSTITUTION_MATCH_FUZZY",                    NA,         NA,              NA,            NA,   NA,     NA,    NA, "PRESERVED_SPECIMEN",    NA,         "PRESENT",            NA,                                               "W. Beutenmuller",       NA,         "Netting",              NA,               1L,                   NA,                             NA, "AMNH_BEE 00068888",          NA,                                              NA,                                                                          "AMNH",                                                                                       "AMNH Hymenoptera",                   NA,                                            "8450fb18-d8e1-11e2-99a2-0026552be7ea",  1353845L,                                     NA,                                              NA,                                              NA,                   NA,    "Lasioglossum perpunctatum (Ellis, 1913)",          "6/20/1906",                      NA,                   NA,                                                                                                                                                  NA,       "FEMALE", "American Museum of Natural History",            NA,                    NA,                     NA,                                                                                 NA,                   NA,           TRUE,                FALSE,        NA,                     NA,               NA,    "GBIF_Halictidae",                   TRUE,                TRUE,                     TRUE,                         TRUE,                        TRUE,    "Yancey Co.",               TRUE,      NA,         TRUE,        TRUE,          TRUE,          TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,   TRUE,        NA,         TRUE,      TRUE,      TRUE,          TRUE,                     NA,
   "NC_checklist_26803",       "Xenoglossa kansensis Cockerell, 1905",     "Apidae",      "Apinae",     "Xenoglossa",        NA,      "Xenoglossa kansensis",      "kansensis",                    NA,  "SPECIES",         "Cockerell, 1905",                       NA,                                                           NA,          NA,                       NA,                      NA,                              NA,                    NA,        35.173889,          -79.3925,                NA,                 NA,                 NA, "North Carolina",                         NA, "NORTH_AMERICA",                                                                                                "Southern Pines",    "Moore",               NA,         "US",      "USA", "United States", "USA.34_1", "North Carolina",    "CC_BY_4_0",                                                            "GEODETIC_DATUM_ASSUMED_WGS84;CONTINENT_DERIVED_FROM_COORDINATES;OCCURRENCE_STATUS_INFERRED_FROM_INDIVIDUAL_COUNT;COLLECTION_MATCH_NONE;INSTITUTION_MATCH_FUZZY",                    NA,         NA,              NA,            NA,   NA,     NA,    NA, "PRESERVED_SPECIMEN",    NA,         "PRESENT",            NA,                                                              NA,       NA,         "Netting",              NA,               1L,                   NA,                             NA, "AMNH_BEE 00014019",          NA,                                              NA,                                                                          "AMNH",                                                                                       "AMNH Hymenoptera",                   NA,                                            "82cf63f6-d8e1-11e2-99a2-0026552be7ea",  1339365L,                                     NA,                                              NA,                                              NA,                   NA,       "Xenoglossa kansensis Cockerell, 1905",          "6/13/1918",                      NA,                   NA,                                                                                                                                                  NA,         "MALE", "American Museum of Natural History",            NA,                    NA,                     NA,                                                                                 NA,                   NA,           TRUE,                FALSE,        NA,                     NA,               NA,        "GBIF_Apidae",                   TRUE,                TRUE,                     TRUE,                         TRUE,                        TRUE,     "Moore Co.",               TRUE,      NA,         TRUE,        TRUE,          TRUE,          TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,   TRUE,        NA,         TRUE,      TRUE,      TRUE,          TRUE,                     NA,
   "NC_checklist_18034", "Augochloropsis metallica (Fabricius, 1793)", "Halictidae",  "Halictinae", "Augochloropsis",        NA,  "Augochloropsis metallica",      "metallica",                    NA,  "SPECIES",       "(Fabricius, 1793)",                       NA,                                                           NA,          NA,                       NA,                      NA,               "Mitchell, T. B.",                    NA,          35.7721,          -78.6386,                NA,                 NA,                 NA, "North Carolina",                         NA, "NORTH_AMERICA",                                                                                                       "Raleigh",     "Wake",               NA,         "US",      "USA", "United States", "USA.34_1", "North Carolina", "CC_BY_NC_4_0",                            "GEODETIC_DATUM_ASSUMED_WGS84;CONTINENT_DERIVED_FROM_COORDINATES;RECORDED_DATE_INVALID;OCCURRENCE_STATUS_INFERRED_FROM_INDIVIDUAL_COUNT;INSTITUTION_MATCH_FUZZY;INSTITUTION_COLLECTION_MISMATCH",                    NA,         NA,              NA,            NA,   NA,     NA,    NA, "PRESERVED_SPECIMEN",    NA,         "PRESENT",            NA,                                               "Mitchell, T. B.",       NA,                NA,              NA,               1L,                   NA,                             NA, "NCSU_ENT 00231139",          NA,                                              NA,                                                                          "NCSU",                                                                                                       NA,                   NA,                                            "87bdfd32-d434-4d7a-9c56-65d3ad72092d",  1353091L,                                     NA,                                              NA,          "d4501850-33c3-41fd-82de-b6b1d86a3f8b",                   NA, "Augochloropsis metallica (Fabricius, 1793)",     "mid April 1921",                      NA,                   NA,                                                                                                                                                  NA,             NA,                                   NA,            NA,                    NA,                     NA, "https://scan-bugs.org:443/portal/collections/individual/index.php?occid=60681396",                   NA,           TRUE,                FALSE,        NA,                     NA,               NA,    "GBIF_Halictidae",                   TRUE,                TRUE,                     TRUE,                         TRUE,                        TRUE,          "Wake",               TRUE,      NA,         TRUE,        TRUE,          TRUE,          TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,   TRUE,        NA,         TRUE,      TRUE,      TRUE,          TRUE,                     NA,
   "NC_checklist_16325",          "Melissodes niveus Robertson, 1895",     "Apidae",      "Apinae",     "Melissodes",        NA,         "Melissodes niveus",         "niveus",                    NA,  "SPECIES",         "Robertson, 1895",                       NA,                                                           NA,          NA,                       NA,                      NA,           "LaBerge, Wallace E.",                    NA,          35.7721,          -78.6386,                NA,                 NA,                 NA, "North Carolina",                         NA, "NORTH_AMERICA",                                                                                                       "Raleigh",     "Wake",               NA,         "US",      "USA", "United States", "USA.34_1", "North Carolina", "CC_BY_NC_4_0",          "GEODETIC_DATUM_ASSUMED_WGS84;CONTINENT_DERIVED_FROM_COORDINATES;RECORDED_DATE_INVALID;TAXON_MATCH_FUZZY;OCCURRENCE_STATUS_INFERRED_FROM_INDIVIDUAL_COUNT;INSTITUTION_MATCH_FUZZY;INSTITUTION_COLLECTION_MISMATCH",                    NA,         NA,              NA,            NA,   NA,     NA,    NA, "PRESERVED_SPECIMEN",    NA,         "PRESENT",            NA,                                               "Mitchell, T. B.",       NA,                NA,              NA,               1L,                   NA,                             NA, "NCSU_ENT 00248253",          NA,                                              NA,                                                                          "NCSU",                                                                                                       NA,                   NA,                                            "cd4ff679-e751-4b96-91c0-f31ea85b7332",  5040625L,                                     NA,                                              NA,          "d4501850-33c3-41fd-82de-b6b1d86a3f8b",                   NA,          "Melissodes niveus Robertson, 1895", "mid September 1921",                      NA,                   NA,                                                                                                                                                  NA,             NA,                                   NA,            NA,                    NA,                     NA, "https://scan-bugs.org:443/portal/collections/individual/index.php?occid=60686464",                   NA,           TRUE,                FALSE,        NA,                     NA,               NA,        "GBIF_Apidae",                   TRUE,                TRUE,                     TRUE,                         TRUE,                        TRUE,          "Wake",               TRUE,      NA,         TRUE,        TRUE,          TRUE,          TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,   TRUE,        NA,         TRUE,      TRUE,      TRUE,          TRUE,                     NA,
   "NC_checklist_36870",                    "Hylaeus Fabricius, 1793", "Colletidae",            NA,        "Hylaeus",        NA,                          NA,               NA,                    NA,    "GENUS",                        NA,                       NA,                                                           NA,          NA,                       NA,                      NA,  "Fullerton, S. M. (Stuart M.)", "2010-01-01T00:00:00",          35.5856,          -83.0731,                NA,                 NA,                 NA, "North Carolina",                         NA, "NORTH_AMERICA",   "\"GSMNP The Purchase, nr. house, 35°35'8\"\"N 83°4'23\"\"W, 1945ft, cut lawn/forest edge, Haywood Co., NC\"",  "Haywood",               NA,         "US",      "USA", "United States", "USA.34_1", "North Carolina",    "CC_BY_4_0",                                                                                   "GEODETIC_DATUM_ASSUMED_WGS84;CONTINENT_DERIVED_FROM_COORDINATES;OCCURRENCE_STATUS_INFERRED_FROM_INDIVIDUAL_COUNT;INSTITUTION_MATCH_NONE", "2003-06-27T00:00:00",         NA,              NA,            NA,  27L,     6L, 2003L, "PRESERVED_SPECIMEN",    NA,         "PRESENT",            NA,                                     "Steck, G., Sutton & Super",       NA,    "Malaise trap",              NA,               1L,                   NA,                             NA,    "UCFC 0 237 724", 1800748733L,                                              NA,                  "University of Central Florida Collection of Arthopods (UCFC)",                                                                                                       NA,                   NA,               "urn:lsid:biosci.ohio-state.edu:osuc_occurrences:UCFC__0__237__724",  1349360L,                                     NA,                                              NA,                                              NA,                   NA,                                    "Hylaeus",          "27-Jun-03",                      NA,                   NA,                                                                                                                "lot_id: 6710; preparation: pointed",             NA,                                   NA,            NA,                    NA,                     NA,                                                                                 NA,                   NA,           TRUE,                FALSE,        NA,                     NA,               NA,    "GBIF_Colletidae",                   TRUE,                TRUE,                     TRUE,                         TRUE,                        TRUE,       "Haywood",               TRUE,      NA,         TRUE,        TRUE,          TRUE,         FALSE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,   TRUE,        NA,         TRUE,      TRUE,      TRUE,          TRUE,                     NA,
   "NC_checklist_27278",       "Bombus pensylvanicus (De Geer, 1773)",     "Apidae",      "Apinae",         "Bombus",        NA,      "Bombus pensylvanicus",  "pensylvanicus",                    NA,  "SPECIES",         "(De Geer, 1773)",                       NA,                                                           NA,          NA,                       NA,                      NA,                              NA,                    NA,        36.067333,        -75.690917,                NA,                 NA,                 NA, "North Carolina",                         NA, "NORTH_AMERICA",                                                                                "Kitty Hawk, 703 Kitty Hawk Rd.",     "Dare",               NA,         "US",      "USA", "United States", "USA.34_1", "North Carolina",    "CC_BY_4_0",                                                            "GEODETIC_DATUM_ASSUMED_WGS84;CONTINENT_DERIVED_FROM_COORDINATES;OCCURRENCE_STATUS_INFERRED_FROM_INDIVIDUAL_COUNT;COLLECTION_MATCH_NONE;INSTITUTION_MATCH_FUZZY",                    NA,         NA,              NA,            NA,   NA,     NA,    NA, "PRESERVED_SPECIMEN",    NA,         "PRESENT",            NA,                                                      "D. Boyce",       NA,         "Netting",              NA,               1L,                   NA,                             NA, "AMNH_BEE 00203662",          NA,                                              NA,                                                                          "AMNH",                                                                                       "AMNH Hymenoptera",                   NA,                                            "8632d73a-d8e1-11e2-99a2-0026552be7ea",  1340416L,                                     NA,                                              NA,                                              NA,                   NA,       "Bombus pensylvanicus (De Geer, 1773)",          "7/30/2011",                      NA,                   NA,                                                                                                                                                  NA,       "FEMALE", "American Museum of Natural History",            NA,                    NA,                     NA,                                                                                 NA,                   NA,           TRUE,                FALSE,        NA,                     NA,               NA,        "GBIF_Apidae",                   TRUE,                TRUE,                     TRUE,                         TRUE,                        TRUE,      "Dare Co.",               TRUE,      NA,         TRUE,        TRUE,          TRUE,          TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,   TRUE,        NA,         TRUE,      TRUE,      TRUE,          TRUE,                     NA,
   "NC_checklist_58897",        "Xylocopa virginica (Linnaeus, 1771)",     "Apidae", "Xylocopinae",       "Xylocopa",        NA,        "Xylocopa virginica",      "virginica",                    NA,         NA,        "(Linnaeus, 1771)",                       NA,                                                           NA,          NA,                       NA,                      NA,                         "15734",                "2014",               NA,                NA,                NA,                 NA,                 NA, "North Carolina", "United States of America",              NA,                                                                                                      "Havelock",   "Craven",               NA,           NA,         NA,              NA,         NA,               NA,             NA,                                                                                                                                                                                                                          NA,                    NA,         NA,              NA,            NA,   NA,     4L,    9L,  "PreservedSpecimen",    NA,                NA,            NA,                                                              NA,       NA,                NA,              NA,               NA,                   NA,                             NA,          "X0408943",          NA,                                              NA,                                                                          "TAMU",                                                                                                       NA,                   NA,                                            "e81ba7ce-2187-465f-86fa-797cf6af17fd",        NA,                                     NA, "urn:uuid:cbb99381-6856-4a89-b44e-8a6877e6ca86",                                              NA,                   NA,                         "Xylocopa virginica",   "10/27/2015 11:56",                      NA,                   NA,                                                                                                                                                  NA,             NA,                                   NA,            NA,                    NA,                     NA, "https://scan-bugs.org:443/portal/collections/individual/index.php?occid=48867181",                   NA,             NA,                   NA, 48867181L,                     NA,               NA,        "SCAN_Apidae",                   TRUE,               FALSE,                     TRUE,                         TRUE,                        TRUE,        "Craven",               TRUE,      NA,         TRUE,        TRUE,          TRUE,          TRUE, FALSE,    NA,    NA,    NA,    NA,    NA,    NA,     NA,     FALSE,         TRUE,      TRUE,      TRUE,          TRUE,                     NA,
    "NC_checklist_3249",                   "Colletes Latreille, 1802", "Colletidae",            NA,       "Colletes",        NA,                          NA,               NA,                    NA,    "GENUS",                        NA,                       NA,                                                           NA,          NA,                       NA,                      NA,       "Wiley, J. R. (James R.)", "2002-01-01T00:00:00",         34.69572,         -76.68854,                NA,                 NA,                 NA, "North Carolina",                         NA, "NORTH_AMERICA",                              "Fort Macon State Park, Carteret Co., NC, Carteret, North Carolina, United States", "Carteret",               NA,         "US",      "USA", "United States", "USA.34_1", "North Carolina",    "CC_BY_4_0",                    "GEODETIC_DATUM_ASSUMED_WGS84;CONTINENT_DERIVED_FROM_COORDINATES;TAXON_MATCH_SCIENTIFIC_NAME_ID_IGNORED;OCCURRENCE_STATUS_INFERRED_FROM_INDIVIDUAL_COUNT;INSTITUTION_MATCH_FUZZY;COLLECTION_MATCH_FUZZY",                    NA,         NA,              NA,            NA,   NA,     NA,    NA, "PRESERVED_SPECIMEN",    NA,         "PRESENT",            NA,                                                    "Newman, R.",       NA,        "sweeping",              NA,               1L,                   NA,                             NA,     "FSCA 00106813",          NA,                                              NA,                                                                          "FSCA",                                                                                                       NA,                   NA, "http://mbd-db.osu.edu/hol/collecting_units/305d11e2-be34-4bd1-abce-0eb2cab04092",  1348504L,                                     NA,                                              NA,                                              NA,                   NA,                                   "Colletes",                   NA,                      NA,                   NA,                                                       "[N CAROLINA: Carteret Co. Fort Macon State Park Sweep Netting 20-X-2005 Coll: Randy Newman]",             NA,                                   NA,            NA,                    NA,                     NA,  "http://mbd-db.osu.edu/hol/collecting_units/305d11e2-be34-4bd1-abce-0eb2cab04092",                   NA,           TRUE,                FALSE,        NA,                     NA,               NA,    "GBIF_Colletidae",                   TRUE,                TRUE,                     TRUE,                         TRUE,                        TRUE,      "Carteret",               TRUE,      NA,         TRUE,        TRUE,          TRUE,         FALSE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,   TRUE,        NA,         TRUE,      TRUE,      TRUE,          TRUE,                     NA
  ) %>%
  dplyr::select(database_id, eventDate, fieldNotes, locationRemarks, verbatimEventDate, locality,
                day, month, year)


testOut2 <- BeeBDC::dateFindR(data = troubleRows,
                             # Years above this are removed (from the recovered dates only)
                             maxYear = lubridate::year(Sys.Date()),
                             # Years below this are removed (from the recovered dates only)
                             minYear = 1700) %>%
  dplyr::arrange(desc(database_id))



# Test the expected results
testthat::test_that("dateFindR results successfuly matched", {
  testthat::expect_equal(sum(complete.cases(testOut2$eventDate) ), 18)
})
testthat::test_that("dateFindR results unsuccessfuly matched", {
  testthat::expect_equal(sum(is.na(testOut2$eventDate) ), 1)
})

testthat::test_that("dateFindR output dates match", {
  testthat::expect_equal(testOut2$eventDate %>% as.character(),
                         c("2010-04-14", "2015-10-27", "1956-06-01", "1970-08-13", "2018-07-31", 
                           "1971-08-24", "1970-08-17", "1970-08-13", "1835-08-01",
                           "2003-06-27", "2023-10-01", "2004-07-14", "2005-10-20", "1906-06-20", 
                           "2011-07-30", "1918-06-13", "1921-04-01", "1921-09-01",
                           NA))
})

# Test classes
testthat::test_that("dateFindR expected class", {
  testthat::expect_type(testOut2, "list")
})
testthat::test_that("dateFindR expected class", {
  testthat::expect_equal(attributes(testOut2)$class, c("tbl_df","tbl" ,"data.frame"))
})
