% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/BuyseTest.R
\name{BuyseTest}
\alias{BuyseTest}
\title{Generalized Pairwise Comparisons (GPC)}
\usage{
BuyseTest(formula, data, treatment = NULL, endpoint = NULL, type = NULL,
  threshold = NULL, censoring = NULL, operator = NULL, strata = NULL,
  method.tte = NULL, neutral.as.uninf = NULL, method.inference = NULL,
  n.resampling = NULL, keep.comparison = NULL, alternative = "two.sided",
  seed = NULL, cpus = NULL, trace = NULL, n.bootstrap, method)
}
\arguments{
\item{formula}{[formula] a symbolic description of the model to be fitted. The response variable should be a binary variable defining the treatment arms. 
The rest of the formula should indicate the strata variables (if any) and the endpoints by order of priority.}

\item{data}{[data.frame] dataset.}

\item{treatment}{[character] the name of the treatment variable identifying the control and the experimental group.
Disregarded if the argument \code{formula} is defined.}

\item{endpoint}{[character vector] the name of the endpoint variable(s).
Disregarded if the argument \code{formula} is defined.}

\item{type}{[character vector] the type of each endpoint: \code{"binary"}, \code{"continuous"} or \code{"timeToEvent"}.}

\item{threshold}{[numeric vector] critical values used to compare the pairs.
There must be one threshold for each endpoint variable.
Disregarded if the argument \code{formula} is defined.}

\item{censoring}{[character vector] the name of the binary variable(s) indicating whether the endpoint was observed or censored.
There must be one threshold for each endpoint variable.
Must value \code{NA} when the endpoint is not a time to event.
Disregarded if the argument \code{formula} is defined.}

\item{operator}{[character vector] the sign defining a favorable endpoint:
">0" indicates that higher values are favorable while "<0" indicates the opposite.
Disregarded if the argument \code{formula} is defined.}

\item{strata}{[numeric vector] if not \code{NULL}, the GPC will be applied within each group of patient defined by the strata variable(s).
Disregarded if the argument \code{formula} is defined.}

\item{method.tte}{[character] defines the method used to handle pairs
which can not be decidedly classified as favorable, unfavorable, or neutral because of censored observations (see details).
Can be \code{"Gehan"}, \code{"Gehan corrected"}, \code{"Peron"}, or \code{"Peron corrected"}.
Only relevant when there is one or more time-to-event endpoints.
Default value read from \code{BuyseTest.options()}.}

\item{neutral.as.uninf}{[logical] should paired classified as neutral be re-analysed using endpoints of lower priority.
Default value read from \code{BuyseTest.options()}.}

\item{method.inference}{[character] should a permutation test (\code{"permutation"} or \code{"stratified permutation"}),
or bootstrap resampling (\code{"bootstrap"} or \code{"stratified boostrap"})
be used to compute p-values and confidence intervals.}

\item{n.resampling}{[integer] the number of simulations used for computing the confidence interval and the p.values. See details.
Default value read from \code{BuyseTest.options()}.}

\item{keep.comparison}{[logical] should the result of each pairwise comparison be kept?}

\item{alternative}{[character] the alternative hypothesis.
Must be one of \code{"two.sided"}, \code{"greater"} or \code{"less"}.}

\item{seed}{[integer, >0] the seed to consider for the permutation test.
Default value read from \code{BuyseTest.options()}.}

\item{cpus}{[integer, >0] the number of CPU to use.
Only the permutation test can use parallel computation.
Default value read from \code{BuyseTest.options()}.}

\item{trace}{[integer] should the execution of the function be traced ? See details.
Default value read from \code{BuyseTest.options()}.}

\item{n.bootstrap}{Obsolete. Alias for 'n.resampling'.}

\item{method}{Obsolete. Alias for 'method.tte'.}
}
\value{
An \R object of class \code{\linkS4class{BuyseRes}}.
}
\description{
Performs Generalized Pairwise Comparisons for binary, continuous and time-to-event outcomes.
}
\details{
\bold{treatment:} The variable corresponding to \code{treatment} in data must have only two levels (e.g. \code{0} and \code{1}). \cr
\bold{endpoint, threshold, censoring, operator, and type:}  they must have the same length. \cr
\code{threshold} must be \code{NA} for binary endpoints and positive for continuous or time to event endpoints. \cr
\code{censoring} must be \code{NA} for binary or continuous endpoints and indicate a variable in data for time to event endpoints. 
Short forms for endpoint \code{type} are \code{"bin"} (binary endpoint), \code{"cont"} (continuous endpoint), \
code{"TTE"} (time-to-event endpoint). 
\bold{operator:} when the operator is set to \code{"<0"} the corresponding column in the dataset is multiplied by \code{-1}.

\bold{n.resampling:} The number of permutation replications must be specified to enable the computation of the confidence intervals and the p.value. 
A large number of permutations (e.g. \code{n.resampling=10000}) are needed to obtain accurate CI and p.value. See (Buyse et al., 2010) for more details. 

\bold{trace:} \code{2} reports all messages, \code{1} reports only the percentage of advancement of the permutation test, and \code{0} remains silent.

\bold{cpus parallelization:} Argument \code{cpus} can be set to \code{"all"} to use all available cpus.
The detection of the number of cpus relies on the \code{detectCores} function from the \emph{parallel} package .

\bold{Dealing with neutral or uninformative pairs:} Neutral pairs correspond to pairs for which the difference between the endpoint of the control observation and the endpoint of the treatment observation is (in absolute value) below the threshold. When \code{threshold=0}, neutral pairs correspond to pairs with equal outcome.\cr
Uninformative pairs correspond to pairs for which the censoring prevent from classifying them into favorable, unfavorable or neutral. Neutral or uninformative pairs for an endpoint with priority \code{l} are, when available, analysed on the endpoint with priority \code{l-1}.

\bold{method.tte:} Pairs which can not be decidedly classified as favorable, unfavorable, or neutral because of censored observations can be classified uninformative (\code{method.tte="Gehan"}, Gehan 1965). 
Another solution is to estimate the probability for such pair to be classified as favorable, unfavorable, or neutral based on the survival functions.
\code{method.tte="Peron"} estimates these probabilities using separate Kaplan-Meier estimators of the survival functions for the two groups of patients. 
Probabilities of survival beyond the last observation are set NA, resulting in a non null probability that the pair is informative.
See Peron et al. (2016) for more details. \cr
Due to the presence of uninformative pairs, the proportion of favorable, unfavorable, or neutral pairs is underestimated. 
\code{method.tte="Gehan corrected"} and \code{method.tte="Peron corrected"} aim at correcting this bias
by multiplying the contribution of each pair by the inverse of the total number of pairs minus the number of uninformative pairs
and setting the number of uninformative pairs to 0.
}
\examples{
# reset the default value of the number of permuation sample
BuyseTest.options(method.inference = "none") # no permutation test

#### simulate some data ####
df.data <- simBuyseTest(1e2, n.strata = 2)

                                        # display 
if(require(survival)){
    resKM_tempo <- survfit(Surv(eventtime,status)~Treatment, data = df.data)
    plot(resKM_tempo)
}

#### one time to event endpoint ####
BT <- BuyseTest(Treatment ~ TTE(eventtime, censoring = status), data=df.data)

summary(BT) # net chance in favor of treatment
summary(BT, percentage = FALSE)  
summary(BT, statistic = "winRatio") # win Ratio

## bootstrap to compute the CI
\dontrun{
    BT <- BuyseTest(Treatment ~ TTE(eventtime, censoring = status), data=df.data,
                    method.inference = "permutation", n.resampling = 1e3)
}
\dontshow{
    BT <- BuyseTest(Treatment ~ TTE(eventtime, censoring = status), data=df.data,
                    method.inference = "permutation", n.resampling = 1e1, trace = 0)
}
summary(BT, statistic = "netChance") ## default
summary(BT, statistic = "winRatio") 

## parallel boostrap
\dontrun{
    BT <- BuyseTest(Treatment ~ TTE(eventtime, censoring = status), data=df.data,
                    method.inference = "permutation", n.resampling = 1e3, cpus = 2)
    summary(BT)
}

## method Gehan is much faster but does not optimally handle censored observations
BT <- BuyseTest(Treatment ~ TTE(eventtime, censoring = status), data=df.data,
                method.tte = "Gehan", trace = 0)
summary(BT)

#### one time to event endpoint: only differences in survival over 1 unit ####
BT <- BuyseTest(Treatment ~ TTE(eventtime, threshold = 1, censoring = status), data=df.data)
summary(BT)

#### one time to event endpoint with a strata variable
BT <- BuyseTest(Treatment ~ strata + TTE(eventtime, censoring = status), data=df.data)
summary(BT)

#### several endpoints with a strata variable
f <- Treatment ~ strata + T(eventtime, 1, status) + B(toxicity) 
f <- update(f, 
            ~. + T(eventtime, 0.5, status) + C(score, 1) + T(eventtime, 0.25, status))

BT <- BuyseTest(f, data=df.data)
summary(BT)

#### real example : Veteran dataset of the survival package ####
#### Only one endpoint. Type = Time-to-event. Thresold = 0. Stratfication by histological subtype
#### method.tte = "Gehan"

if(require(survival)){
\dontrun{
  data(veteran,package="survival")
 
  ## method.tte = "Gehan"
  BT_Gehan <- BuyseTest(trt ~ celltype + TTE(time,threshold=0,censoring=status), 
                        data=veteran, method.tte="Gehan",
                        method.inference = "permutation", n.resampling = 1e3)
  
  summary_Gehan <- summary(BT_Gehan)
  summary_Gehan <- summary(BT_Gehan, statistic = "winRatio")
  
  ## method.tte = "Peron"
  BT_Peron <- BuyseTest(trt ~ celltype + TTE(time,threshold=0,censoring=status), 
                        data=veteran, method.tte="Peron",
                        method.inference = "permutation", n.resampling = 1e3)

  class(BT_Peron)
  summary(BT_Peron)
}
}
}
\references{
Marc Buyse (2010). \bold{Generalized pairwise comparisons of prioritized endpoints in the two-sample problem}. \emph{Statistics in Medicine} 29:3245-3257 \cr
D. Wang, S. Pocock (2016). \bold{A win ratio approach to comparing continuous non-normal outcomes in clinical trials}. \emph{Pharmaceutical Statistics} 15:238-245 \cr
J. Peron, M. Buyse, B. Ozenne, L. Roche and P. Roy (2016). \bold{An extension of generalized pairwise comparisons for prioritized outcomes in the presence of censoring}. Statistical Methods in Medical Research. \cr
Efron B (1967). \bold{The two sample problem with censored data}. \emph{Proceedings of the Fifth Berkeley Symposium on Mathematical Statistics and Probability} 4:831-583 \cr
Peto R, Peto J (1972). \bold{Asymptotically efficient rank invariant test procedures}. \emph{Journal of the Royal Statistical Society - series A} 135(2):185-198 \cr
Gehan EA (1965). \bold{A generalized two-sample Wilcoxon test for doubly censored data}. \emph{Biometrika}  52(3):650-653 \cr
}
\seealso{
\code{\link{BuyseRes-summary}} for a summary of the results of generalized pairwise comparison. \cr
\code{\link{BuyseRes-class}} for a presentation of the \code{BuyseRes} object. \cr
\code{\link{constStrata}} to create a strata variable from several clinical variables. \cr
}
\keyword{BuyseTest}
\keyword{function}
