% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calc_nreps.R
\name{calc_nreps}
\alias{calc_nreps}
\title{Determine sample sizes for a set of algorithms on a single problem instance}
\usage{
calc_nreps(instance, algorithms, se.max, dif = "simple",
  comparisons = "all.vs.all", method = "param", nstart = 20,
  nmax = 1000, seed = NULL, boot.R = 499, ncpus = 1,
  force.balanced = FALSE, load.folder = NA, save.folder = NA)
}
\arguments{
\item{instance}{a list object containing the definitions of the problem
instance.
See Section \code{Instance} for details.}

\item{algorithms}{a list object containing the definitions of all algorithms.
See Section \code{Algorithms} for details.}

\item{se.max}{desired upper limit for the standard error of the estimated
difference between pairs of algorithms. See Section
\code{Pairwise Differences} for details.}

\item{dif}{type of difference to be used. Accepts "perc" (for percent
differences) or "simple" (for simple differences)}

\item{comparisons}{type of comparisons being performed. Accepts "all.vs.first"
(in which cases the first object in \code{algorithms} is considered to be
the reference algorithm) or "all.vs.all" (if there is no reference
and all pairwise comparisons are desired).}

\item{method}{method to use for estimating the standard errors. Accepts
"param" (for parametric) or "boot" (for bootstrap)}

\item{nstart}{initial number of algorithm runs for each algorithm.
See Section \code{Initial Number of Observations} for details.}

\item{nmax}{maximum \strong{total} allowed number of runs to execute. Loaded
results (see \code{load.folder} below) do not count towards this
total.}

\item{seed}{seed for the random number generator}

\item{boot.R}{number of bootstrap resamples to use (if \code{method == "boot"})}

\item{ncpus}{number of cores to use}

\item{force.balanced}{logical flag to force the use of balanced sampling for
the algorithms on each instance}

\item{load.folder}{name of folder to load results from. Use either "" or
"./" for the current working directory. Accepts relative paths.
Use \code{NA} for not saving. \code{calc_nreps()} will look for a .RDS file
with the same name}

\item{save.folder}{name of folder to save the results. Use either "" or
"./" for the current working directory. Accepts relative paths.
Use \code{NA} for not saving.}
}
\value{
a list object containing the following items:
\itemize{
\item \code{instance} - alias for the problem instance considered
\item \code{Xk} - list of observed performance values for all \code{algorithms}
\item \code{Nk} - vector of sample sizes generated for each algorithm
\item \code{Diffk} - data frame with point estimates, standard errors and
other information for all algorithm pairs of interest
\item \code{seed} - seed used for the PRNG
\item \code{dif} - type of difference used
\item \code{method} - method used ("param" / "boot")
\item \code{comparisons} - type of pairings ("all.vs.all" / "all.vs.first")
}
}
\description{
Iteratively calculates the required sample sizes for K algorithms
on a given problem instance, so that the standard errors of the estimates of
the pairwise differences in performance is controlled at a predefined level.
}
\section{Instance}{

Parameter \code{instance} must be a named list containing all relevant parameters
that define the problem instance. This list must contain at least the field
\code{instance$FUN}, with the name of the function implementing the problem
instance, that is, a routine that calculates y = f(x). If the instance
requires additional parameters, these must also be provided as named fields.
}

\section{Algorithms}{

Object \code{algorithms} is a list in which each component is a named
list containing all relevant parameters that define an algorithm to be
applied for solving the problem instance. In what follows \code{algorithm[[k]]}
refers to any algorithm specified in the \code{algorithms} list.

\code{algorithm[[k]]} must contain an \code{algorithm[[k]]$FUN} field, which is a
character object with the name of the function that calls the algorithm; as
well as any other elements/parameters that \code{algorithm[[k]]$FUN} requires
(e.g., stop criteria, operator names and parameters, etc.).

The function defined by the routine \code{algorithm[[k]]$FUN} must have the
following structure: supposing that the list in \code{algorithm[[k]]} has
fields \code{algorithm[[k]]$FUN = "myalgo"}, \code{algorithm[[k]]$par1 = "a"} and
\code{algorithm$par2 = 5}, then:

\preformatted{
         myalgo <- function(par1, par2, instance, ...){
               # do stuff
               # ...
               return(results)
         }
   }

That is, it must be able to run if called as:

\preformatted{
         # remove '$FUN' and '$alias' fields from list of arguments
         # and include the problem definition as field 'instance'
         myargs          <- algorithm[names(algorithm) != "FUN"]
         myargs          <- myargs[names(myargs) != "alias"]
         myargs$instance <- instance

         # call function
         do.call(algorithm$FUN,
                 args = myargs)
   }

The \code{algorithm$FUN} routine must return a list containing (at
least) the performance value of the final solution obtained, in a field named
\code{value} (e.g., \code{result$value}) after a given run.
}

\section{Initial Number of Observations}{

In the \strong{general case} the initial number of observations per algorithm
(\code{nstart}) should be relatively high. For the parametric case
we recommend between 10 and 20 if outliers are not expected, or between 30
and 50 if that assumption cannot be made. For the bootstrap approach we
recommend using at least 20. However, if some distributional assumptions can
be made - particularly low skewness of the population of algorithm results on
the test instances), then \code{nstart} can in principle be as small as 5 (if the
output of the algorithms were known to be normal, it could be 1).

In general, higher sample sizes are the price to pay for abandoning
distributional assumptions. Use lower values of \code{nstart} with caution.
}

\section{Pairwise Differences}{

Parameter \code{dif} informs the type of difference in performance to be used
for the estimation (\eqn{\mu_a} and \eqn{\mu_b} represent the mean
performance of any two algorithms on the test instance, and \eqn{mu}
represents the grand mean of all algorithms given in \code{algorithms}):
\itemize{
\item If \code{dif == "perc"} and \code{comparisons == "all.vs.first"}, the estimated quantity is
\eqn{\phi_{1b} = (\mu_1 - \mu_b) / \mu_1 = 1 - (\mu_b / \mu_1)}.
\item If \code{dif == "perc"} and \code{comparisons == "all.vs.all"}, the estimated quantity is
\eqn{\phi_{ab} = (\mu_a - \mu_b) / \mu}.
\item If \code{dif == "simple"} it estimates \eqn{\mu_a - \mu_b}.
}
}

\examples{
# Example using dummy algorithms and instances. See ?dummyalgo for details.
# We generate dummy algorithms with true means 15, 10, 30, 15, 20; and true
# standard deviations 2, 4, 6, 8, 10.
algorithms <- mapply(FUN = function(i, m, s){
                          list(FUN   = "dummyalgo",
                               alias = paste0("algo", i),
                               distribution.fun  = "rnorm",
                               distribution.pars = list(mean = m, sd = s))},
                     i = c(alg1 = 1, alg2 = 2, alg3 = 3, alg4 = 4, alg5 = 5),
                     m = c(15, 10, 30, 15, 20),
                     s = c(2, 4, 6, 8, 10),
                     SIMPLIFY = FALSE)

# Make a dummy instance with a centered (zero-mean) exponential distribution:
instance = list(FUN = "dummyinstance", distr = "rexp", rate = 5, bias = -1/5)

# Explicitate all other parameters (just this one time:
# most have reasonable default values)
myreps <- calc_nreps(instance   = instance,
                      algorithms = algorithms,
                      se.max     = 0.05,          # desired (max) standard error
                      dif        = "perc",        # type of difference
                      comparisons = "all.vs.all", # differences to consider
                      method     = "param",       # method ("param", "boot")
                      nstart     = 15,            # initial number of samples
                      nmax       = 1000,          # maximum allowed sample size
                      seed       = 1234,          # seed for PRNG
                      boot.R     = 499,           # number of bootstrap resamples (unused)
                      ncpus      = 1,             # number of cores to use
                      force.balanced = FALSE,     # force balanced sampling?
                      load.folder   = NA,         # file to load results from
                      save.folder = NA)         # folder to save results
summary(myreps)
}
\references{
\itemize{
\item F. Campelo, F. Takahashi:
Sample size estimation for power and accuracy in the experimental
comparison of algorithms. Journal of Heuristics 25(2):305-338, 2019.
\item P. Mathews.
Sample size calculations: Practical methods for engineers and scientists.
Mathews Malnar and Bailey, 2010.
\item A.C. Davison, D.V. Hinkley:
Bootstrap methods and their application. Cambridge University Press (1997)
\item E.C. Fieller:
Some problems in interval estimation. Journal of the Royal Statistical
Society. Series B (Methodological) 16(2), 175–185 (1954)
\item V. Franz:
Ratios: A short guide to confidence limits and proper use (2007).
https://arxiv.org/pdf/0710.2024v1.pdf
\item D.C. Montgomery, C.G. Runger:
Applied Statistics and Probability for Engineers, 6th ed. Wiley (2013)
}
}
\author{
Felipe Campelo (\email{fcampelo@gmail.com})
}
