% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fitFunction.R
\name{fitFunction}
\alias{fitFunction}
\title{Fit Function to Data via Nonlinear Regression}
\usage{
fitFunction(
  dataset = cs.in.dataset(),
  preds = cs.in.predictors(),
  resps = cs.in.responses(),
  groups = cs.in.groupvars(),
  auxs = cs.in.auxiliaries(),
  scriptvars = cs.in.scriptvars(),
  return.results = FALSE,
  ...
)
}
\arguments{
\item{dataset}{[\code{\link{data.frame}}]\cr
Dataset with named columns. The names correspond to predictors and responses.}

\item{preds}{[\code{character}]\cr
Character vector of predictor variables.}

\item{resps}{[\code{character}]\cr
Character vector of response variables.}

\item{groups}{[\code{character}]\cr
Character vector of group variables.}

\item{auxs}{[\code{character}]\cr
Character vector of auxiliary variables.}

\item{scriptvars}{[\code{list}]\cr
Named list of script variables set via the Cornerstone "Script Variables" menu.
For details see below.}

\item{return.results}{[\code{logical(1)}]\cr
If \code{FALSE} the function returns \code{TRUE} invisibly.
If \code{TRUE}, it returns a \code{\link{list}} of results.
Default is \code{FALSE}.}

\item{...}{[ANY]\cr
Additional arguments to be passed to 
\code{\link{nls}}
. Please consider possible script variables (\code{scriptvars}) to prevent duplicates.}
}
\value{
Logical [\code{TRUE}] invisibly and outputs to Cornerstone or, 
  if \code{return.results = TRUE}, \code{\link{list}} of 
  resulting \code{\link{data.frame}} objects:
  \item{coeff}{Estimated coefficients and standard errors for each group.
    Convergence information is available for every group (for details see
    \code{link[minpack.lm]{nls.lm}}).
  }
  \item{vcov}{Variance-Covariance matrix of the main coefficients for the fitted model of each
    group (for details see \code{link[stats]{vcov}}).
  }
  \item{predictions}{
    Dataset to brush with predictions and residuals added to original values and groups,
    if available.
  }
}
\description{
Fit predefined functions to data via nonlinear least squares using Levenberg-Marquardt
  algorithm via \code{\link[minpack.lm]{nlsLM}}.
}
\details{
The following script variables are summarized in \code{scriptvars} list:\cr
  \describe{
    \item{math.fun}{[\code{character(1)}]\cr
      Function selection for fitting data. It is possible to choose a predefined model, or
      compose a model manually by selecting \code{User Defined}.\cr
      Default is \code{User Defined}}
    \item{preds.frml}{[\code{character(1)}]\cr
      Required if \code{math.fun} is set to \code{User Defined}.
      Valid R \code{\link{formula}} for the right hand side (predictors) of the model equation.}
    \item{resp.frml}{[\code{character(1)}]\cr
      Required if \code{math.fun} is set to \code{User Defined}.
      Valid R \code{\link{formula}} for the left hand side (response) of the model equation.}
    \item{limits}{[\code{character(1)}]\cr
      Optional if \code{math.fun} is set to \code{User Defined}.
      Specifies minimum and maximum value for function \code{math.fun} as a comma separated list
      of \code{min} and \code{max}.
      It is possible to assign variables, e.g. \code{min=a}, which need start values in
      \code{start.vals}, as well as real numbers, e.g. \code{min=4.5}, with a period as decimal
      separator.}
    \item{start.vals}{[\code{character(1)}]\cr
      Required if \code{math.fun} is set to \code{User Defined}.
      Specify starting values for all terms of the right hand side as a comma separated list
      with a period as decimal separator.}
    \item{weights}{[\code{character(1)}]\cr
      Select a weighting variable from the auxiliary variables.}
    \item{max.iter}{Maximum number of iterations.
      For details see \code{link[minpack.lm]{nls.lm.control}}}
    \item{max.ftol}{Maximum relative error desired in the sum of squares. If \code{0},
      the default is used.
      For details see \code{link[minpack.lm]{nls.lm.control}}}
  }
}
\examples{
# Generate data from logistic function:
fun = function(x, a, b, c, d, sigma = 1) {
  a+(b-a) / (1+exp(-d*(x-c))) + rnorm(length(x), sd = sigma)
  }
library(data.table)
dt = data.table(  x1 = sample(seq(-10, 10, length.out = 100))
                  , group1 = sample(x = c("A", "B"), replace = TRUE, size = 100)
                  )
dt[group1 == "A", y1 := fun(x1, 1, 10, 1, 0.6, 0.1)]
dt[group1 == "B", y1 := fun(x1, 8, 2, -1, 0.3, 0.1)]
# Set script variables
scriptvars = list(math.fun = "Logistic", resp.frml = "", preds.frml = "", limits = ""
                  , start.vals = "", weights = "", max.iter = 50, max.ftol = 0
                  )
# Fit the logistic function:
res = fitFunction(dt, "x1", "y1", "group1", character(0), scriptvars, TRUE)
# Show estimated coefficients:
res$coeff
# Variance-Covariance matrix:
res$vcov
# Plot fitted vs. residuals:
plot(res$predictions$Fitted, res$predictions$Residuals)
}
