% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bw-pi.R
\name{bw_dir_pi}
\alias{bw_dir_pi}
\alias{bw_dir_rot}
\alias{bw_dir_ami}
\alias{R_Psi_mixvmf}
\alias{bw_dir_emi}
\title{Plug-in bandwidth selectors for directional data}
\usage{
bw_dir_rot(data)

bw_dir_ami(data, fit_mix = NULL, L = NULL)

R_Psi_mixvmf(q, mu, kappa, p)

bw_dir_emi(data, fit_mix = NULL, optim = TRUE,
  h_grid = exp(seq(log(0.05), log(1.5), l = 100)), plot_it = TRUE,
  optim_par = 0.25, optim_lower = 0.06, optim_upper = 10)
}
\arguments{
\item{data}{directional data, a matrix of size \code{c(n, q + 1)}.}

\item{fit_mix}{output from \code{\link{bic_vmf_mix}}. Computed internally
if \code{NULL} (default).}

\item{L}{kernel function. Set internally to \code{function(x) exp(-x)}
(von Mises--Fisher kernel) if \code{NULL} (default).}

\item{q}{dimension of \eqn{S^q}, \eqn{q\ge 1}.}

\item{mu, kappa, p}{mixture parameters. \code{mu} is the mean matrix of size
\code{c(length(p), q + 1)}, \code{kappa} is vector of \code{length(p)}
concentration parameters, and \code{p} is the vector of mixture proportions.}

\item{optim}{run an optimization? Defaults to \code{TRUE}. Otherwise,
a grid search on \code{h} is done. Only effective if \code{L = NULL}.}

\item{h_grid}{vector of bandwidths for performing a grid search. Defaults
to\cr \code{exp(seq(log(0.05), log(1.5), l = 100))}.}

\item{plot_it}{display an informative plot on the optimization's grid search?
Defaults to \code{FALSE}.}

\item{optim_par, optim_lower, optim_upper}{parameters passed to \code{par},
\code{lower}, and \code{upper} in \code{\link[stats]{optim}} when using
the \code{"L-BFGS-B"} method. Default to \code{0.25}, \code{0.06}
(to avoid numerical instabilities), and \code{10}.}
}
\value{
Selected bandwidth for \code{bw_dir_rot} and \code{bw_dir_ami}.
\code{bw_dir_emi} returns a list with entries:
\itemize{
  \item{\code{h_opt}: cross-validation bandwidth.}
  \item{\code{h_grid}: \code{h_grid}, if used (otherwise \code{NULL}).}
  \item{\code{MISE_opt}: minimum of the MISE loss.}
  \item{\code{MISE_grid}: value of the MISE function at \code{h_grid}, if
   used (otherwise \code{NULL}).}
}
}
\description{
Plug-in bandwidth selectors for kernel density estimation
with directional data, including Rule-Of-Thumb (ROT),
Asymptotic MIxtures (AMI), and Exact MIxtures (EMI).
}
\details{
See Algorithms 1 (AMI) and 2 (EMI) in García-Portugués (2013). The ROT
selector is implemented according to Proposition 2, \bold{but} without
the paper's typo in equation (6), case \eqn{q = 2}, where an incorrect
extra \eqn{\hat\kappa} appears premultiplying
\eqn{(1 + 4 \hat\kappa^2) \sinh(2 \hat\kappa)} in the denominator.

\code{bw_dir_ami} uses \code{R_Psi_mixvmf} for computing the curvature
term of a mixture of von Mises--Fisher densities.

\code{bw_dir_emi} employs Monte Carlo integration for \eqn{q > 2}, which
results in a random output. Use \code{set.seed} before to avoid it.
}
\examples{
# Sample
n <- 25
q <- 2
set.seed(42)
samp <- rotasym::r_vMF(n = n, mu = c(1, rep(0, q)), kappa = 2)

# Mixture fit
fit_mix <- bic_vmf_mix(data = samp, plot_it = TRUE)

# ROT
bw_dir_rot(samp)

# AMI
bw_dir_ami(samp)
bw_dir_ami(samp, fit_mix = fit_mix)
bw_dir_ami(samp, fit_mix = fit_mix, L = function(x) exp(-x))

# EMI
bw_dir_emi(samp)
bw_dir_emi(samp, fit_mix = fit_mix, optim = FALSE, plot_it = TRUE)
}
\references{
García-Portugués, E. (2013). Exact risk improvement of bandwidth selectors
for kernel density estimation with directional data. \emph{Electronic
Journal of Statistics}, 7:1655--1685.
\doi{10.1214/13-ejs821}
}
