% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{kernel}
\alias{kernel}
\alias{kernel_DBH_fast}
\alias{kernel_DBH_crit}
\alias{kernel_ADBH_fast}
\alias{kernel_ADBH_crit}
\alias{kernel_DBR_fast}
\alias{kernel_DBR_crit}
\title{Kernel functions}
\usage{
kernel_DBH_fast(pCDFlist, pvalues, stepUp = FALSE, alpha = 0.05,
  support = 0L)

kernel_DBH_crit(pCDFlist, pvalues, sorted_pv, stepUp = FALSE,
  alpha = 0.05)

kernel_ADBH_fast(pCDFlist, pvalues, stepUp = FALSE, alpha = 0.05,
  support = 0L)

kernel_ADBH_crit(pCDFlist, pvalues, sorted_pv, stepUp = FALSE,
  alpha = 0.05)

kernel_DBR_fast(pCDFlist, pvalues, lambda = 0.05)

kernel_DBR_crit(pCDFlist, pvalues, sorted_pv, lambda = 0.05,
  alpha = 0.05)
}
\arguments{
\item{pCDFlist}{a list of the supports of the CDFs of the p-values. Each support is represented by a vector that must be in increasing order.}

\item{pvalues}{a numeric vector. Contains all values of the p-values supports if we search for the critical constants. If not, contains only the observed p-values. Must be sorted in increasing order!}

\item{stepUp}{a numeric vector. Identical to \code{pvalues} for a step-down procedure. Equals \code{c.m} for a step-up procedure.}

\item{alpha}{the target FDR level, a number strictly between 0 and 1. For \code{*.fast} kernels, it is only necessary, if \code{stepUp = TRUE}.}

\item{support}{a numeric vector. Contains all values of the p-values supports. Ignored, if \code{stepUp = FALSE}. Must be sorted in increasing order!}

\item{sorted_pv}{a vector of observed p-values, in increasing order.}

\item{lambda}{a number strictly between 0 and 1. If \code{lambda=NULL} (by default), then \code{lambda} is chosen equal to \code{alpha}.}
}
\value{
For \code{kernel.DBH.fast}, \code{kernel.ADBH.fast} and
\code{kernel.DBR.fast}, a vector of transformed p-values is returned.
\code{kernel.DBH.crit,} \code{kernel.ADBH.crit} and \code{kernel.DBR.crit}
return a list object with critical constants (\code{$crit.consts}) and
transformed p-values (\code{$pval.transf}), but if \code{stepUp = FALSE},
there are critical values only.
}
\description{
Kernel functions that transform observed p-values or their support according
to [HSU], [HSD], [AHSU], [AHSD] and [HBR-\eqn{\lambda}]. The output is used
by \code{\link{discrete.BH}} or \code{\link{DBR}}, respectively.
Additionally, \code{kernel.DBH.crit}, \code{kernel.ADBH.crit} and
\code{kernel.DBR.crit} compute and return the critical constants. The end
user should not use these functions directly.
}
\details{
When computing critical constants under step-down, that is, when using
\code{kernel.DBH.crit}, \code{kernel.ADBH.crit} or \code{kernel.DBR.crit}
with \code{stepUp = FALSE} (i.e. the step-down case), we still need to get
transformed p-values to compute the adjusted p-values.

This version: 2019-11-15.
}
\examples{
X1 <- c(4, 2, 2, 14, 6, 9, 4, 0, 1)
X2 <- c(0, 0, 1, 3, 2, 1, 2, 2, 2)
N1 <- rep(148, 9)
N2 <- rep(132, 9)
Y1 <- N1 - X1
Y2 <- N2 - X2
df <- data.frame(X1, Y1, X2, Y2)
df

#Construction of the p-values and their support
df.formatted <- fisher.pvalues.support(counts = df, input = "noassoc")
raw.pvalues <- df.formatted$raw
pCDFlist <- df.formatted$support

alpha <- 0.05

# Compute the step functions from the supports

# We stay in a step-down context, where pv.numer = pv.denom,
# for the sake of simplicity

# If not searching for critical constants, we use only the observed p-values
sorted.pvals <- sort(raw.pvalues)
y.DBH.fast <- kernel_DBH_fast(pCDFlist, sorted.pvals)
y.ADBH.fast <- kernel_ADBH_fast(pCDFlist, sorted.pvals)
# transformed values
y.DBH.fast
y.ADBH.fast

# compute transformed support
pv.list <- sort(unique(unlist(pCDFlist)))
y.DBH.crit <- kernel_DBH_crit(pCDFlist, pv.list, sorted.pvals)
y.ADBH.crit <- kernel_ADBH_crit(pCDFlist, pv.list, sorted.pvals)
y.DBR.crit <- kernel_DBR_crit(pCDFlist, pv.list, sorted.pvals)
# critical constants
y.DBH.crit$crit.consts
y.ADBH.crit$crit.consts
y.DBR.crit$crit.consts
# The following exist only for step-down direction or DBR
y.DBH.crit$pval.transf
y.ADBH.crit$pval.transf
y.DBR.crit$pval.transf

}
\seealso{
\code{\link{discrete.BH}}, \code{\link{DiscreteFDR}},
\code{\link{DBR}}
}
