% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/P_owl.R
\name{owl}
\alias{owl}
\title{Outcome Weighted Learning}
\usage{
owl(
  ...,
  moPropen,
  data,
  reward,
  txName,
  regime,
  response,
  lambdas = 2,
  cvFolds = 0L,
  kernel = "linear",
  kparam = NULL,
  surrogate = "hinge",
  verbose = 2L
)
}
\arguments{
\item{...}{Used primarily to require named input. However, inputs for
the optimization methods can be sent through the ellipsis. If surrogate
is hinge, the optimization method is kernlab::ipop(). For all other 
surrogates, stats::optim() is used.}

\item{moPropen}{An object of class modelObj, which defines the model and
R methods to be used to obtain parameter estimates and
predictions for the propensity for treatment.
See ?moPropen for details.}

\item{data}{A data frame of the covariates and tx histories}

\item{reward}{The response vector}

\item{txName}{A character object.
The column header of \emph{data} that corresponds to the tx covariate}

\item{regime}{A formula object or a character vector.
The covariates to be included in classification}

\item{response}{A numeric vector.
The reward. Allows for naming convention followed in most
DynTxRegime methods.}

\item{lambdas}{A numeric object or a numeric vector object giving the
penalty tuning parameter. If more than 1 is provided,
the finite set of values to be considered in the
cross-validation algorithm}

\item{cvFolds}{If cross-validation is to be used to select the tuning
parameters, the number of folds.}

\item{kernel}{A character object.
must be one of {linear, poly, radial}}

\item{kparam}{A numeric object of NULL.
If kernel = linear, kparam is ignored.
If kernel = poly, kparam is the degree of the polynomial
If kernel = radial, kparam is the inverse bandwidth of the
kernel. If a vector of bandwidth parameters is given,
cross-validation will be used to select the parameter}

\item{surrogate}{The surrogate 0-1 loss function must be one of
logit, exp, hinge, sqhinge, huber}

\item{verbose}{An integer or logical. If 0, no screen prints are generated. If 1,
screen prints are generated with the exception of optimization results
obtained in iterative algorithm. If 2, all screen prints are generated.}
}
\value{
an OWL object
}
\description{
Outcome Weighted Learning
}
\examples{
 
# Load and process data set
data(bmiData)

# define the negative 12 month change in BMI from baseline
y12 <- -100*(bmiData[,6L] - bmiData[,4L])/bmiData[,4L]

# propensity model
moPropen <- buildModelObj(model = ~parentBMI+month4BMI,
                          solver.method = 'glm',
                          solver.args = list('family'='binomial'),
                          predict.method = 'predict.glm',
                          predict.args = list(type='response'))

fitOWL <- owl(moPropen = moPropen,
              data = bmiData, reward = y12,  txName = 'A2', 
              regime = ~ parentBMI + month4BMI,
              surrogate = 'hinge', kernel = 'linear', kparam = NULL)

##Available methods

  # Coefficients of the propensity score regression
  coef(fitOWL)

  # Description of method used to obtain object
  DTRstep(fitOWL)

  # Estimated value of the optimal treatment regime for training set
  estimator(fitOWL)

  # Value object returned by propensity score regression method
  fitObject(fitOWL)

  # Summary of optimization routine
  optimObj(fitOWL)

  # Estimated optimal treatment for training data
  optTx(fitOWL)

  # Estimated optimal treatment for new data
  optTx(fitOWL, bmiData)

  # Plots if defined by propensity regression method
  dev.new()
  par(mfrow = c(2,4))

  plot(fitOWL)
  plot(fitOWL, suppress = TRUE)

  # Value object returned by propensity score regression method
  propen(fitOWL)

  # Parameter estimates for decision function
  regimeCoef(fitOWL)

  # Show main results of method
  show(fitOWL)

  # Show summary results of method
  summary(fitOWL)
 
}
\references{
Yingqi Zhao, Donglin Zeng, A. John Rush,
  Michael R. Kosorok (2012)
 Estimated individualized treatment rules using outcome weighted
 learning. Journal of the American Statistical Association,
 107(409): 1106-1118. PMCID: 3636816
}
\seealso{
Other statistical methods: 
\code{\link{bowl}()},
\code{\link{earl}()},
\code{\link{iqLearn}},
\code{\link{optimalClass}()},
\code{\link{optimalSeq}()},
\code{\link{qLearn}()},
\code{\link{rwl}()}

Other weighted learning methods: 
\code{\link{bowl}()},
\code{\link{earl}()},
\code{\link{rwl}()}

Other single decision point methods: 
\code{\link{earl}()},
\code{\link{optimalClass}()},
\code{\link{optimalSeq}()},
\code{\link{qLearn}()},
\code{\link{rwl}()}
}
\concept{single decision point methods}
\concept{statistical methods}
\concept{weighted learning methods}
