% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/PBsampler.R
\name{PBsampler}
\alias{PBsampler}
\title{Parametric bootstrap sampler for lasso, group lasso, scaled lasso or scaled group lasso estimator}
\usage{
PBsampler(X, PE_1, sig2_1, lbd_1, PE_2, sig2_2, lbd_2, weights = rep(1,
  max(group)), group = 1:ncol(X), niter = 2000, type, PEtype = "coeff",
  Btype = "gaussian", Y = NULL, parallel = FALSE, ncores = 2L,
  verbose = FALSE)
}
\arguments{
\item{X}{predictor matrix.}

\item{PE_1, sig2_1, lbd_1}{parameters of target distribution.
(point estimate of beta or \code{E(y)} depends on \code{PEtype}, variance estimate of error and lambda)
sig2_1 is only needed when \code{Btype = "wild"}.}

\item{PE_2, sig2_2, lbd_2}{additional parameters of target distribution. This is required
only if mixture distribution is used. sig2_2 is only needed when \code{Btype = "wild"}.}

\item{weights}{weight vector with length equal to the number of groups. Default is
\code{rep(1, max(group))}.}

\item{group}{\code{p} x \code{1} vector of consecutive integers describing the group structure.
The number of groups should be the same as max(group). Default is \code{group = 1:p}
, where \code{p} is number of covariates. See examples for a guideline.}

\item{niter}{integer. The number of iterations. Default is \code{niter = 2000}}

\item{type}{type of penalty. Must be specified to be one of the following:
\code{"lasso", "grlasso", "slasso"} or \code{"sgrlasso"}.}

\item{PEtype}{Type of \code{PE} which is needed to characterize the target distribution.
Users can choose either \code{"coeff"} or \code{"mu"}.}

\item{Btype}{Type of bootstrap method. Users can choose either \code{"gaussian"}
for guassian bootstrap or \code{"wild"} for wild multiplier bootstrap. Default
is \code{"gaussian"}.}

\item{Y}{response vector. This is only required when \code{Btype = "wild"}.}

\item{parallel}{logical. If \code{parallel = TRUE}, uses parallelization.
Default is \code{parallel = FALSE}.}

\item{ncores}{integer. The number of cores to use for parallelization.}

\item{verbose}{logical. This works only when
\code{parallel = FALSE}.}
}
\value{
\item{beta}{coefficient estimate.}

\item{subgrad}{subgradient.}

\item{hsigma}{standard deviation estimator, for type="slasso" or type="sgrlasso" only.}

\item{X, PE, sig2, weights, group, type, PEtype, Btype, Y, mixture}{model parameters.}
}
\description{
Draw gaussian bootstrap or wild multiplier bootstrap samples for
lasso, group lasso, scaled lasso and scaled group lasso estimators along with their subgradients.
}
\details{
This function provides bootstrap samples for lasso, group lasso,
scaled lasso or scaled group lasso estimator
and its subgradient. \cr
The sampling distribution is chracterized by \code{(PE, sig2, lbd)}.
If \code{Btype = "gaussian"}, \code{error_new} is generated from \code{N(0, sig2)}.
If \code{Btype = "wild"}, we first generate \code{error_new} from \code{N(0, 1)}
 and multiply with the residuals.
Then, if \code{PEtype = "coeff"}, \code{y_new} is generated by \code{X * PE + error_new}
and if \code{PEtype = "mu"}, \code{y_new} is generated by \code{PE + error_new}. \cr
By providing \code{(PE_2, sig2_2, lbd_2)}, this function simulates from a mixture distribution.
With 1/2 probability, samples will be drawn from the distribution with parameters
(PE_1, sig2_1, lbd_1) and with another 1/2 probability, they will be drawn from
the distribution with parameters (PE_2, sig2_2, lbd_2).
Four distict penalties can be used; \code{"lasso"} for lasso, \code{"grlasso"} for group lasso,
\code{"slasso"} for scaled lasso and \code{"sgrlasso"} for scaled group lasso.
See Zhou(2014) and Zhou and Min(2017) for details.
}
\examples{
set.seed(1234)
n <- 10
p <- 30
Niter <-  10
Group <- rep(1:(p/10), each = 10)
Weights <- rep(1, p/10)
x <- matrix(rnorm(n*p), n)
#
# Using non-mixture distribution
#
PBsampler(X = x, PE_1 = rep(0, p), sig2_1 = 1, lbd_1 = .5,
 weights = Weights, group = Group, type = "grlasso", niter = Niter, parallel = FALSE)
PBsampler(X = x, PE_1 = rep(0, p), sig2_1 = 1, lbd_1 = .5,
 weights = Weights, group = Group, type = "grlasso", niter = Niter, parallel = TRUE)
#
# Using mixture distribution
#
PBsampler(X = x, PE_1 = rep(0, p), sig2_1 = 1, lbd_1 = .5,
 PE_2 = rep(1, p), sig2_2 = 2, lbd_2 = .3, weights = Weights,
 group = Group, type = "grlasso", niter = Niter, parallel = TRUE)
}
\references{
Zhou, Q. (2014), "Monte Carlo simulation for Lasso-type problems by estimator augmentation,"
Journal of the American Statistical Association, 109, 1495-1516.

Zhou, Q. and Min, S. (2017), "Estimator augmentation with applications in
high-dimensional group inference," Electronic Journal of Statistics, 11(2), 3039-3080.
}
