\name{EDA.ecdf}
\alias{EDA.ecdf}
\title{
Distribution estimation algorithm with empirical cumulative distribution function (ECDF).
}
\description{
Implements an EDA with ECDF.
}
\usage{
EDA.ecdf(fun,
  lower,
  upper,
  n = 30,
  maxiter,
  k = 2,
  k0 = 10,
  tolerance = 0.01,
  ...)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
\item{fun}{
  A function to be minimized, with first argument the vector of parameters over which minimization is to take place.
  }

\item{lower}{
    Lower limits of the parameters.
  }
\item{upper}{
    Upper bounds on the parameters.
  }
\item{n}{Number of individuals per generation}

\item{maxiter}{Maximum number of iterations.}

\item{k}{The minimum number of consecutive generations using the same search distribution.}

\item{k0}{The minimum number of consecutive generations using the uniform distribution.}

\item{tolerance}{Criterion for determining whether the distribution has changed.}

\item{...}{Additional arguments of the objective function.}


}

\details{
Implements an EDA with ecdf that uses the proposal presented in Salinas-Gutiérrez and Zavala, 2023 before changing the model.
}
\value{
Returns a list with the following entries:
\item{\code{sol}}{The best solution found.}
\item{\code{par}}{The n best individuals generated by the last model.}
\item{\code{value}}{The best value of the objective function.}
\item{\code{historical}}{The best fitness value in each generatio.}
}

\examples{

#Example happy cat
fun <- function(X){
  D <- length(X)
  f <- abs(sum(X^2) - D)^(1/4) + (0.5 * sum(X^2) + sum(X))/D + 0.5
  return(f)
}
n <- 30
k <- 2
tolerance <- 0.01
lower <- rep(-5,2)
upper <- rep(5,2)
res <- EDA.ecdf(fun, lower = lower,
                upper = upper,n = n,
                k = k, tolerance = tolerance,
                maxiter = 200)
z <- outer(X = seq(-5, 5, 0.05), Y = seq(-5, 5, 0.05),
           FUN = Vectorize(function(X, Y) { fun(c(X, Y)) }))
contour(seq(-5, 5, 0.05),seq(-5, 5, 0.05),z,
        nlevels = 20, cex.axis = .8)
points(res$sol[1],res$sol[2],
       col = "blue", pch = 19)

#Multiple regression example
set.seed(pi)
x1 <- rnorm(n = 100, mean = 10, sd = 5)
x2 <- rnorm(n = 100, mean = 8, sd = 2)
y <- 7 + 4 * x1 - 6 * x2 + rnorm(n = 100, mean = 0, sd = 3)

ec <- function(par,y,x1,x2){
  b0 <- par[1]
  b1 <- par[2]
  b2 <- par[3]
  y_hat <- b0 + b1 * x1 + b2 * x2
  value <- sum((y - y_hat)^2)
  return(value)
}

opt <- EDA.ecdf(fun = ec, lower = c(-20,-20,-20),
                 upper = c(20,20,20), maxiter = 150,
                 y = y, x1 = x1, x2 = x2)
opt$sol
}

\references{
{ Salinas-Gutiérrez, R., & Zavala, A. E. M. (2023).An explicit exploration strategy for
 evolutionary algorithms. Applied Soft Computing, 140.
 https://doi.org/10.1016/j.asoc.2023.110230}
}

\seealso{
\code{\link{EDA.hist}}, \code{\link{EDA.mnorm}},
\code{\link{EDA.norm}}, \code{\link{ExplicitExploration}}
}
