\name{weierstrass}
\alias{weierstrass}
\title{Weierstrass Function for Benchmarking Optimization Algorithms}

\usage{
weierstrass(x,
            a = 0.5,
            b = 3,
            kmax = 20)
}

\arguments{
\item{x}{
  A numeric vector representing the input variables. The length of \code{x} determines the dimensionality of the problem.
  }

\item{a}{
  A numeric value representing the scaling factor. Typically, \code{a = 0.5}.
  }

\item{b}{
  A numeric value representing the frequency factor. Typically, \code{b = 3}.
  }

\item{kmax}{
  An integer specifying the maximum number of terms in the summation. Typically, \code{kmax = 20}.
  }
}

\value{
Returns a numeric value representing the evaluation of the Weierstrass function at the given input vector \code{x}.
}

\description{
The Weierstrass function is a continuous but non-differentiable function commonly used for testing optimization algorithms. It is highly multimodal, with numerous local minima, making it challenging for optimization algorithms to locate the global minimum.
}

\examples{

# Evaluation 1: Global minimum point in a four-dimensional space
x <- rep(0, 4)
weierstrass(x)

# Evaluation 2: A point in a six-dimensional space
x <- c(0, 0.24, 11, -1, -0.7, pi)
weierstrass(x)

# Contour Plot: Visualizing the Weierstrass Function
x1 <- seq(-10, 10, length.out = 100)
x2 <- seq(-10, 10, length.out = 100)
z <- outer(x1, x2, FUN = Vectorize(function(x, y) weierstrass(c(x, y))))
contour(x1, x2, z, nlevels = 20, main = "Contour of the Weierstrass Function")

# EDA.mnorm() example
res = EDA.mnorm(fun = weierstrass, lower = c(-10,-10), upper = c(10,10), n = 30, 
                k = 2, tolerance = 0.01, maxiter = 200)
res$sol

# Contour plot: Visualizing solution with EDA.mnorm()
x1 <- seq(-10, 10, length.out = 100)
x2 <- seq(-10, 10, length.out = 100)
z <- outer(x1, x2, FUN = Vectorize(function(x, y) weierstrass(c(x, y))))
contour(x1, x2, z, nlevels = 20, cex.axis = 0.8, 
        main = "Contour plot of the Weierstrass Function with EDA.mnorm solution")
points(res$sol[1], res$sol[2], col = "red", pch = 19)
}

\references{
Weierstrass, K. (1872). On Continuous Non-Differentiable Functions. In \emph{Mathematische Werke}, 2(1), 71–74.
}
