% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Hull.R
\name{Hull}
\alias{Hull}
\title{the Hull Approach}
\usage{
Hull(
  response,
  fa = "pc",
  nfact = 1,
  cor.type = "pearson",
  use = "pairwise.complete.obs",
  vis = TRUE,
  plot = TRUE
)
}
\arguments{
\item{response}{A required \code{N} × \code{I} matrix or data.frame consisting of the responses of \code{N} individuals
to × \code{I} items.}

\item{fa}{A string that determines the method used to obtain eigenvalues in PA. If 'pc', it represents
Principal Component Analysis (PCA); if 'fa', it represents Principal Axis Factoring (a widely
used Factor Analysis method; @seealso \code{\link[EFAfactors]{factor.analysis}};
Auerswald & Moshagen, 2019). (Default = 'pc')}

\item{nfact}{A numeric value that specifies the number of factors to extract, only effective when \code{fa = 'fa'}. (Default = 1)}

\item{cor.type}{A character string indicating which correlation coefficient (or covariance) is to be computed. One of "pearson" (default),
"kendall", or "spearman". @seealso \link[stats]{cor}.}

\item{use}{an optional character string giving a method for computing covariances in the presence of missing values. This
must be one of the strings "everything", "all.obs", "complete.obs", "na.or.complete", or "pairwise.complete.obs" (default).
@seealso \link[stats]{cor}.}

\item{vis}{A Boolean variable that will print the factor retention results when set to TRUE, and will not print
when set to FALSE. (default = TRUE)}

\item{plot}{A Boolean variable that will print the CD plot when set to TRUE, and will not print it when set to
FALSE. @seealso \link[EFAfactors]{plot.Hull}. (Default = TRUE)}
}
\value{
A list with the following components:
\item{nfact}{The optimal number of factors according to the Hull method.}
\item{CFI}{A numeric vector of CFI values for each number of factors considered.}
\item{df}{A numeric vector of model degrees of freedom for each number of factors considered.}
\item{Hull.CFI}{A numeric vector of CFI values with points below the convex Hull curve removed.}
\item{Hull.df}{A numeric vector of model degrees of freedom with points below the convex Hull curve removed.}
}
\description{
The Hull method is a heuristic approach used to determine the optimal number of common factors
in factor analysis. It evaluates models with increasing numbers of factors and uses goodness-of-fit
indices relative to the model degrees of freedom to select the best-fitting model. The method is known
for its effectiveness and reliability compared to other methods like the scree plot.
}
\details{
The Hull method (Lorenzo-Seva & Timmerman, 2011) is a heuristic approach used to determ
ine the number of common factors in factor analysis. This method is similar to
non-graphical variants of Cattell's scree plot but relies on goodness-of-fit indices
relative to the model degrees of freedom. The Hull method finds the optimal number of
factors by following these steps:
}
\examples{
library(EFAfactors)
set.seed(123)

##Take the data.bfi dataset as an example.
data(data.bfi)

response <- as.matrix(data.bfi[, 1:25]) ## loading data
response <- na.omit(response) ## Remove samples with NA/missing values

## Transform the scores of reverse-scored items to normal scoring
response[, c(1, 9, 10, 11, 12, 22, 25)] <- 6 - response[, c(1, 9, 10, 11, 12, 22, 25)] + 1


## Run EKC function with default parameters.
\donttest{
 Hull.obj <- Hull(response)

 print(Hull.obj)

 plot(Hull.obj)

 ## Get the CFI, df and  nfact results.
 CFI <- Hull.obj$CFI
 df <- Hull.obj$df
 nfact <- Hull.obj$nfact

 print(CFI)
 print(df)
 print(nfact)

}


}
\references{
Auerswald, M., & Moshagen, M. (2019). How to determine the number of factors to retain in exploratory factor analysis: A comparison of extraction methods under realistic conditions. Psychological methods, 24(4), 468-491. https://doi.org/https://doi.org/10.1037/met0000200.

Lorenzo-Seva, U., Timmerman, M. E., & Kiers, H. A. L. (2011). The Hull Method for Selecting the Number of Common Factors. Multivariate Behavioral Research, 46(2), 340-364. https://doi.org/10.1080/00273171.2011.564527.
}
\author{
Haijiang Qin <Haijiang133@outlook.com>
}
