% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/clr_functions.R
\name{clr}
\alias{clr}
\title{Clr and inverse clr transformation}
\usage{
clr(f, w = 1, inverse = FALSE)
}
\arguments{
\item{f}{a vector containing the function values (evaluated on a grid) of the 
function \eqn{f} to transform. If \code{inverse = TRUE}, \code{f} must be a density,
i.e., all entries must be positive and usually \code{f} integrates to one. 
If \code{inverse = FALSE}, \code{f} should integrate to zero, see Details.}

\item{w}{a vector of length one or of the same length as \code{f} containing 
positive integration weights. If \code{w} has length one, this
weight is used for all function values. The integral of \eqn{f} is approximated
via \eqn{\int_{\mathcal{T}} f \, \mathrm{d}\mu \approx 
\sum_{j=1}^m}{\sum_{j=1}^m} \code{w}\eqn{_j} \code{f}\eqn{_j},
where \eqn{m} equals the length of \code{f}.}

\item{inverse}{if \code{TRUE}, the inverse clr transformation is computed.}
}
\value{
A vector of the same length as \code{f} containing the (inverse) clr 
transformation of \code{f}.
}
\description{
\code{clr} computes the clr or inverse clr transformation of a vector \code{f}
with respect to integration weights \code{w}, corresponding to a Bayes Hilbert space
\eqn{B^2(\mu) = B^2(\mathcal{T}, \mathcal{A}, \mu)}{B^2(\mu) = B^2(T, A, \mu)}.
}
\details{
The clr transformation maps a density \eqn{f} from \eqn{B^2(\mu)} to
\eqn{L^2_0(\mu) := \{ f \in L^2(\mu) ~|~ \int_{\mathcal{T}} f \, \mathrm{d}\mu = 0\}}{L^2_0(\mu) := {f \in L^2(\mu) | \int_T f d\mu = 0}}
via
\deqn{\mathrm{clr}(f) := \log f - \frac{1}{\mu (\mathcal{T})} \int_{\mathcal{T}} \log f \, \mathrm{d}\mu.}{clr(f) := log f - 1/\mu(T) * \int_T log f d\mu.}
The inverse clr transformation maps a function \eqn{f} from
\eqn{L^2_0(\mu)} to \eqn{B^2(\mu)} via
\deqn{\mathrm{clr}^{-1}(f) := \frac{\exp f}{\int_{\mathcal{T}} \exp f \, \mathrm{d}\mu}.}{clr^{-1}(f) := (exp f) / (\int_T \exp f d\mu).}
Note that in contrast to Maier et al. (2021), this definition of the inverse
clr transformation includes normalization, yielding the respective probability 
density function (representative of the equivalence class of proportional
functions in \eqn{B^2(\mu)}). 

The (inverse) clr transformation depends not only on \eqn{f}, but also on the
underlying measure space \eqn{\left( \mathcal{T}, \mathcal{A}, \mu\right)}{(T, A, \mu)}, 
which determines the integral. In \code{clr} this is specified via the 
integration weights \code{w}. E.g., for a discrete set \eqn{\mathcal{T}}{T}
with \eqn{\mathcal{A} = \mathcal{P}(\mathcal{T})}{A = P(T)} the power set of 
\eqn{\mathcal{T}}{T} and \eqn{\mu = \sum_{t \in T} \delta_t} the sum of dirac
measures at \eqn{t \in \mathcal{T}}{t \in T}, the default \code{w = 1} is
the correct choice. In this case, integrals are indeed computed exactly, not
only approximately. 
For an interval \eqn{\mathcal{T} = [a, b]}{T = [a, b]}
with \eqn{\mathcal{A} = \mathcal{B}}{A = B} the Borel \eqn{\sigma}-algebra 
restricted to \eqn{\mathcal{T}}{T} and \eqn{\mu = \lambda} the Lebesgue measure,
the choice of \code{w} depends on the grid on which the function was evaluated:
\code{w}\eqn{_j} must correspond to the length of the subinterval of \eqn{[a, b]}, which 
\code{f}\eqn{_j} represents.
E.g., for a grid with equidistant distance \eqn{d}, where the boundary grid 
values are \eqn{a + \frac{d}{2}}{a + d/2} and \eqn{b - \frac{d}{2}}{b - d/2}
(i.e., the grid points are centers of intervals of size \eqn{d}),
equal weights \eqn{d} should be chosen for \code{w}. 

The clr transformation is crucial for density-on-scalar regression 
since estimating the clr transformed model in \eqn{L^2_0(\mu)} is equivalent
to estimating the original model in \eqn{B^2(\mu)} (as the clr transformation
is an isometric isomorphism), see also the vignette "FDboost_density-on-scalar_births"
and Maier et al. (2021).
}
\examples{
### Continuous case (T = [0, 1] with Lebesgue measure):
# evaluate density of a Beta distribution on an equidistant grid
g <- seq(from = 0.005, to = 0.995, by = 0.01)
f <- dbeta(g, 2, 5)
# compute clr transformation with distance of two grid points as integration weight
f_clr <- clr(f, w = 0.01)
# visualize result
plot(g, f_clr , type = "l")
abline(h = 0, col = "grey")
# compute inverse clr transformation (w as above)
f_clr_inv <- clr(f_clr, w = 0.01, inverse = TRUE)
# visualize result
plot(g, f, type = "l")
lines(g, f_clr_inv, lty = 2, col = "red")

### Discrete case (T = {1, ..., 12} with sum of dirac measures at t in T):
data("birthDistribution", package = "FDboost")
# fit density-on-scalar model with effects for sex and year
model <- FDboost(birth_densities_clr ~ 1 + bolsc(sex, df = 1) + 
                   bbsc(year, df = 1, differences = 1),
                 # use bbsc() in timeformula to ensure integrate-to-zero constraint
                 timeformula = ~bbsc(month, df = 4, 
                                     # December is followed by January of subsequent year
                                     cyclic = TRUE, 
                                     # knots = {1, ..., 12} with additional boundary knot
                                     # 0 (coinciding with 12) due to cyclic = TRUE
                                     knots = 1:11, boundary.knots = c(0, 12), 
                                     # degree = 1 with these knots yields identity matrix 
                                     # as design matrix
                                     degree = 1),
                 data = birthDistribution, offset = 0, 
                 control = boost_control(mstop = 1000))
# Extract predictions (clr-transformed!) and transform them to Bayes Hilbert space
predictions_clr <- predict(model)
predictions <- t(apply(predictions_clr, 1, clr, inverse = TRUE))

}
\references{
Maier, E.-M., Stoecker, A., Fitzenberger, B., Greven, S. (2021):
Additive Density-on-Scalar Regression in Bayes Hilbert Spaces with an Application to Gender Economics.
arXiv preprint arXiv:2110.11771.
}
\author{
Eva-Maria Maier
}
