% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/FRK_wrapper.R, R/SREutils.R
\docType{methods}
\name{FRK}
\alias{FRK}
\alias{SRE}
\alias{SRE.fit}
\alias{SRE.predict}
\alias{predict,SRE-method}
\alias{loglik}
\title{Construct SRE object, fit and predict}
\usage{
FRK(f, data, basis = NULL, BAUs = NULL, est_error = TRUE,
  average_in_BAU = TRUE, fs_model = "ind", vgm_model = NULL,
  K_type = "block-exponential", n_EM = 100, tol = 0.01, method = "EM",
  lambda = 0, print_lik = FALSE, ...)

SRE(f, data, basis, BAUs, est_error = TRUE, average_in_BAU = TRUE,
  fs_model = "ind", vgm_model = NULL, K_type = "block-exponential",
  normalise_basis = TRUE)

SRE.fit(SRE_model, n_EM = 100L, tol = 0.01, method = "EM", lambda = 0,
  print_lik = FALSE)

SRE.predict(SRE_model, obs_fs = FALSE, newdata = NULL, pred_polys = NULL,
  pred_time = NULL, covariances = FALSE)

\S4method{predict}{SRE}(object, newdata = NULL, obs_fs = FALSE,
  pred_polys = NULL, pred_time = NULL, covariances = FALSE)

loglik(SRE_model)
}
\arguments{
\item{f}{\code{R} formula relating the dependent variable (or transformations thereof) to covariates}

\item{data}{list of objects of class \code{SpatialPointsDataFrame}, \code{SpatialPolygonsDataFrame}, \code{STIDF}, or  \code{STFDF}. If using space-time objects, the data frame must have another field, \code{t}, containing the time index of the data point}

\item{basis}{object of class \code{Basis} (or \code{TensorP_Basis})}

\item{BAUs}{object of class \code{SpatialPolygonsDataFrame}, \code{SpatialPixelsDataFrame}, \code{STIDF}, or \code{STFDF}. The object's data frame must contain covariate information as well as a field \code{fs} describing the fine-scale variation up to a constant of proportionality. If the function \code{FRK} is used directly, then BAUs are created automatically, but only coordinates can then be used as covariates}

\item{est_error}{flag indicating whether the measurement-error variance should be estimated from variogram techniques. If this is set to 0, then \code{data} must contain a field \code{std}. Measurement-error estimation is currently not implemented for spatio-temporal datasets}

\item{average_in_BAU}{if \code{TRUE}, then multiple data points falling in the same BAU are averaged; the measurement error of the averaged data point is taken as the average of the individual measurement errors}

\item{fs_model}{if "ind" then the fine-scale variation is independent at the BAU level. If "ICAR", then an ICAR model for the fine-scale variation is placed on the BAUs}

\item{vgm_model}{an object of class \code{variogramModel} from the package \code{gstat} constructed using the function \code{vgm}. This object contains the variogram model that will be fit to the data. The nugget is taken as the measurement error when \code{est_error = TRUE}. If unspecified, the variogram used is \code{gstat::vgm(1, "Lin", d, 1)}, where \code{d} is approximately one third of the maximum distance between any two data points}

\item{K_type}{the parameterisation used for the \code{K} matrix. Currently this can be "unstructured" or "block-exponential" (default)}

\item{n_EM}{maximum number of iterations for the EM algorithm}

\item{tol}{convergence tolerance for the EM algorithm}

\item{method}{parameter estimation method to employ. Currently only ``EM'' is supported}

\item{lambda}{ridge-regression regularisation parameter for when \code{K} is unstructured (0 by default). Can be a single number, or a vector (one parameter for each resolution)}

\item{print_lik}{flag indicating whether likelihood value should be printed or not after convergence of the EM estimation algorithm}

\item{...}{other parameters passed on to \code{auto_basis} and \code{auto_BAUs} when calling the function \code{FRK}}

\item{normalise_basis}{flag indicating whether to normalise the basis functions so that they reproduce a stochastic process with approximately constant variance spatially}

\item{SRE_model}{object returned from the constructor \code{SRE()} containing all the parameters and information on the SRE model}

\item{obs_fs}{flag indicating whether the fine-scale variation sits in the observation model (systematic error, Case 1) or in the process model (fine-scale process variation, Case 2, default)}

\item{newdata}{object of class \code{SpatialPoylgons} indicating the regions over which prediction will be carried out. The BAUs are used if this option is not specified}

\item{pred_polys}{deprecated. Please use \code{newdata} instead}

\item{pred_time}{vector of time indices at which prediction will be carried out. All time points are used if this option is not specified}

\item{covariances}{logical variable indicating whether prediction covariances should be returned or not. If set to \code{TRUE}, a maximum of 4000 prediction locations or polygons are allowed.}

\item{object}{object of class \code{SRE}}
}
\description{
The Spatial Random Effects (SRE) model is the central object in FRK. The function \code{FRK} provides a wrapper for the construction and estimation of the SRE object from data, using the functions \code{SRE} (the object constructor) and \code{SRE.fit} (for fitting it to the data). Please see \code{\link{SRE-class}} for more details on the SRE object's properties and methods.
}
\details{
\code{SRE()} is the main function in the package: It constructs a spatial random effects model from the user-defined formula, data object, basis functions and a set of Basic Areal Units (BAUs). The function first takes each object in the list \code{data} and maps it to the BAUs -- this entails binning the point-referenced data into the BAUs (and averaging within the BAU) if \code{average_in_BAU = TRUE}, and finding which BAUs are influenced by the polygon datasets. Following this, the incidence matrix \code{Cmat} is constructed, which appears in the observation model \eqn{Z = CY + C\delta + e}, where \eqn{C} is the incidence matrix and \eqn{\delta} is systematic error at the BAU level.

The SRE model for the hidden process is given by \eqn{Y = T\alpha + S\eta + \xi}, where \eqn{T} are the covariates at the BAU level, \eqn{\alpha} are the regression coefficients, \eqn{S} are the basis functions evaluated at the BAU level, \eqn{\eta} are the basis-function coefficients, and \eqn{\xi} is the fine scale variation (at the BAU level). The covariance matrix of \eqn{\xi} is diagonal, with its diagonal elements proportional to the field `fs' in the BAUs (typically set to one). The constant of proportionality is estimated in the EM algorithm. All required matrices (\eqn{S,T} etc.) are initialised using sensible defaults and returned as part of the object, please see \code{\link{SRE-class}} for more details.

\code{SRE.fit()} takes an object of class \code{SRE} and estimates all unknown parameters, namely the covariance matrix \eqn{K}, the fine scale variance (\eqn{\sigma^2_{\xi}} or \eqn{\sigma^2_{\delta}}, depending on whether Case 1 or Case 2 is chosen; see the vignette) and the regression parameters \eqn{\alpha}. The only method currently implemented is the Expectation Maximisation (EM) algorithm, which the user configures through \code{n_EM} and \code{tol}. The log-likelihood (given in Section 2.2 of the vignette) is evaluated at each iteration at the current parameter estimate, and convergence is assumed to have been reached when this quantity stops changing by more than \code{tol}.

The actual computations for the E-step and M-step are relatively straightforward. The E-step contains an inverse of an \eqn{r \times r} matrix, where \code{r} is the number of basis functions which should not exceed 2000. The M-step first updates the matrix \eqn{K}, which only depends on the sufficient statistics of the basis-function coefficients \eqn{\eta}. Then, the regression parameter \eqn{\alpha} is updated and a simple optimisation routine (a line search) is used to update the fine-scale variance \eqn{\sigma^2_{\delta}} or \eqn{\sigma^2_{\xi}}. If the fine-scale errors and measurement random errors are homoscedastic, then a closed-form solution is available for the update of \eqn{\sigma^2_{\xi}} or \eqn{\sigma^2_{\delta}}. Irrespectively, since the udpates of \eqn{\alpha}, and \eqn{\sigma^2_{\delta}} or \eqn{\sigma^2_{\xi}}, are dependent, these two updates are iterated until the change in \eqn{\sigma^2_{\cdot}} is no more than 0.1\%. Information on the fitting (convergence etc.) can be extracted using \code{info_fit(SRE_model)}.

The function \code{FRK} acts as a wrapper for the functions \code{SRE} and \code{SRE.fit}. An added advantage of using \code{FRK} directly is that it automatically generates BAUs and basis functions based on the data. Hence \code{FRK} can be called using only a list of data objects and an \code{R} formula, although the \code{R} formula can only contain space or time as covariates when BAUs are not explicitly supplied with the covariate data.

Once the parameters are fitted, the \code{SRE} object is passed onto the function \code{predict()} in order to carry out optimal predictions over the same BAUs used to construct the SRE model with \code{SRE()}. The first part of the prediction process is to construct the matrix \eqn{S} over the prediction polygons. This is made computationally efficient by treating the prediction over polygons as that of the prediction over a combination of BAUs. This will yield valid results only if the BAUs are relatively small. Once the matrix \eqn{S} is found, a standard Gaussian inversion (through conditioning) using the estimated parameters is used for prediction.

\code{predict} returns the BAUs, which are of class \code{SpatialPolygonsDataFrame}, \code{SpatialPixelsDataFrame}, or \code{STFDF}, with two added attributes, \code{mu} and \code{var}. These can then be easily plotted using \code{spplot} or \code{ggplot2} (possibly in conjunction with \code{\link{SpatialPolygonsDataFrame_to_df}}) as shown in the package vignettes.
}
\examples{
library(sp)

### Generate process and data
n <- 100
sim_process <- data.frame(x = seq(0.005,0.995,length=n))
sim_process$y <- 0
sim_process$proc <- sin(sim_process$x*10) + 0.3*rnorm(n)

sim_data <- sim_process[sample(1:n,50),]
sim_data$z <- sim_data$proc + 0.1*rnorm(50)
sim_data$std <- 0.1
coordinates(sim_data) = ~x + y # change into an sp object
grid_BAUs <- auto_BAUs(manifold=real_line(),data=sim_data,
                       nonconvex_hull=FALSE,cellsize = c(0.01),type="grid")
grid_BAUs$fs = 1

### Set up SRE model
G <- auto_basis(manifold = real_line(),
                data=sim_data,
                nres = 2,
                regular = 6,
                type = "bisquare",
                subsamp = 20000)
f <- z ~ 1
S <- SRE(f,list(sim_data),G,
         grid_BAUs,
         est_error = FALSE)

### Fit with 5 EM iterations so as not to take too much time
S <- SRE.fit(S,n_EM = 5,tol = 0.01,print_lik=TRUE)

### Check fit info


### Predict over BAUs
grid_BAUs <- predict(S)

### Plot
\dontrun{
library(ggplot2)
X <- slot(grid_BAUs,"data")
X <- subset(X, x >= 0 & x <= 1)
 g1 <- LinePlotTheme() +
    geom_line(data=X,aes(x,y=mu)) +
    geom_errorbar(data=X,aes(x=x,ymax = mu + 2*sqrt(var), ymin= mu - 2*sqrt(var))) +
    geom_point(data = data.frame(sim_data),aes(x=x,y=z),size=3) +
    geom_line(data=sim_process,aes(x=x,y=proc),col="red")
 print(g1)}
}
\seealso{
\code{\link{SRE-class}} for details on the SRE object internals, \code{\link{auto_basis}} for automatically constructing basis functions, and \code{\link{auto_BAUs}} for automatically constructing BAUs. See also the paper \url{https://arxiv.org/abs/1705.08105} for details on code operation.
}
\keyword{spatial}
