#' Modify a message in the chat history of a Generative AI model and generate a new response based on a prompt
#'
#' This function establishes a connection to a Generative AI model by providing essential
#' parameters. It then edits a specific message in the chat history, generates a new response based on the
#' prompt, and updates the chat history list.
#'
#' @param model.parameter A character vector containing information about the Generative AI service provider,
#' corresponding model, version, API key, and proxy status.
#' @param temperature A numeric value. Higher values result in more creative responses,
#' while lower values produce more straightforward text.
#' @param history A list containing all messages exchanged between the user and model in a chat session.
#' @param message.to.edit An integer representing the index of the message to be edited.
#' @param prompt A character string representing the query for text generation.
#'
#' @return If successful, the full chat history, including the most recent response, will be returned. If
#' the API response indicates an error, the function halts execution and provides an error message.
#'
#' @details Providing accurate and valid information for each parameter is crucial for successful
#' text generation by the Generative AI model. This function modifies the chat history. If any
#' parameter is incorrect, the function responds with an error message based on the API feedback.
#' To view all supported Generative AI models, use the function \code{\link{available.models}}.
#' Additionally, utilize \code{\link{chat.setup}} to set up a chat history. To retrieve the most recent
#' response generated by the model, use the function \code{\link{chat.recent}}.
#'
#' @seealso
#' \href{https://genai.gd.edu.kg/r/documentation/}{GenAI - R Documentation}
#'
#' @examples
#' \dontrun{
#'  # Get available models
#'  models = available.models()
#'
#'  # Connect to the model, replace API_KEY with your api key
#'  google.model = connect.genai("google",
#'                               models$google$model[1],
#'                               models$google$version[1],
#'                               "API_KEY",
#'                               FALSE)
#'
#'  # Connect to the model, replace API_KEY with your api key
#'  openai.model = connect.genai("openai",
#'                               models$openai$model[1],
#'                               models$openai$version[1],
#'                               "API_KEY",
#'                               FALSE)
#'  # Setup an empty chat history
#'  google.history = chat.setup(google.model)
#'  openai.history = chat.setup(openai.model)
#'
#'  # Start chat
#'  temperature = 0.9
#'  prompt = "Write a story about a magic backpack in about 100 words."
#'  google.history = chat(google.model,
#'                        temperature,
#'                        google.history,
#'                        prompt)
#'
#'  # Get the most recent message generated by the model
#'  cat(chat.recent(google.model, google.history))
#'
#'  openai.history = chat(openai.model,
#'                        temperature,
#'                        openai.history,
#'                        prompt)
#'
#'  # Get the most recent message generated by the model
#'  cat(chat.recent(openai.model, openai.history))
#'
#'  prompt = "What is the word count of the story you just wrote?"
#'  google.history = chat(google.model,
#'                        temperature,
#'                        google.history,
#'                        prompt)
#'
#'  # Get the most recent message generated by the model
#'  cat(chat.recent(google.model, google.history))
#'
#'  openai.history = chat(openai.model,
#'                        temperature,
#'                        openai.history,
#'                        prompt)
#'
#'  # Get the most recent message generated by the model
#'  cat(chat.recent(openai.model, openai.history))
#'
#'  # Edit chat
#'  prompt = "Rewrite the story with a word limit of 50 words."
#'  google.history = chat.edit(google.model,
#'                             temperature,
#'                             google.history,
#'                             3,
#'                             prompt)
#'  # Get the most recent message generated by the model
#'  cat(chat.recent(google.model, google.history))
#'
#'  openai.history = chat.edit(openai.model,
#'                             temperature,
#'                             openai.history,
#'                             4,
#'                             prompt)
#'  # Get the most recent message generated by the model
#'  cat(chat.recent(openai.model, openai.history))
#' }
#'
#' @export
#'
#' @importFrom jsonlite toJSON
#' @importFrom httr POST add_headers content
chat.edit = function(model.parameter,
                     temperature,
                     history,
                     message.to.edit,
                     prompt) {
  if (prompt == "" ||
      is.na(prompt) || !inherits(prompt, "character")) {
    stop("Prompt is not in correct format.")
  }
  switch (model.parameter["provider"],
          google = {
            if (message.to.edit == "" || is.na(message.to.edit) ||
                !inherits(message.to.edit, "numeric") ||
                message.to.edit %% 2 == 0 || message.to.edit < 1) {
              stop("Input: message.to.edit must be a non-empty even numeric value.")
            }
            api.URL = ifelse(
              model.parameter["proxy"],
              paste0(
                "https://api.genai.gd.edu.kg/google/",
                model.parameter["version"],
                "/models/",
                model.parameter["model"],
                ":generateContent?key=",
                model.parameter["api"]
              ),
              paste0(
                "https://generativelanguage.googleapis.com/",
                model.parameter["version"],
                "/models/",
                model.parameter["model"],
                ":generateContent?key=",
                model.parameter["api"]
              )
            )
            requestNewContent = list(list(role = "user",
                                          parts = list(text = prompt)))
            config = list(generationConfig = list(temperature = temperature))
            requestBody = history
            requestBody$contents = append(history$contents[1:message.to.edit - 1], requestNewContent)
            requestBody = append(requestBody, config)
            requestBodyJSON = jsonlite::toJSON(requestBody, auto_unbox = TRUE)

            response = httr::POST(
              url = api.URL,
              body = requestBodyJSON,
              httr::add_headers("Content-Type" = "application/json")
            )
            responseJSON = httr::content(response, "parsed")
            if (!is.null(responseJSON$error)) {
              stop(responseJSON$error$message)
            }
            if (!is.null(responseJSON$blockReason)) {
              stop("The prompt may contain harmful content.")
            }
            history$contents = append(history$contents[1:message.to.edit - 1], requestNewContent)
            respondContent = list(list(
              role = "model",
              parts = list(text = responseJSON$candidates[[1]]$content$parts[[1]]$text)
            ))
            history$contents = append(history$contents, respondContent)
            return (history)
          },
          openai = {
            if (message.to.edit == "" || is.na(message.to.edit) ||
                !inherits(message.to.edit, "numeric") ||
                message.to.edit %% 2 == 1 || message.to.edit < 2) {
              stop("Input: message.to.edit must be a non-empty even numeric value.")
            }
            moderation.openai(model.parameter, prompt)
            api.URL = ifelse(
              model.parameter["proxy"],
              paste0(
                "https://api.genai.gd.edu.kg/openai/",
                model.parameter["version"],
                "/chat/completions"
              ),
              paste0(
                "https://api.openai.com/",
                model.parameter["version"],
                "/chat/completions"
              )
            )
            requestNewContent = list(list(role = "user", content = prompt))
            requestBody = history
            requestBody$messages = append(history$messages[1:message.to.edit - 1], requestNewContent)
            requestBody = append(requestBody, list(temperature = temperature))
            requestBodyJSON = jsonlite::toJSON(requestBody, auto_unbox = TRUE)
            response = httr::POST(
              url = api.URL,
              body = requestBodyJSON,
              httr::add_headers(
                "Content-Type" = "application/json",
                "Authorization" = paste("Bearer", model.parameter["api"])
              )
            )
            responseJSON = httr::content(response, "parsed")
            if (!is.null(responseJSON$error)) {
              stop(responseJSON$error$message)
            }
            history$messages = append(history$messages[1:message.to.edit - 1], requestNewContent)
            respondContent = list(list(
              role = "assistant",
              content = responseJSON$choices[[1]]$message$content
            ))
            history$messages = append(history$messages, respondContent)
            return (history)
          })
}
