## ----setup, include=FALSE-----------------------------------------------------
library(shiny)
library(knitr)

knitr::opts_chunk$set(
    fig.align = "center" # Center images in the export file
  , out.width = "98%" # Figure width in html
  , echo = FALSE # Avoid print code in the export file
  , message = FALSE # Avoid print messages in the export file
  , warning = FALSE # Avoid print messages in the export file
  , collapse = TRUE
  , comment = "#>"
  )

options(
    OutDec= "." # Use "." instead of "," in the decimal values
  , scipen = 99 # Avoid use "6e-04"
  , knitr.kable.NA = "" # NA values will appear as empty cell
  , knitr.table.format = "pipe" # Format for export tables
  , citr.use_betterbiblatex = FALSE # For zotero addin 
  ) 

## ---- echo=TRUE---------------------------------------------------------------
# Install packages and dependencies

library(GerminaR)
library(dplyr)

# load data

fb <- prosopis %>% 
   dplyr::mutate(across(c(nacl, temp, rep), as.factor))

# Prosopis data set

fb %>% 
   head(10) %>% 
   kable(caption = "Prosopis dataset loaded")

## ---- echo=TRUE---------------------------------------------------------------

# germination analysis (ten variables)

gsm <- ger_summary(SeedN = "seeds"
                   , evalName = "D"
                   , data = fb
                   )

# Prosopis data set processed

gsm %>% 
   head(10) %>% 
   kable(caption = "Function ger_summary performe ten germination indices")


## ---- echo=TRUE, fig.cap="Germination  experiment with *Prosopis juliflor* under different osmotic potentials and temperatures. Bar graph with germination percentage in a factorial analisys"----

## Germination Percentage (GRP)

# analysis of variance

av <- aov(formula = grp ~ nacl*temp + rep, data = gsm)

# mean comparison test

mc_grp <- ger_testcomp(aov = av
                       , comp = c("temp", "nacl")
                       , type = "snk"
                       )

# data result

mc_grp$table %>% 
   kable(caption = "Germination percentage mean comparision")

# bar graphics for germination percentage

grp <- mc_grp$table %>% 
   fplot(data = .
       , type = "bar"
       , x = "temp"
       , y = "grp"
       , groups = "nacl"
       , limits = c(0,150)
       , brakes = 30
       , ylab = "Germination ('%')"
       , xlab = "Temperature (ºC)"
       , glab = "NaCl (MPa)"
       , legend = "top"
       , error = "ste"
       , sig = "sig"
       , color = F
       )

grp


## ---- echo=TRUE, fig.cap="Germination  experiment with *Prosopis juliflor* under different osmotic potentials and temperatures. Bar graph for mean germination time in a factorial analisys."----

## Mean Germination Time (MGT)

# analysis of variance

av <- aov(formula = mgt ~ nacl*temp + rep, data = gsm)

# mean comparison test

mc_mgt <- ger_testcomp(aov = av
                       , comp = c("temp", "nacl")
                       , type = "snk")

# data result

mc_mgt$table %>% 
   kable(caption = "Mean germination time comparison")

# bar graphics for mean germination time

mgt <- mc_mgt$table %>% 
   fplot(data = .
       , type = "bar" 
       , x = "temp"
       , y = "mgt"
       , groups = "nacl"
       , limits = c(0,9)
       , brakes = 1
       , ylab = "Mean germination time (days)"
       , xlab = "Temperature (ºC)"
       , glab = "NaCl (MPa)"
       , legend = "top"
       , sig = "sig"
       , error = "ste"
       , color = T
       )

mgt

## ---- echo=TRUE, fig.cap="Germination  experiment with *Prosopis juliflor* under different osmotic potentials and temperatures. Line graph from cumulative germination under different osmotic potentials."----

# data frame with percentage or relative germination in time by NaCl

git <- ger_intime(Factor = "nacl"
                  , SeedN = "seeds"
                  , evalName = "D"
                  , method = "percentage"
                  , data = fb
                  )

# data result

git %>% 
   head(10) %>% 
   kable(caption = "Cumulative germination by nacl factor")

# graphic germination in time by NaCl

nacl <- git %>% 
   fplot(data = .
        , type = "line"
        , x = "evaluation"
        , y = "mean"
        , groups = "nacl"
        , limits = c(0, 110)
        , brakes = 10
        , ylab = "Germination ('%')"
        , xlab = "Day"
        , glab = "NaCl (MPa)"
        , legend = "top"
        , color = T
        , error = "ste"
        )
nacl


## ---- echo=TRUE, fig.cap="Germination  experiment with *Prosopis juliflor* under different osmotic potentials and temperatures. Line graph from cumulative germination under different temperatures."----

# data frame with percentage or relative germination in time by temperature

git <- ger_intime(Factor = "temp"
                  , SeedN = "seeds"
                  , evalName = "D"
                  , method = "percentage"
                  , data = fb) 

# data result

git %>% 
   head(10) %>% 
   kable(caption = "Cumulative germination by temperature factor")

# graphic germination in time by temperature

temp <- git %>% 
   fplot(data = .
        , type = "line"
        , x = "evaluation"
        , y = "mean"
        , groups = "temp"
        , limits = c(0, 110)
        , brakes = 10
        , ylab = "Germination ('%')"
        , xlab = "Day"
        , glab = "Temperature ('°C')"
        , legend = "top"
        , color = F
        ) 
temp


## ---- echo=TRUE---------------------------------------------------------------

library(ggplot2)

git <- ger_intime(Factor = "temp"
                  , SeedN = "seeds"
                  , evalName = "D"
                  , method = "percentage"
                  , data = fb
                  ) 

ggplot <- git %>% 
   fplot(data = .
        , type = "line"
        , x = "evaluation"
        , y = "mean"
        , groups = "temp"
        , limits = c(0, 110)
        , brakes = 10
        , ylab = "Germination ('%')"
        , xlab = "Day"
        , glab = "Temperature ('°C')"
        , legend = "top"
        , color = F
        ) +
   scale_x_continuous(n.breaks = 10
                      , limits = c(0, 11)
                      ) 

ggplot


