% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MHalgoGen.R
\name{MHalgoGen}
\alias{MHalgoGen}
\title{Monte-Carlo Markov-chain with Metropolis-Hastings algorithm}
\usage{
MHalgoGen(likelihood = stop("A likelihood function must be supplied"),
  parameters = stop("Priors  must be supplied"), ..., n.iter = 10000,
  n.chains = 1, n.adapt = 100, thin = 30, trace = FALSE,
  intermediate = NULL, filename = "intermediate.Rdata", previous = NULL)
}
\arguments{
\item{likelihood}{The function that returns -ln likelihood using data and parameters}

\item{parameters}{A data.frame with priors; see description and examples}

\item{...}{Parameters to be transmitted to likelihood function}

\item{n.iter}{Number of iterations for each chain}

\item{n.chains}{Number of chains}

\item{n.adapt}{Number of iteration to stabilize likelihood}

\item{thin}{Interval for thinning likelihoods}

\item{trace}{Or FALSE or period to show progress}

\item{intermediate}{Or NULL of period to save intermediate result}

\item{filename}{Name of file in which intermediate results are saved}

\item{previous}{The content of the file in which intermediate results are saved}
}
\value{
A mcmcComposite object with all characteristics of the model and mcmc run
}
\description{
The parameters must be stored in a data.frame with named rows for each parameter with the following columns:\cr
\itemize{
  \item Density. The density function name, example \code{dnorm}, \code{dlnorm}, \code{dunif}
  \item Prior1. The first parameter to send to the \code{Density} function
  \item Prior2. The second parameter to send to the \code{Density} function
  \item SDProp. The standard error from new proposition value of this parameter
  \item Min. The minimum value for this parameter
  \item Max. The maximum value for this parameter
  \item Init. The initial value for this parameter
}
This script has been deeply modified from a MCMC script provided by Olivier Martin (INRA, Paris-Grignon).\cr
The likelihood function must take x parameters.
}
\details{
MHalgoGen is a function to use mcmc with Metropolis-Hastings algorithm
}
\examples{
\dontrun{
library(HelpersMG)
require(coda)
val <- rnorm(30, 10, 2)
dnormx <- function(data, x) {
 data <- unlist(data)
 return(-sum(dnorm(data, mean=x['mean'], sd=x['sd'], log=TRUE)))
}
parameters_mcmc <- data.frame(Density=c('dnorm', 'dlnorm'), 
Prior1=c(10, 0.5), Prior2=c(2, 0.5), SDProp=c(0.35, 0.2), 
Min=c(-3, 0), Max=c(100, 10), Init=c(10, 2), stringsAsFactors = FALSE, 
row.names=c('mean', 'sd'))
mcmc_run <- MHalgoGen(n.iter=50000, parameters=parameters_mcmc, data=val, 
likelihood=dnormx, n.chains=1, n.adapt=100, thin=1, trace=1)
plot(mcmc_run, xlim=c(0, 20))
plot(mcmc_run, xlim=c(0, 10), parameters="sd")
library(graphics)
library(fields)
# show a scatter plot of the result
x <- mcmc_run$resultMCMC[[1]][, 1]
y <- mcmc_run$resultMCMC[[1]][, 2]
marpre <- par(mar=c(4, 4, 2, 6)+0.4)
smoothScatter(x, y)
# show a scale
n <- matrix(0, ncol=128, nrow=128)
xrange <- range(x)
yrange <- range(y)
for (i in 1:length(x)) {
  posx <- 1+floor(127*(x[i]-xrange[1])/(xrange[2]-xrange[1]))
  posy <- 1+floor(127*(y[i]-yrange[1])/(yrange[2]-yrange[1]))
  n[posx, posy] <- n[posx, posy]+1
}
image.plot(legend.only=TRUE, zlim= c(0, max(n)), nlevel=128, 
 col=colorRampPalette(c("white", blues9))(128))
# Compare with a heatmap
x <- seq(from=8, to=12, by=0.2)
y <- seq(from=1, to=4, by=0.2)
df <- expand.grid(mean=x, sd=y)
df <- cbind(df, L=rep(0, length(nrow(df))))
for (i in 1:nrow(df)) df[i, "L"] <- -sum(dnorm(val, df[i, 1], df[i, 2], log = TRUE))
hm <- matrix(df[, "L"], nrow=length(x))
par(mar = marpre)
image.plot(x=x, y=y, z=hm, las=1)
# Diagnostic function from coda library
mcmcforcoda <- as.mcmc(mcmc_run)
#' heidel.diag(mcmcforcoda)
raftery.diag(mcmcforcoda)
autocorr.diag(mcmcforcoda)
acf(mcmcforcoda[[1]][,"mean"], lag.max=20, bty="n", las=1)
acf(mcmcforcoda[[1]][,"sd"], lag.max=20, bty="n", las=1)
batchSE(mcmcforcoda, batchSize=100)
# The batch standard error procedure is usually thought to 
# be not as accurate as the time series methods used in summary
summary(mcmcforcoda)$statistics[,"Time-series SE"]
summary(mcmc_run)
as.parameters(mcmc_run)
lastp <- as.parameters(mcmc_run, index="last")
parameters_mcmc[,"Init"] <- lastp
# The n.adapt set to 1 is used to not record the first set of parameters
# then it is not duplicated (as it is also the last one for 
# the object mcmc_run)
mcmc_run2 <- MHalgoGen(n.iter=1000, parameters=parameters_mcmc, x=x, 
likelihood=dnormx, n.chains=1, n.adapt=1, thin=1, trace=1)
mcmc_run3 <- merge(mcmc_run, mcmc_run2)
####### no adaptation, n.adapt must be 0
parameters_mcmc[,"Init"] <- c(mean(x), sd(x))
mcmc_run3 <- MHalgoGen(n.iter=1000, parameters=parameters_mcmc, x=x, 
likelihood=dnormx, n.chains=1, n.adapt=0, thin=1, trace=1)
}
}
\author{
Marc Girondot
}
\seealso{
Other mcmcComposite functions: \code{\link{as.mcmc.mcmcComposite}},
  \code{\link{as.parameters}},
  \code{\link{merge.mcmcComposite}},
  \code{\link{plot.mcmcComposite}},
  \code{\link{summary.mcmcComposite}}
}

