% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/3-UserMinimaxFunctions.R
\name{sensminimax}
\alias{sensminimax}
\title{Verifying Optimality of Minimax and Standardized maximin D-optimal Designs}
\usage{
sensminimax(formula, predvars, parvars, family = gaussian(), x, w, lx, ux, lp,
  up, fimfunc = NULL, standardized = FALSE, localdes = NULL,
  sens.minimax.control = list(), calculate_criterion = TRUE,
  crt.minimax.control = list(), plot_3d = c("lattice", "rgl"),
  plot_sens = TRUE, npar = length(lp), silent = FALSE)
}
\arguments{
\item{formula}{A nonlinear model \code{\link[stats]{formula}}.
A symbolic description of the model consists of predictors and the unknown model parameters.
Will be coerced to a \code{\link[stats]{formula}} if necessary.}

\item{predvars}{A vector of characters. Denotes the predictors in the \code{\link[stats]{formula}}.}

\item{parvars}{A vector of characters. Denotes the unknown parameters in the \code{\link[stats]{formula}}.}

\item{family}{A description of the response distribution and link function to be used in the model.
This can be a family function, a call to a family function or a character string naming the family.
 Every family function has a link argument allowing to specify the link function to be applied on the response variable.
  If not specified, default links are used. For details see \code{\link[stats]{family}}.
   By default, a linear gaussian model \code{gaussian()} is applied.}

\item{x}{Vector of design (support) points. See 'Details' of \code{\link{sensminimax}}.}

\item{w}{Vector of corresponding design weights for \code{x}.}

\item{lx}{Vector of lower bounds for the predictors. Should be in the same order as \code{predvars}.}

\item{ux}{Vector of upper bounds for the predictors. Should be in the same order as \code{predvars}.}

\item{lp}{Vector of lower bounds for the model parameters. Should be in the same order as \code{parvars} or \code{param} in the argument \code{fimfunc}.}

\item{up}{Vector of upper bounds for the model parameters. Should be in the same order as \code{parvars} or \code{param} in the argument \code{fimfunc}.
When a parameter is known (has a fixed value), set its associated lower and upper bounds  in \code{lp} and \code{up}  equal to the known value.}

\item{fimfunc}{A function. Returns the FIM as a \code{matrix}. Required when \code{formula} is missing. See 'Details' of \code{\link{minimax}}.}

\item{standardized}{Maximin standardized design? When \code{standardized = TRUE}, the argument \code{localdes} must be given.
Defaults to \code{FALSE}. See 'Details' of \code{\link{minimax}}.}

\item{localdes}{A function that takes the parameter values  as inputs and returns the design points and weights of the locally optimal design.
Required when \code{standardized = "TRUE"}. See 'Details' of \code{\link{minimax}}.}

\item{sens.minimax.control}{Control parameters to verify the general equivalence theorem. For details, see the function \code{\link{sens.minimax.control}}.}

\item{calculate_criterion}{Evaluate the D-criterion? See 'Details' of \code{\link{sensminimax}}.}

\item{crt.minimax.control}{Control parameters to optimize the minimax or standardized maximin criterion at a given design over a \strong{continuous} parameter space.
For details, see the function \code{\link{crt.minimax.control}}. Only applicable when \code{calculate_criterion = TRUE}.}

\item{plot_3d}{Which package should be used to plot the sensitivity (derivative) function for models with two predictors.
Either \code{"rgl"} or \code{"lattice"} (default).}

\item{plot_sens}{Plot the sensitivity (derivative) function? Defaults to \code{TRUE}.}

\item{npar}{Number of model parameters.  Used when \code{fimfunc} is given instead of \code{formula} to specify the number of model parameters.
If not specified truly, the sensitivity (derivative) plot may be shifted below the y-axis. When \code{NULL}, it will be set here to \code{length(lp)}.}

\item{silent}{Do not print anything? Defaults to \code{FALSE}.}
}
\value{
an object of class \code{sensminimax} that is a list contains:
 \describe{
 \item{\code{type}}{Argument \code{type} that is required for print methods.}
 \item{\code{optima}}{A  \code{matrix} that stores all the local optima over the parameter space given the best design of the iteration. The cost  (criterion) values are stored in a column named \code{Criterion_Value}.
 The last column (\code{Answering_Set}) shows if the optimum, based on a merging tolerance \code{merge_tol}, belongs to the answering set (1) or not (0).
   Only applicable for minimax or standardized maximin designs.}
 \item{\code{mu}}{Probability measure on the answering set. Corresponds to the rows of \code{optima} when the associated row in column \code{Answering_Set} is equal to 1.
   Only applicable for minimax or standardized maximin designs.}
 \item{\code{max_deriv}}{Global maximum of the sensitivity (derivative) function (\eqn{\epsilon} in 'Details').}
 \item{\code{ELB}}{D-efficiency lower bound. Can not be larger than 1. See 'Note' in \code{\link{sensminimax}} or 'Details' in \code{\link{sens.minimax.control}}.}
 \item{\code{merge_tol}}{Merging tolerance to create the answering set from the set of all local optima. See 'Details' in \code{\link{sens.minimax.control}}.
  Only applicable for minimax or standardized maximin designs.}
 \item{\code{crtval}}{Criterion value. Compare with column \code{Crtiterion_Value} in \code{optima} for minimax and standardized maximin designs.}
 \item{\code{time}}{Required CPU time (rough approximation).}
 }
}
\description{
It plots the sensitivity (derivative) function of the minimax or
 standardized maximin D-optimal criterion
at a given approximate (continuous) design and also
 calculates its efficiency lower bound (ELB) with respect
to the optimality criterion. The user can confirm or reject
 the optimality of a design by looking at the sensitivity plot.
 ELB shows the proximity of a design to the true optimal design without knowing the latter.
 See, for more details, Masoudi et al. (2017).
}
\details{
Let the unknown parameters belong to \eqn{\Theta},
 that is  a Cartesian product of (user-given) intervals for the unknown model parameters.
A design \eqn{\xi^*}{\xi*} is minimax D-optimal among all designs on \eqn{\chi} if and only if there exists a probability measure \eqn{\mu^*}{\mu*} on
   \deqn{A(\xi^*) = \left\{\nu \in \Theta \mid -log|M(\xi^*, \nu)| = \max_{\theta \in \Theta} -log|M(\xi^*, \theta)| \right\},}{
     A(\xi*) = {\nu belongs to \Theta where -log|M(\xi*, \nu) = maxima of function -log|M(\xi*, \theta)| with respect to \theta over \Theta} ,}
       such that the following inequality holds for all \eqn{\boldsymbol{x} \in \chi}{x belong to \chi}
        \deqn{c(\boldsymbol{x}, \mu^*, \xi^*) = \int_{A(\xi^*)} tr M^{-1}(\xi^*, \nu)I(\boldsymbol{x}, \nu)\mu^* d(\nu)-p \leq 0,}{
         c(x, \mu*, \xi*) = integration over A(\xi*) with integrand  tr M^-1(\xi*, \nu)I(x, \nu)\mu* d(\nu)-p <= 0,}
          with equality at all support points of \eqn{\xi^*}{\xi*}.
           Here, \eqn{p} is the number of model parameters. \eqn{c(\boldsymbol{x}, \mu^*, \xi^*)}{c(x, \mu*, \xi*)} is called \strong{sensitivity} or \strong{derivative} function.
The set \eqn{A(\xi^*)}{A(\xi*)} is sometimes called \bold{answering set} of
 \eqn{\xi^*}{\xi*} and the measure \eqn{\mu^*}{\mu*} is a subgradient of the
   non-differentiable criterion evaluated at \eqn{M(\xi^*,\nu)}{M(\xi*,\nu)}.\cr
For standardized maximin D-optimal designs, the answering set \eqn{N(\xi^*)}{N(\xi*)} is
   \deqn{N(\xi^*) = \left\{\boldsymbol{\nu} \in \Theta \mid \mbox{eff}_D(\xi^*, \boldsymbol{\nu}) = \min_{\boldsymbol{\theta} \in \Theta} \mbox{eff}_D(\xi^*, \boldsymbol{\theta}) \right\}.
     }{N(\xi*) = \{\nu belongs to \Theta  where eff_D(\xi*, \nu) = minima of function eff_D(\xi*, \theta)  with respect to \theta over \Theta\},} where
     \eqn{\mbox{eff}_D(\xi, \boldsymbol{\theta}) =  (\frac{|M(\xi, \boldsymbol{\theta})|}{|M(\xi_{\boldsymbol{\theta}}, \boldsymbol{\theta})|})^\frac{1}{p}}{
     eff_D(\xi, \theta) =  (|M(\xi, \theta)|/|M(\xi_\theta, \theta)|)^(1/p)} and \eqn{\xi_\theta} is the locally D-optimal design with respect to \eqn{\theta}. \cr
     See 'Details' of \code{\link{sens.minimax.control}} on how to calculate the answering set \eqn{N(\xi^*)}{N(\xi*)} and \eqn{A(\xi^*)}{A(\xi*)}.

The argument  \code{x} is the vector of design points.
 For design points with more than one dimension,
   it is a concatenation of the design points, but \strong{dimension-wise}.
   For example, let the model has three predictors   \eqn{(I, S, Z)}.
    Then,  (three-dimensional) design points of a two-point optimal design are
   \eqn{\{\mbox{point1} = (I_1, S_1, Z_1), \mbox{point2} = (I_2, S_2, Z_2)\}}{{point1 = (I1, S1, Z1), point2 = (I2, S2, Z2)}}.
    Then, the argument \code{x} is equivalent to
    \code{x = c(I1, I2, S1, S2, Z1, Z2)}.

ELB is a measure of  proximity of a design to the optimal design without knowing the latter.
Given a design, let \eqn{\epsilon} be the global maximum
 of the sensitivity (derivative) function with respect \eqn{x} where \eqn{x \in \chi}{x belong to \chi}.
ELB is given by \deqn{ELB = p/(p + \epsilon),}
where \eqn{p} is the number of model parameters. Obviously,
calculating ELB requires finding \eqn{\epsilon} and
another optimization problem to be solved.
The tuning parameters of this optimization can be regulated via the argument \code{\link{sens.minimax.control}}.
See, for more details, Masoudi et al. (2017).

  The criterion value for the minimax D-optimal design is (global maximum over \eqn{\Theta})
 \deqn{\max_{\theta \in \Theta} -\log|M(\xi, \theta)|;}{max -log|M(\xi, \theta)|;}
 for standardized maximin D-optimal design is (global minimum over \eqn{\Theta})
 \deqn{\inf_{\theta \in \Theta} \left[\left(\frac{|M(\xi, \theta)|}{|M(\xi_{\theta}, \theta)|}\right)^\frac{1}{p}\right].}{
  inf {|M(\xi, \theta)| / |M(\xi_\theta, \theta)|}^p.}
  For locally optimal designs, the criterion value is obtained
  simply by evaluating the criterion at the design (design points and weights) and the initial values for the parameters.


 This function confirms the optimality assuming only a continuous parameter space \eqn{\Theta}.
}
\note{
Theoretically, ELB can not be larger than 1. But if so, it may have one of the following reasons:
\itemize{
\item \code{max_deriv} is not a GLOBAL maximum.  Please increase  the value of the parameter \code{maxeval} in \code{\link{sens.minimax.control}} to find the global maximum.
\item The sensitivity function is shifted below the y-axis because
the number of model parameters has not been specified correctly (less value given).
Please specify the correct number of model parameters via argument \code{npar}.
}
 Please increase the value of the parameter
  \code{n_seg} in \code{\link{sens.minimax.control}}
   for  models with large number of parameters or large parameter space.
}
\examples{
##########################
# Power logistic model
##########################
# verifying the minimax D-optimality of a design with points x0 and weights w0
x0 <- c(-4.5515, 0.2130, 2.8075)
w0 <- c(0.4100, 0.3723, 0.2177)
# Power logistic model when s = .2
sensminimax(formula =  ~ (1/(1 + exp(-b * (x-a))))^.2,
            predvars = "x",
            parvars = c("a", "b"),
            family = binomial(),
            x = x0, w = w0,
            lx = -5, ux = 5,
            lp = c(0, 1), up = c(3, 1.5))

##############################
# A model with two predictors
##############################
# Verifying the minimax D-optimality of a design for a model with two predictors
# The model is the mixed inhibition model.
# X0 is the vector of four design points that are:
# (3.4614, 0) (4.2801, 3.1426) (30, 0) (30, 4.0373)
x0 <- c(3.4614, 4.2801, 30, 30, 0, 3.1426, 0, 4.0373)
w0 <- rep(1/4, 4)
sensminimax(formula =  ~ V*S/(Km * (1 + I/Kic)+ S * (1 + I/Kiu)),
            predvars = c("S", "I"),
            parvars = c("V", "Km", "Kic", "Kiu"),
            family = "gaussian",
            x = x0, w = w0,
            lx = c(0, 0), ux = c(30, 60),
            lp = c(1, 4, 2, 4), up = c(1, 5, 3, 5))

##########################################
# Standardized maximin D-optimal designs
##########################################
# Verifying the standardized maximin D-optimality of a design for
# the loglinear model
# First we should define the function for 'localdes' argument
# The function LDOD takes the parameters and returns the points and
# weights of the locally D-optimal design
LDOD <- function(theta0, theta1, theta2){
  ## param is the vector of theta = (theta0, theta1, theta2)
  lx <- 0 # lower bound of the design space
  ux <- 150 # upper bound of the design space
  param <- c()
  param[1] <- theta0
  param[2] <- theta1
  param[3] <- theta2
  xstar <- (ux+param[3]) * (lx + param[3]) *
    (log(ux + param[3]) - log(lx + param[3]))/(ux - lx) - param[3]
  return(list(x = c(lx, xstar, ux) , w = rep(1/3, 3)))
}
x0 <- c(0, 4.2494, 17.0324, 149.9090)
w0 <- c(0.3204, 0.1207, 0.2293, 0.3296)
\dontrun{
sensminimax(formula = ~theta0 + theta1* log(x + theta2),
            predvars = c("x"),
            parvars = c("theta0", "theta1", "theta2"),
            x = x0, w = w0,
            lx = 0, ux = 150,
            lp = c(2, 2, 1), up = c(2, 2, 15),
            localdes = LDOD,
            standardized = TRUE,
            sens.minimax.control = list(answering.set = list(n_seg = 10)))
}
################################################################
# Not necessary!
# The rest of the examples here are only for professional uses.
################################################################
# Imagine you have written your own FIM, say in Rcpp that is faster than
# the FIM created by the formula interface here.

##########################
# Power logistic model
##########################
# For example, th cpp FIM function for the power logistic model is named:
FIM_power_logistic
args(FIM_power_logistic)
# The arguments do not match the standard of the argument 'fimfunc'
# in 'sensminimax'
# So we reparameterize it:
myfim1 <- function(x, w, param)
  FIM_power_logistic(x = x, w = w, param =param, s = .2)

args(myfim1)
\dontrun{
# Verify minimax D-optimality of a design
sensminimax(fimfunc = myfim1,
            x = c(-4.5515, 0.2130, 2.8075),
            w = c(0.4100, 0.3723, 0.2177),
            lx = -5, ux = 5,
            lp = c(0, 1), up = c(3, 1.5))
}
##############################
# A model with two predictors
##############################
# An example of a  model with two-predictors: mixed inhibition model
# Fisher information matrix:
FIM_mixed_inhibition
args(FIM_mixed_inhibition)

# We should first reparameterize the FIM to match the standard of the
# argument 'fimfunc'
myfim2 <- function(x, w, param){
  npoint <- length(x)/2
  S <- x[1:npoint]
  I <- x[(npoint+1):(npoint*2)]
  out <- FIM_mixed_inhibition(S = S, I = I, w = w, param = param)
  return(out)
}
args(myfim2)
\dontrun{
# Verifyng minimax D-optimality of a design
sensminimax(fimfunc = myfim2,
            x = c(3.4614, 4.2801, 30, 30, 0, 3.1426, 0, 4.0373),
            w = rep(1/4, 4),
            lx = c(0, 0), ux = c(30, 60),
            lp = c(1, 4, 2, 4), up = c(1, 5, 3, 5))
}

#########################################
# Standardized maximin D-optimal designs
#########################################
# An example of a user-written FIM function:
help(FIM_loglin)
# An example of verfying standardaized maximin D-optimality for a design
# Look how we re-define the function LDOD above
LDOD2 <- function(param){
  ## param is the vector of theta = (theta0, theta1, theta2)
  lx <- 0 # lower bound of the design space
  ux <- 150 # upper bound of the design space
  xstar <- (ux + param[3]) * (lx + param[3]) *
    (log(ux + param[3]) - log(lx + param[3]))/(ux - lx) - param[3]
  return(list(x = c(lx, xstar, ux) , w = rep(1/3, 3)))
}

args(LDOD2)

sensminimax(fimfunc = FIM_loglin,
            x = x0,
            w = w0,
            lx = 0, ux = 150,
            lp = c(2, 2, 1), up = c(2, 2, 15),
            localdes = LDOD2,
            standardized = TRUE)



}
\references{
Masoudi E, Holling H, Wong W.K. (2017). Application of Imperialist Competitive Algorithm to Find Minimax and Standardized Maximin Optimal Designs. Computational Statistics and Data Analysis, 113, 330-345. \cr
}
