% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simu.GDO.R
\name{simu.GDO}
\alias{simu.GDO}
\title{Simulate Progeny with GD-O Strategy}
\usage{
simu.GDO(
  fittedA.t,
  fittedD.t = NULL,
  fittedmu.t = NULL,
  geno.t,
  marker,
  geno.c = NULL,
  npl = NULL,
  better.c = FALSE,
  weight = NULL,
  direction = NULL,
  outcross = FALSE,
  nprog = 50,
  nsele = NULL,
  ngen = 10,
  nrep = 30,
  cri = 10000,
  console = TRUE
)
}
\arguments{
\item{fittedA.t}{matrix. An n*t matrix denotes the fitted values of each traits
of the training population. The missing value must have been already imputed.
If outcross is set to be TRUE, this argument must be the additive effect part
of fitted values.}

\item{fittedD.t}{matrix. An n*t matrix denotes the dominance effect part of
fitted values when outcross is set to be TRUE. The missing value must have been
already imputed.}

\item{fittedmu.t}{numeric or vector. A p*1 vector denote the average value of
fitted values when outcross is set to be TRUE. The length must be the same as
the number of traits.}

\item{geno.t}{matrix. An n*p matrix denotes the marker score matrix of the
training population. The markers must be coded as 1, 0, or -1 for alleles
AA, Aa, or aa. The missing value must have been already imputed.}

\item{marker}{matrix. A p*2 matrix whose first column indicates the chromosome
number to which a marker belongs; and second column indicates the position of
the marker in centi-Morgan (cM).}

\item{geno.c}{matrix. An nc*p matrix denotes the marker score matrix of the
candidate population with nc individuals and p markers. It should be pure lines
and markers must be coded as 1, or -1 for alleles AA, or aa. The missing value
must have been already imputed. If geno.c is set to be NULL, the candidate
population is exactly the training population.}

\item{npl}{integer. An integer indicates the number of individuals who will
be chosen as the parental lines. If npl = NULL, it will be 4 times the number
of traits.}

\item{better.c}{logical. A logical variable, if better.c is set to be TRUE,
the candidate individuals with GEBVs better than average for all the target
traits will comprise the candidate set. Otherwise, all the candidate
individuals will comprise the candidate set.}

\item{weight}{vector. A vector with length t indicates the weights of target
traits in selection index. If weight is set to be NULL, the equal weight will
be assigned to all the target traits. The weights should be a positive number.}

\item{direction}{vector. A vector with length t indicates the selecting
directions for target traits. The elements of direction are Inf, or -inf
representing the rule that the larger the better; or the smaller the better.
Or if the element is a number, it will select the individuals with the trait
value close to the number. If direction is set to be NULL, the selecting
direction will be the larger the better for all trait.}

\item{outcross}{logical. A logical variable, if outcross is set to be TRUE,
the crop is regarded as an outcross crop. The kinship matrix of dominance
effects are also considered in the model, and crossing and selection will be
performed in F1 generation. The detail can be seen in the references.}

\item{nprog}{integer. An integer indicates the number of progenies which
will be produced for each of the best individuals at every generation.}

\item{nsele}{integer. An integer indicates the number of the best individuals
which will be selected at each generation. If nsele is set to be NULL, the
number will be the same as the number of F1 individuals.}

\item{ngen}{integer. An integer indicates the number of generations in the
simulation process.}

\item{nrep}{integer. An integer indicates the number of repetitions in the
simulation process.}

\item{cri}{integer. An integer indicates the stopping criterion, note that
cri < 1e+06. The genetic algorithm will stop if the number of iterations
reaches cri.}

\item{console}{logical. A logical variable, if console is set to be TRUE,
the simulation process will be shown in the R console.}
}
\value{
\item{method}{The GD-O strategy.}
\item{weight}{The weights of target traits in selection index.}
\item{direction}{The selecting directions of target traits in selection index.}
\item{mu}{The mean vector of target traits.}
\item{sd}{The standard deviation vector of target traits.}
\item{GEBV.value}{The GEBVs of target traits in each generation and each
repetition.}
\item{parental.lines}{The IDs and D-score of parental lines selected in
each repetition.}
\item{suggested.subset}{The most frequently selected parental lines by this
strategy.}
}
\description{
Identify parental lines based on GD-O strategy and simulate their offsprings.
}
\note{
The function output.best and output.gain can be used to summarize the result.

The fitted value data in the input data can be obtained by the function
GBLUP.fit and mmer, that can be seen in the Examples shown below.
}
\examples{
# generate simulated data
set.seed(2000)
t1 <- rnorm(10,30,10)
t2 <- rnorm(10,10,5)
t3 <- NULL
t4 <- NULL
t5 <- NULL
geno.test <- matrix(sample(c(1, -1), 200, replace = TRUE), 10, 20)
marker.test <- cbind(rep(1:2, each=10), rep(seq(0, 90, 10), 2))
fit <- GBLUP.fit(t1, t2, t3, t4, t5, geno = geno.test)
fitvalue <- fit$fitted.value

geno.candidate <- matrix(sample(c(1,-1), 300, replace = TRUE), 15, 20)

# run and output
result <- simu.GDO(fitvalue, geno.t = geno.test, marker = marker.test,
geno.c = geno.candidate, nprog = 5, nsele = 10, ngen = 5, nrep = 5, cri = 250)
result$suggested.subset



# other method: use mmer to obtain the fitted value
\dontrun{
set.seed(2000)
t1 <- rnorm(10,30,10)
t2 <- rnorm(10,10,5)
phe <- cbind(t1, t2)
nt <- ncol(phe)
geno.test <- matrix(sample(c(1, -1), 200, replace = TRUE), 10, 20)
marker.test <- cbind(rep(1:2, each=10), rep(seq(0, 90, 10), 2))
rownames(geno.test) <- 1:nrow(geno.test)
id <- rownames(geno.test)
K0 <- geno.test\%*\%t(geno.test)/ncol(geno.test)

dat <- data.frame(id, phe)
fit0 <- sommer::mmer(cbind(t1, t2)~1,
      random = ~sommer::vs(id, Gu = K0, Gtc = sommer::unsm(nt)),
      rcov = ~sommer::vs(units, Gtc = sommer::unsm(nt)),
      data = dat,
      tolparinv = 0.01)

u0 <- fit0$U$`u:id`
fit <- matrix(unlist(u0), ncol = nt)
colnames(fit) <- names(u0)

fit <- fit+matrix(fit0$fitted[1,], nrow(fit), nt, byrow = TRUE)
fitvalue <- fit[order(as.numeric(names((u0[[1]])))),]
}
}
\references{
Chung PY, Liao CT. 2020. Identification of superior parental lines for
biparental crossing via genomic prediction. PLoS ONE 15(12):e0243159.
}
\seealso{
\code{\link[sommer]{mmer}}
\code{\link[IPLGP]{GBLUP.fit}}
\code{\link[IPLGP]{GA.Dscore}}
\code{\link[IPLGP]{simu.gamete}}
\code{\link[IPLGP]{simu.GDO}}
\code{\link[IPLGP]{simu.GEBVGD}}
\code{\link[IPLGP]{output.best}}
\code{\link[IPLGP]{output.gain}}
}
