% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/isofind.R
\name{isofind}
\alias{isofind}
\alias{print.ISOFIND}
\alias{summary.ISOFIND}
\title{Infer spatial origins}
\usage{
isofind(
  data,
  isoscape,
  calibfit = NULL,
  mask = NA,
  neglect_covPredCalib = TRUE,
  verbose = interactive()
)
}
\arguments{
\item{data}{A \emph{dataframe} containing the assignment data (see note below)}

\item{isoscape}{The output of the function \link{isoscape}}

\item{calibfit}{The output of the function \link{calibfit} (This
argument is not needed if the isoscape had been fitted using isotopic
ratios from sedentary animals.)}

\item{mask}{A polygon of class \emph{SpatVector} representing a mask to replace values on all
rasters by NA inside polygons (see details)}

\item{neglect_covPredCalib}{A \emph{logical} indicating whether to neglect the
covariance between the uncertainty of predictions from the isoscape mean
fit and the uncertainty in predictions from the calibration fit (default =
\code{TRUE}). See \strong{Details}.}

\item{verbose}{A \emph{logical} indicating whether information about the
progress of the procedure should be displayed or not while the function is
running. By default verbose is \code{TRUE} if users use an interactive R
session and \code{FALSE} otherwise.}
}
\value{
This function returns a \emph{list} of class \emph{ISOFIND} containing
itself three lists (\code{sample}, \code{group}, and \code{sp_points})
storing all rasters built during assignment and the spatial points for
sources, calibration and assignments. The \emph{list} \code{sample} contains
three set of raster layers: one storing the value of the test statistic
("stat"), one storing the value of the variance of the test statistic
("var") and one storing the p-value of the test ("pv"). The \emph{list}
\code{group} contains one raster storing the p-values of the assignment for
the group. The \emph{list} \code{sp_points} contains two spatial point
objects: \code{sources} and \code{calibs}.
}
\description{
This function performs the assignment of samples of unknown origins.
}
\details{
An assignment is a comparison, for a given organism, of the predicted
isotopic source value at its location of origin and the predicted isotopic
source value at each location of the \code{isoscape}. The difference between
these two values constitute the statistic of the assignment test. Under the
null hypothesis (the organism is at a location with the same isotopic value
than its original location), the test statistics follows a normal
distribution with mean zero and a certain variance that stems from both the
isoscape model fits and the calibration fit. The function
\link{isofind} computes the map of p-value for such an assignment test
(i.e. the p-values in all locations of the isoscape) for all samples in the
dataframe \code{data}. The function also performs a single assignment for the
entire group by combining the p-value maps of all samples using the Fisher's
method (Fisher 1925). Significant p-values are strong evidence that the
sample do NOT come from the candidate location (and not the opposite!). For
statistical details about this procedure as well as a discussion of which
uncertainties are captured and which are not, please refer to Courtiol et al.
2019.

\strong{Details on parameters:}
\itemize{
\item \emph{neglect_covPredCalib}: as long as the calibration method used in
\link{calibfit} is "wild", a covariance is expected between the
uncertainty of predictions from the isoscape mean fit and the uncertainty in
predictions from the calibration fit. This is because both the isoscape and
the calibration use in part the same data. By default this term is omitted
(i.e. the value for the argument \code{neglect_covPredCalib} is \code{TRUE})
since in practice it seems to affect the results only negligibly in our
trials and the computation of this term can be quite computer intensive. We
nonetheless recommend to set \code{neglect_covPredCalib} to \code{FALSE} in
your final analysis. If the calibration method used in \link{calibfit}
is not "wild", this parameter has no effect.
\item \emph{mask}: a mask can be used so to remove all values falling in the mask.
This can be useful for performing for example assignments on lands only and
discard anything falling in large bodies of water (see example). By default
our \link{OceanMask} is considered. Setting \code{mask} to NULL allows
to prevent this automatic behaviour.
}
}
\note{
See \link{AssignDataAlien} to know which variables are needed to
perform the assignment and their names.
}
\examples{

## The examples below will only be run if sufficient time is allowed
## You can change that by typing e.g. options_IsoriX(example_maxtime = XX)
## if you want to allow for examples taking up to ca. XX seconds to run
## (so don't write XX but put a number instead!)

if (getOption_IsoriX("example_maxtime") > 200) {
  ## We fit the models for Germany
  GNIPDataDEagg <- prepsources(data = GNIPDataDE)

  GermanFit <- isofit(
    data = GNIPDataDEagg,
    mean_model_fix = list(elev = TRUE, lat_abs = TRUE)
  )


  ## We build the isoscape
  GermanScape <- isoscape(
    raster = ElevRasterDE,
    isofit = GermanFit
  )


  ## We fit the calibration model
  CalibAlien <- calibfit(
    data = CalibDataAlien,
    isofit = GermanFit
  )

  ## We perform the assignment on land only
  AssignmentDry <- isofind(
    data = AssignDataAlien,
    isoscape = GermanScape,
    calibfit = CalibAlien
  )

  ## perform the assignment on land and water
  Assignment <- isofind(
    data = AssignDataAlien,
    isoscape = GermanScape,
    calibfit = CalibAlien,
    mask = NULL
  )

  ## We plot the group assignment
  plot(Assignment, who = "group", mask = list(mask = NULL))

  plot(AssignmentDry, who = "group", mask = list(mask = NULL))

  ## We plot the assignment for the 8 first samples
  plot(AssignmentDry,
    who = 1:8,
    sources = list(draw = FALSE),
    calibs = list(draw = FALSE)
  )

  ## We plot the assignment for the sample "Alien_10"
  plot(AssignmentDry, who = "Alien_10")


  ### Other example without calibration:
  ### We will try to assign a weather station
  ### in the water isoscape

  ## We create the assignment data taking
  ## GARMISCH-PARTENKIRCHEN as the station to assign
  GPIso <- GNIPDataDEagg[GNIPDataDEagg$source_ID == "GARMISCH-PARTENKIRCHEN", "mean_source_value"]
  AssignDataGP <- data.frame(
    sample_value = GPIso,
    sample_ID = "GARMISCH-PARTENKIRCHEN"
  )

  ## We perform the assignment
  AssignedGP <- isofind(
    data = AssignDataGP,
    isoscape = GermanScape,
    calibfit = NULL
  )
  ## We plot the assignment and
  ## show where the station really is (using lattice)
  plot(AssignedGP) +
    xyplot(47.48 ~ 11.06,
      panel = panel.points,
      cex = 5, pch = 13, lwd = 2, col = "black"
    )
}

}
\references{
Courtiol A, Rousset F, Rohwäder M, Soto DX, Lehnert L, Voigt CC, Hobson KA, Wassenaar LI, Kramer-Schadt S (2019). Isoscape
computation and inference of spatial origins with mixed models using the R package IsoriX. In Hobson KA, Wassenaar LI (eds.),
Tracking Animal Migration with Stable Isotopes, second edition. Academic Press, London.

Fisher, R.A. (1925). Statistical Methods for Research Workers.
Oliver and Boyd (Edinburgh). ISBN 0-05-002170-2.
}
\keyword{models}
\keyword{regression}
