package org.apache.lucene.search.spell;



/*   < cost table explaination >

1) 'insert', 'delete'   =>   1
   'space'  : 'insert', 'delete' => 0.5
    number  : 'insert', 'delete' => 2

2) a small letter <=> a small letter within 1 keybord space           =>   0.5

3) a small letter <=> a small letter within 2 keybord space           =>   1

4) a small letter <=> a small letter within 3 keybord space           =>   1.5

5) a small letter <=> a small letter over 3 keybord space             =>   2

6) a capital letter <=> a capital letter within 1 keybord space	=>   0.5

7) a capital letter <=> a small letter within 1 keybord space	        =>   0.5

6) a capital letter <=> a small letter                                =>   0.3

7) letter           <=> number		                        =>   1.5

8) letter           <=> space		                                =>   1.5

9) space            <=> number                        	        =>   1.5

10)number           <=> number	                  		=>   2

11) 'ㄸ' <=> 'ㅌ' , 'ㅋ' <=> 'ㄱ'                                	     => 0.5 (phonetic distance)
12) a small letter <=> a capital letter over 3 keyboard space  =>   *3*
*/

//09.01.05
					//{a,A,b,B,c,C,d,D,e,E,f,F,g,G,h,H,i,I,j,J,k,K,l,L,m,M,n,N,o,O,p,P,q,Q,r,R,s,S,t,T,u,U,v,V,w,W,x,X,y,Y,z,Z,0,1,2,3,4,5,6,7,8,9,_,$,#,delete},

public final class KoLevensteinDistance implements StringDistance {
	
	static final int COST_TABLE_SIZE = 66;
	
	static final double cost[][] =
	{
			{0,0.3,2,3,1.5,3,1,3,1,1,1.5,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,2,2,2,0.5,0.5,1.5,1.5,0.5,0.5,2,2,2,3,2,3,0.5,0.5,1,3,2,3,0.5,0.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,100,100,1},
			{0.3,0,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,0.5,0.5,3,3,0.5,0.5,3,3,3,3,3,3,0.5,0.5,3,3,3,3,0.5,0.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,100,100,1},
			{2,3,0,0.3,1,3,1.5,3,2,2,1,3,0.5,0.5,0.5,0.5,1.5,3,1,3,1.5,3,2,3,1,3,0.5,0.5,2,2,2,2,2,2,1.5,1.5,2,3,1,1,1,3,0.5,0.5,2,2,1.5,3,1,3,2,3,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,100,100,1},
			{3,3,0.3,0,3,3,3,3,3,3,3,3,0.5,0.5,0.5,0.5,3,3,3,3,3,3,3,3,3,3,0.5,0.5,3,3,3,3,3,3,3,3,3,3,3,3,3,3,0.5,0.5,3,3,3,3,3,3,3,3,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,100,100,1},
			{1.5,3,1,3,0,0.3,0.5,0.5,0.5,1,0.5,0.5,1,3,1.5,3,2,3,2,3,2,3,2,3,2,3,1.5,3,2,2,2,2,2,2,1,1,1,3,0.5,0.5,2,3,3,3,0.5,0.5,0.5,0.5,1.5,3,1,3,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,100,100,1},
			{3,3,3,3,0.3,0,0.5,0.5,3,3,0.5,0.5,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,0.5,0.5,3,3,3,3,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,100,100,1},
			{1,3,1.5,3,0.5,0.5,0,0.3,0.5,0.5,0.5,0.5,1,3,1.5,3,2,3,2,3,2,3,2,3,2,3,2,3,2,2,2,2,1.5,1.5,0.5,0.5,0.5,0.5,1,1,2,3,0.5,0.5,1,1,0.5,0.5,1.5,3,1,3,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,100,100,1},
			{3,3,3,3,0.5,0.5,0.3,0,0.5,0.5,0.5,0.5,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,0.5,0.5,0.5,0.5,3,3,3,3,0.5,0.5,3,3,0.5,0.5,3,3,3,3,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,100,100,1},
			{1,3,2,3,0.5,3,0.5,3,0,0.3,1,3,1.5,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,2,2,2,1,1,0.5,0.5,0.5,0.5,0.5,0.5,2,3,1.5,3,0.5,0.5,0.5,3,1.5,3,1,3,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,100,100,1},
			{1,3,2,3,1,3,0.5,0.5,0.3,0,1,3,1.5,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,2,2,2,1,1,0.5,0.5,0.5,0.5,1,1,2,3,1.5,3,0.5,0.5,0.5,3,1.5,3,1,3,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,100,100,1},
			{1.5,3,1,3,0.5,0.5,0.5,0.5,1,1,0,0.3,0.5,0.5,1,3,2,3,1.5,3,2,3,2,3,2,3,1.5,3,2,2,2,2,2,2,0.5,0.5,1,3,0.5,0.5,1.5,3,0.5,0.5,1.5,1.5,1,3,1,3,1.5,3,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,100,100,1},
			{3,3,3,3,0.5,0.5,0.5,0.5,3,3,0.3,0,0.5,0.5,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,0.5,0.5,3,3,0.5,0.5,3,3,0.5,0.5,3,3,3,3,3,3,3,3,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,100,100,1},
			{2,3,0.5,0.5,1,3,1,3,1.5,1.5,0.5,0.5,0,0.3,0.5,0.5,1.5,3,1,3,1.5,3,2,3,1.5,3,1,3,2,2,2,2,2,2,1,1,1.5,3,0.5,0.5,1,3,0.5,0.5,2,2,1.5,3,0.5,0.5,2,3,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,100,100,1},
			{3,3,0.5,0.5,3,3,3,3,3,3,0.5,0.5,0.3,0,0.5,0.5,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,0.5,0.5,3,3,0.5,0.5,3,3,3,3,0.5,0.5,3,3,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,100,100,1},
			{2,3,0.5,0.5,1.5,3,1.5,3,2,2,1,3,0.5,0.5,0,0.3,1,3,0.5,0.5,1,3,1.5,3,1,3,0.5,0.5,1.5,1.5,2,2,2,2,1.5,1.5,2,3,1,1,0.5,0.5,1,3,2,2,2,3,0.5,0.5,2,3,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,100,100,1},
			{3,3,0.5,0.5,3,3,3,3,3,3,3,3,0.5,0.5,0.3,0,3,3,0.5,0.5,3,3,3,3,3,3,0.5,0.5,3,3,3,3,3,3,3,3,3,3,3,3,0.5,0.5,3,3,3,3,3,3,0.5,0.5,3,3,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,100,100,1},
			{2,3,1.5,3,2,3,2,3,2,2,2,3,1.5,3,1,3,0,0.3,0.5,0.5,0.5,0.5,1,3,1,3,1,3,0.5,0.5,1,1,2,2,2,2,2,3,1.5,1.5,0.5,0.5,2,3,2,2,2,3,1,3,2,3,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,100,100,1},
			{3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,0.3,0,0.5,0.5,0.5,0.5,3,3,3,3,3,3,0.5,0.5,3,3,3,3,3,3,3,3,3,3,0.5,0.5,3,3,3,3,3,3,3,3,3,3,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,100,100,1},
			{2,3,1,3,2,3,2,3,2,2,1.5,3,1,3,0.5,0.5,0.5,0.5,0,0.3,0.5,0.5,1,3,0.5,0.5,0.5,0.5,1,1,1,1.5,2,2,2,2,2,3,1.5,1.5,0.5,0.5,1.5,3,2,2,2,3,1,3,2,3,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,100,100,1},
			{3,3,3,3,3,3,3,3,3,3,3,3,3,3,0.5,0.5,0.5,0.5,0.3,0,0.5,0.5,3,3,0.5,0.5,0.5,0.5,3,3,3,3,3,3,3,3,3,3,3,3,0.5,0.5,3,3,3,3,3,3,3,3,3,3,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,100,100,1},
			{2,3,1.5,3,2,3,2,3,2,2,2,3,1.5,3,1,3,0.5,3,0.5,0.5,0,0.3,0.5,0.5,0.5,0.5,1,3,0.5,0.5,1,1,2,2,2,2,2,3,2,2,1,3,2,3,2,2,2,3,1.5,3,2,3,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,100,100,1},
			{3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,0.5,0.5,0.5,0.5,0.3,0,0.5,0.5,0.5,0.5,3,3,0.5,0.5,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,100,100,1},
			{2,3,2,3,2,3,2,3,2,2,2,3,2,3,1.5,3,1,3,1,3,0.5,0.5,0,0.3,1,3,1.5,3,0.5,0.5,0.5,0.5,2,2,2,2,2,3,2,2,1.5,3,2,3,2,2,2,3,2,3,2,3,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,100,100,1},
			{3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,0.5,0.5,0.3,0,3,3,3,3,0.5,0.5,0.5,0.5,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,100,100,1},
			{2,3,1,3,2,3,2,3,2,2,2,3,1.5,3,1,3,1,3,0.5,0.5,0.5,0.5,1,3,0,0.3,0.5,0.5,1,1,1.5,1.5,2,2,2,2,2,3,2,2,1,3,1.5,3,2,2,2,3,1.5,3,2,3,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,100,100,1},
			{3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,0.5,0.5,0.5,0.5,3,3,0.3,0,0.5,0.5,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,100,100,1},
			{2,3,0.5,0.5,1.5,3,2,3,2,2,1.5,3,1,3,0.5,0.5,1,3,0.5,0.5,1,3,1.5,3,0.5,0.5,0,0.3,1.5,1.5,2,2,2,2,2,2,2,3,1.5,1.5,1,3,1,3,2,2,2,3,1,3,2,3,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,100,100,1},
			{3,3,0.5,0.5,3,3,3,3,3,3,3,3,3,3,0.5,0.5,3,3,0.5,0.5,3,3,3,3,0.5,0.5,0.3,0,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,100,100,1},
			{2,3,2,3,2,3,2,3,2,2,2,3,2,3,1.5,3,0.5,0.5,1,3,0.5,0.5,0.5,0.5,1,3,1.5,3,0,0.3,0.5,0.5,2,2,2,2,2,3,2,2,1,3,2,3,2,2,2,3,1.5,3,2,3,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,100,100,1},
			{2,3,2,3,2,3,2,3,2,2,2,3,2,3,1.5,3,0.5,0.5,1,3,0.5,0.5,0.5,0.5,1,3,1.5,3,0.3,0,0.5,0.5,2,2,2,2,2,3,2,2,1,3,2,3,2,2,2,3,1.5,3,2,3,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,100,100,1},
			{2,3,2,3,2,3,2,3,2,2,2,3,2,3,2,3,1,3,1,3,1,3,0.5,0.5,1.5,3,2,3,0.5,0.5,0,0.3,2,2,2,2,2,3,2,2,1.5,3,2,3,2,2,2,3,2,3,2,3,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,100,100,1},
			{2,3,2,3,2,3,2,3,2,2,2,3,2,3,2,3,1,3,1.5,3,1,3,0.5,0.5,1.5,3,2,3,0.5,0.5,0.3,0,2,2,2,2,2,3,2,2,1.5,3,2,3,2,2,2,3,2,3,2,3,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,100,100,1},
			{0.5,0.5,2,3,2,3,1.5,3,1,1,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,2,2,2,0,0.3,1.5,1.5,1,3,2,2,2,3,0.5,3,0.5,0.5,1.5,3,2,3,1,3,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,100,100,1},
			{0.5,0.5,2,3,2,3,1.5,3,1,1,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,2,2,2,0.3,0,1.5,1.5,1,3,2,2,2,3,0.5,3,0.5,0.5,1.5,3,2,3,1,3,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,100,100,1},
			{1.5,3,1.5,3,1,3,0.5,0.5,0.5,0.5,0.5,0.5,1,3,1.5,3,2,3,2,3,2,3,2,3,2,3,2,3,2,2,2,2,1.5,1.5,0,0.3,1,3,0.5,0.5,1.5,3,1,3,1,1,1,3,1,3,0.5,3,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,100,100,1},
			{1.5,3,1.5,3,1,3,0.5,0.5,0.5,0.5,0.5,0.5,1,3,1.5,3,2,3,2,3,2,3,2,3,2,3,2,3,2,2,2,2,1.5,1.5,0.3,0,1,3,0.5,0.5,1.5,3,1,3,1,1,1,3,1,3,0.5,3,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,100,100,1},
			{0.5,0.5,2,3,1,3,0.5,0.5,0.5,0.5,1,3,1.5,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,2,2,2,1,1,1,1,0,0.3,1.5,1.5,2,3,1.5,3,0.5,0.5,0.5,0.5,2,3,0.5,0.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,100,100,1},
			{0.5,0.5,3,3,3,3,0.5,0.5,0.5,0.5,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,0.3,0,3,3,3,3,3,3,0.5,0.5,0.5,0.5,3,3,0.5,0.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,100,100,1},
			{2,3,1,3,0.5,3,1,3,0.5,1,0.5,0.5,0.5,0.5,1,3,1.5,3,1.5,3,2,3,2,3,2,3,1.5,3,2,2,2,2,2,2,0.5,0.5,1.5,3,0,0.3,1,3,1,3,0.5,1.5,0.5,3,0.5,0.5,2,3,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,100,100,1},
			{2,3,1,3,0.5,3,1,3,0.5,1,0.5,0.5,0.5,0.5,1,3,1.5,3,1.5,3,2,3,2,3,2,3,1.5,3,2,2,2,2,2,2,0.5,0.5,1.5,3,0.3,0,1,3,1,3,0.5,1.5,0.5,3,0.5,0.5,2,3,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,100,100,1},
			{2,3,1,3,2,3,2,3,2,2,1.5,3,1,3,0.5,0.5,0.5,0.5,0.5,0.5,1,3,1.5,3,1,3,1,3,1,1,1.5,1.5,2,2,1.5,1.5,2,3,1,1,0,0.3,1.5,3,2,2,2,3,0.5,0.5,2,3,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,100,100,1},
			{3,3,3,3,3,3,3,3,3,3,3,3,3,3,0.5,0.5,0.5,0.5,0.5,0.5,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,0.3,0,3,3,3,3,3,3,0.5,0.5,3,3,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,100,100,1},
			{2,3,0.5,0.5,0.5,0.5,1,3,1.5,1.5,0.5,0.5,0.5,0.5,1,3,2,3,1.5,3,2,3,2,3,1.5,3,1,3,2,2,2,2,0.5,0.5,1,1,1.5,3,1,1,1.5,3,0,0.3,2,2,1,3,1,3,1.5,3,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,100,100,1},
			{3,3,0.5,0.5,0.5,0.5,3,3,3,3,0.5,0.5,0.5,0.5,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,0.3,0,3,3,3,3,3,3,3,3,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,100,100,1},
			{0.5,0.5,2,3,0.5,3,1,3,0.5,0.5,1.5,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,2,2,2,0.5,0.5,1,1,0.5,0.5,0.5,0.5,2,3,2,3,0,0.3,0.5,3,2,3,1,3,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,100,100,1},
			{0.5,0.5,2,3,0.5,3,1,3,0.5,0.5,1.5,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,2,2,2,0.5,0.5,1,1,0.5,0.5,1.5,1.5,2,3,2,3,0.3,0,1,3,2,3,1,3,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,100,100,1},
			{1,3,1.5,3,0.5,0.5,0.5,0.5,0.5,0.5,1,3,1.5,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,2,2,2,1.5,1.5,1,1,0.5,0.5,0.5,0.5,2,3,1,3,0.5,1,0,0.3,2,3,0.5,0.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,100,100,1},
			{3,3,3,3,0.5,0.5,0.5,0.5,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,0.5,0.5,3,3,3,3,3,3,3,3,0.3,0,3,3,0.5,0.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,100,100,1},
			{2,3,1,3,1.5,3,1.5,3,1.5,1.5,1,3,0.5,0.5,0.5,0.5,1,3,1,3,1.5,3,2,3,1.5,3,1,3,1.5,1.5,2,2,2,2,1,1,2,3,0.5,0.5,0.5,0.5,1,3,2,2,2,3,0,0.3,2,3,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,100,100,1},
			{3,3,3,3,3,3,3,3,3,3,3,3,0.5,0.5,0.5,0.5,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,0.5,0.5,0.5,0.5,3,3,3,3,3,3,0.3,0,3,3,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,100,100,1},
			{0.5,0.5,2,3,1,3,1,3,1,1,1.5,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,2,2,2,1,1,0.5,0.5,0.5,0.5,2,2,2,3,1.5,3,1,1,0.5,0.5,2,3,0,0.3,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,100,100,1},
			{0.5,0.5,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,0.5,0.5,3,3,3,3,3,3,3,3,0.5,0.5,3,3,0.3,0,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,100,100,1},
			{1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,0,2,2,2,2,2,2,2,2,2,1.5,100,100,2},
			{1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,2,0,2,2,2,2,2,2,2,2,1.5,100,100,2},
			{1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,2,2,0,2,2,2,2,2,2,2,1.5,100,100,2},
			{1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,2,2,2,0,2,2,2,2,2,2,1.5,100,100,2},
			{1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,2,2,2,2,0,2,2,2,2,2,1.5,100,100,2},
			{1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,2,2,2,2,2,0,2,2,2,2,1.5,100,100,2},
			{1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,2,2,2,2,2,2,0,2,2,2,1.5,100,100,2},
			{1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,2,2,2,2,2,2,2,0,2,2,1.5,100,100,2},
			{1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,2,2,2,2,2,2,2,2,0,2,1.5,100,100,2},
			{1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,2,2,2,2,2,2,2,2,2,0,1.5,100,100,2},
			{1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,1.5,0,100,100,0.5},
			{100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,0,100,0.15},
			{100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,0,0.2},
			{1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,2,2,2,2,2,2,2,2,2,2,0.5,0.15,0.2,1}

	 };
	
	
	private float insertCost(char c){
	    double cost_sum = 0;
	    if(c >= 97 && c <= 122)
	        cost_sum =  cost[(c-'a')*2][ COST_TABLE_SIZE-1 ]  ;
	    else if(c >= 65 && c <=90)
	        cost_sum = cost[(c-'A')*2+1][ COST_TABLE_SIZE-1 ]  ;
	    else if(c >= 48 && c <= 57)
	        cost_sum = cost[c + 4][ COST_TABLE_SIZE-1 ]  ;
	    else if( c == 32 )
	        cost_sum =  cost[ 62 ][ COST_TABLE_SIZE-1 ]  ;
	    else
	        assert(false);

	    return (float)cost_sum;
	}
	
	private float deleteCost(char c){
	    double cost_sum = 0;
	    if(c >= 97 && c <= 122)
	        cost_sum = cost[ COST_TABLE_SIZE-1 ][(c-'a')*2]  ;
	    else if(c >= 65 && c <=90)
	        cost_sum = cost[ COST_TABLE_SIZE-1 ][(c-'A')*2+1]  ;
	    else if(c >= 48 && c <= 57)
	        cost_sum = cost[ COST_TABLE_SIZE-1 ][ c + 4 ]  ;
	    else if( c == 32 )
	        cost_sum = cost[ COST_TABLE_SIZE-1 ][ 62 ]  ;
	    else
	        assert(false);

	    return (float)cost_sum;
	}
	
	private float transCost(char s, char t){
	    if(s == t)
	        return 0;
	    double cost_sum = 0;
	    if(s >= 97 && s <= 122){
	        if(t >= 97 && t <= 122)
	            cost_sum = cost[(s-'a')*2][(t-'a')*2]  ;
	        else if(t >= 65 && t <= 90)
	            cost_sum = cost[(s-'a')*2][(t-'A')*2+1]  ;
	        else if(t >= 48 && t <= 57)
	            cost_sum = cost[(s-'a')*2][t + 4]  ;
	        else if(t == 32)
	            cost_sum = cost[(s-'a')*2][ 62 ]  ;
	        else
	            assert(false);
	    }else if(s >= 65 && s <=90){
	        if(t >= 97 && t <= 122)
	            cost_sum = cost[(s-'A')*2+1][(t-'a')*2]  ;
	        else if(t >= 65 && t <= 90)
	            cost_sum = cost[(s-'A')*2+1][(t-'A')*2+1]  ;
	        else if(t >= 48 && t <= 57)
	            cost_sum = cost[(s-'A')*2+1][ t +4 ]  ;
	        else if(t == 32)
	            cost_sum = cost[(s-'A')*2+1][ 62 ]  ;
	        else
	            assert(false);
	    }else if( s >= 48 && s <= 57){
	        if(t >= 97 && t <= 122)
	            cost_sum = cost[s + 4][(t-'a')*2]  ;
	        else if(t >= 65 && t <= 90)
	            cost_sum = cost[s + 4][(t-'A')*2+1]  ;
	        else if(t >= 48 && t <= 57)
	            cost_sum = cost[s + 4][ t +4 ]  ;
	        else if(t == 32)
	            cost_sum = cost[s + 4][ 62 ]  ;
	        else
	            assert(false);
	    }else if(s == 32){
	        if(t >= 97 && t <= 122)
	            cost_sum = cost[ 62 ][(t-'a')*2]  ;
	        else if(t >= 65 && t <= 90)
	            cost_sum = cost[ 62 ][(t-'A')*2+1]  ;
	        else if(t >= 48 && t <= 57)
	            cost_sum = cost[ 62 ][ t +4 ]  ;
	        else if(t == 32)
	            cost_sum = cost[ 62 ][ 62 ]  ;
	        else
	            assert(false);
	    }else
	        assert(false);

	    return (float)cost_sum;
	}
	
	
	public float getDistance(String target, String other) {
		char[] sa;
		int n;
		float p[]; //'previous' cost array, horizontally
		float d[]; // cost array, horizontally
		float _d[]; //placeholder to assist in swapping p and d

		/*
	           The difference between this impl. and the previous is that, rather
	           than creating and retaining a matrix of size s.length()+1 by t.length()+1,
	           we maintain two single-dimensional arrays of length s.length()+1.  The first, d,
	           is the 'current working' distance array that maintains the newest distance cost
	           counts as we iterate through the characters of String s.  Each time we increment
	           the index of String t we are comparing, d is copied to p, the second int[].  Doing so
	           allows us to retain the previous cost counts as required by the algorithm (taking
	           the minimum of the cost count to the left, up one, and diagonally up and to the left
	           of the current cost count being calculated).  (Note that the arrays aren't really
	           copied anymore, just switched...this is clearly much better than cloning an array
	           or doing a System.arraycopy() each time  through the outer loop.)

	           Effectively, the difference between the two implementations is this one does not
	           cause an out of memory condition when calculating the LD over two very large strings.
		 */

		sa = target.toCharArray();
		n = sa.length;
		p = new float[n+1]; 
		d = new float[n+1]; 

		final int m = other.length();
		if (n == 0 || m == 0) {
			if (n == m) {
				return 1;
			}
			else {
				return 0;
			}
		} 


		// indexes into strings s and t
		int i; // iterates through s
		int j; // iterates through t

		char t_j; // jth character of t

		float cost; // cost

		for (i = 0; i<=n; i++) {
			p[i] = i;
		}

		for (j = 1; j<=m; j++) {
			t_j = other.charAt(j-1);
			d[0] = j;

			for (i=1; i<=n; i++) {
				cost = sa[i-1]==t_j ? 0 : transCost(sa[i-1], t_j);
				// minimum of cell to the left+1, to the top+1, diagonally left and up +cost
				d[i] =  Math.min(
							Math.min(d[i-1] + insertCost(t_j), p[i] + deleteCost(t_j)),  
							p[i-1]+cost);
			}

			// copy current distance counts to 'previous row' distance counts
			_d = p;
			p = d;
			d = _d;
		}

		// our last action in the above loop was to switch d and p, so p now
		// actually has the most recent cost counts
		//return 1.0f - ((float) p[n] / Math.max(other.length(), sa.length));
		return p[n];
	}
	
	
   
}
