% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/LGICARL_CUSUM_Up.R
\name{ARL_Clplus}
\alias{ARL_Clplus}
\title{ARL Estimation in CUSUM Control Charts with Gamma Distribution and Cautious Learning
for upward detection}
\usage{
ARL_Clplus(
  alpha,
  beta,
  alpha0_est,
  beta0_est,
  known_alpha,
  beta_ratio,
  H_delta,
  H_plus,
  n_I,
  replicates,
  K_l,
  delay,
  tau
)
}
\arguments{
\item{alpha}{Shape parameter of the Gamma distribution.}

\item{beta}{Scale parameter of the Gamma distribution.}

\item{alpha0_est}{Initial estimate of the shape parameter \code{alpha}. If \code{known_alpha} is \code{TRUE}, this value will be equal to \code{alpha}.}

\item{beta0_est}{Initial estimate of the scale parameter \code{beta}. This value is updated dynamically during the simulation.}

\item{known_alpha}{\code{TRUE} if \code{alpha0_est} is fixed, \code{FALSE} if it must be estimated.}

\item{beta_ratio}{Ratio between \code{beta} and its posterior estimate.}

\item{H_delta}{Increment of the upper control limit in the CUSUM chart.}

\item{H_plus}{Initial control limit of the CUSUM chart.}

\item{n_I}{Sample size in Phase I.}

\item{replicates}{Number of Monte Carlo simulations.}

\item{K_l}{Secondary control threshold for parameter updating.}

\item{delay}{Number of observations before updating \code{beta0_est}.}

\item{tau}{Time point at which \code{beta} changes. A value of 1 is recommended for IC states.}
}
\value{
A numeric value corresponding to the \strong{ARL} estimate for the upward CUSUM control chart with cautious learning.
}
\description{
This function calculates the \strong{Average Run Length (ARL)} of a CUSUM control chart based on the Gamma distribution,
incorporating a cautious learning scheme for the progressive update of parameters and optimization of performance in upward detection.

The function allows for the evaluation of the CUSUM chart’s behavior under different parameterization scenarios,
ensuring efficient detection of process changes.

Following the methodology presented in the work of Madrid-Alvarez, García-Díaz, and Tercero-Gómez (2024),
this implementation utilizes Monte Carlo simulations in C++ for efficient execution,
ensuring a dynamic adjustment of parameters based on the evolution of the process.

The values of \code{H_plus}, \code{H_delta}, \code{K_l}, \code{delay}, and \code{tau} can be referenced in the tables from the article:

\strong{Madrid-Alvarez, H. M., García-Díaz, J. C., & Tercero-Gómez, V. G. (2024).}
\emph{A CUSUM control chart for the Gamma distribution with cautious parameter learning.}
Quality Engineering, 1-23.
\subsection{Usage Scenarios:}{

\strong{Scenario 1: Known \code{alpha} and estimated \code{beta}}
\itemize{
\item The \code{alpha} parameter is assumed to be fixed and known in advance.
\item \code{beta} is estimated from a dataset or defined by the user.
\item The user must provide \code{alpha} and an initial estimate of \code{beta} (\code{beta0_est}).
}

\strong{Scenario 2: Both \code{alpha} and \code{beta} are estimated}
\itemize{
\item Both \code{alpha} and \code{beta} are estimated from a dataset or external data source.
\item The user must calculate \code{alpha0_est} and \code{beta0_est} before running the function.
\item \code{beta0_est} is dynamically updated during the simulation when a predefined condition is met.
}
}

\subsection{Features:}{
\itemize{
\item Implements Monte Carlo simulations optimized in C++ for ARL estimation.
\item Allows dynamic updating of \code{beta0_est} to improve the model's adaptability.
\item Compatible with scenarios where \code{alpha} is known or estimated.
\item Ensures stable and reliable performance in detecting changes in processes modeled with Gamma distributions.
\item Recommended values for \code{H_plus}, \code{H_delta}, \code{K_l}, \code{delay}, and \code{tau} can be found in the reference article.
}
}
}
\examples{
# Option 1: Provide parameters directly
ARL_Clplus(
  alpha = 1,
  beta = 1,
  alpha0_est = 1,  # alpha = known_alpha
  beta0_est = 1.1,   # Estimated Beta
  known_alpha = TRUE,
  beta_ratio = 2,
  H_delta = 4.2433,
  H_plus = 8.7434,
  n_I = 200,
  replicates = 100,
  K_l = 2,
  delay = 25,
  tau = 1
)

# Option 2: Use generated data
set.seed(123)
datos_faseI <- rgamma(n = 200, shape = 1, scale = 1)
alpha0_est <- mean(datos_faseI)^2 / var(datos_faseI)  # Alpha estimation
beta0_est <- mean(datos_faseI) / alpha0_est  # Beta estimation

ARL_Clplus(
  alpha = 1,
  beta = 1,
  alpha0_est = alpha0_est,
  beta0_est = beta0_est,
  known_alpha = FALSE,
  beta_ratio = 2,
  H_delta = 4.2433,
  H_plus = 8.7434,
  n_I = 200,
  replicates = 1000,
  K_l = 2,
  delay = 25,
  tau = 1
)
}
