% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/get_table_history.R
\name{get_table_history}
\alias{get_table_history}
\title{Stratify Person Table with Time Varying Co-variate}
\usage{
get_table_history(
  persondf,
  rateobj,
  batch_size = 25,
  strata = dplyr::vars(),
  historydf,
  exp,
  cutpt,
  lag = 0
)
}
\arguments{
\item{persondf}{data.frame like object containing one row per person with the required demographic information}

\item{rateobj}{a rate object created by the \code{parseRate} function}

\item{batch_size}{a number specifying how many persons to stratify at a time. Default is 25.}

\item{strata}{any additional variables contained in \code{persondf} on which to stratify.
Must be wrapped in a \code{vars()} call from \code{dplyr}.}

\item{historydf}{data.frame like object containing temporal exposure data. Each row represents a period
of time during which the values f the temporal stratifiers remain constant. Multiple rows
per \code{id} are typical.}

\item{exp}{the name of the exposure variable found within \code{historydf}}

\item{cutpt}{a numeric vector containing the cut-points to use to stratify the
calculated cumulative exposure for variable \code{exp}. Should include min and max values (typically -Inf and Inf).}

\item{lag}{An optional numeric variable indicating a lag, in years,
to be applied to exposure variables. Specified in years. Default is 0 yrs (un-lagged).}
}
\value{
A data.frame with a row for each strata containing the number of observed
deaths within each of the defined minors/outcomes (\verb{_o1}-\verb{_oxxx}) and the number of person days.
}
\description{
\code{get_table_history} reads in a data.frame/tibble containing basic demographic information
for each person of the cohort as well as a data.frame/tibble containing time varying exposure
information and stratifies the person-time and deaths into 5-year age, 5-year calendar period,
race, sex and exposure categories. See \code{Details} for information on how the person file
and history file must be formatted.
}
\details{
The person file must contain the variables:
\itemize{
\item id,
\item gender (numeric: 'M'/'F'),
\item race (numeric: 'W'/'N'),
\item dob (date),
\item pybegin (date),
\item dlo	(date),
\item rev (numeric: 5-10),
\item code (character: ICD code)
}

Additionally, a history file is read in containing daily exposures levels and the begin and
end dates of that level. The history file must contain the following variables:
\itemize{
\item id,
\item begin_dt (date),
\item end_dt (date),
\item <daily exposure levels>

}
}
\examples{
library(LTASR)
library(dplyr)
library(lubridate)

#Import example person file
person <- person_example \%>\%
  mutate(dob = as.Date(dob, format='\%m/\%d/\%Y'),
         pybegin = as.Date(pybegin, format='\%m/\%d/\%Y'),
         dlo = as.Date(dlo, format='\%m/\%d/\%Y'))

#Import example history file
history <- history_example \%>\%
  mutate(begin_dt = as.Date(begin_dt, format='\%m/\%d/\%Y'),
         end_dt = as.Date(end_dt, format='\%m/\%d/\%Y'))

#Import default rate object
rateobj <- us_119ucod_19602020

#Stratify cohort employed variable. The employed variable indicates (0/1) periods of employment
#and will be summed each day of each exposure period. Therefore, this calculates
#duration of employment in days. The cut-points used below will stratify
#by person-time with less than and greater than a year of employment (365 days of employment).
py_table <- get_table_history(persondf = person,
                              rateobj = rateobj,
                              historydf = history,
                              exp = employed,
                              cutpt = c(-Inf, 365, Inf))

#Investigate cumulative exposure values for the exposure_level variable.
#This aids in determining appropriate cut-points for stratification.
history_example \%>\%
  mutate(begin_dt = as.Date(begin_dt, format='\%m/\%d/\%Y'),
         end_dt = as.Date(end_dt, format='\%m/\%d/\%Y')) \%>\%
  mutate(dur = difftime(end_dt, begin_dt, units = 'days') \%>\%
           as.numeric() \%>\%
           `+`(1),
         cum = dur*exposure_level) \%>\%
  group_by(id) \%>\%
  summarize(cum = sum(cum))

#Stratify cohort with exposure variable. This will stratify by person-time with
#with less than and greater than 15000 unit-days of cumulative exposure.
py_table <- get_table_history(persondf = person,
                              rateobj = rateobj,
                              historydf = history,
                              exp = exposure_level,
                              cutpt = c(-Inf, 15000, Inf),
                              lag = 10)

}
