% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/runLWFB90.R
\name{run_LWFB90}
\alias{run_LWFB90}
\title{Run the LWF-Brook90 hydrologic model}
\usage{
run_LWFB90(
  options_b90,
  param_b90,
  climate,
  precip = NULL,
  soil = NULL,
  output_fun = NULL,
  rtrn_input = TRUE,
  rtrn_output = TRUE,
  chk_input = TRUE,
  run = TRUE,
  timelimit = Inf,
  verbose = FALSE,
  ...
)
}
\arguments{
\item{options_b90}{Named list of model control options. Use
\code{\link{set_optionsLWFB90}} to generate a list with default model
control options.}

\item{param_b90}{Named list of model input parameters. Use
\code{\link{set_paramLWFB90}} to generate a list with default model
parameters.}

\item{climate}{Data.frame with daily climatic data, or a function that
returns a suitable data.frame. See details for the required variables.}

\item{precip}{Data.frame with columns 'dates' and 'prec' to supply
precipitation data separately from climate data. Can be used to provide
sub-day resolution precipitation data to LWFBrook90. For each day in dates,
1 (daily resolution) to 240 values of precipitation can be provided, with
the number of values per day defined in \code{options_b90$prec_interval}.}

\item{soil}{Data.frame containing the hydraulic properties of the soil
layers. See section 'Soil parameters'}

\item{output_fun}{A function or a list of functions of the form
\code{f(x,...)}, where \code{x} is the object regularly returned by
\code{run_LWFB90}. During function evaluation, \code{x} contains model
input and selected output objects, irrespective of \code{rtrn_input} and
\code{rtrn_output}. Can be used to  aggregate output on-the-fly, and is
especially useful if the function is evaluated within a large multi-run
application, for which the output might overload the memory. (see
\code{\link{run_multi_LWFB90}} and \code{\link{run_multisite_LWFB90}}).}

\item{rtrn_input}{Logical: append \code{param_b90}, \code{options_b90}, and
daily plant properties (\code{standprop_daily}, as derived from parameters)
to the result?}

\item{rtrn_output}{Logical: return the simulation results select via
\code{output}?}

\item{chk_input}{Logical wether to check \code{param_b90},
\code{options_b90}, \code{climate}, \code{precip}, and \code{soil} for
completeness and consistency.}

\item{run}{Logical: run LWF-Brook90 or only return model input objects?
Useful to inspect the effects of options and parameters on model input.
Default is TRUE.}

\item{timelimit}{Integer to set elapsed time limits (seconds) for running LWF-Brook90.}

\item{verbose}{Logical: print messages to the console? Default is FALSE.}

\item{...}{Additional arguments passed to \code{output_fun} and/or
\code{climate}, if the latter is a function.}
}
\value{
A list containing the selected model output (if \code{rtrn_output ==
  TRUE}), the model input (if \code{rtrn_input == TRUE}, except for
  \code{climate}), and the return values of \code{output_fun} if specified.
}
\description{
Sets up the input objects for the LWF-Brook90 hydrologic model, starts the
model, and returns the selected results.
}
\section{Climate input data}{
 The \code{climate} data.frame (or function) must
 contain (return) the following variables in columns named 'dates' (Date),
 'tmax' (deg C), 'tmin' (deg C), 'tmean' (deg C), 'windspeed' (m/s), 'prec'
 (mm) , 'vappres' (kPa), and either 'globrad' ( MJ/(m²d) ) or 'sunhours' (h).
 When using \code{sunhours}, please set \code{options_b90$fornetrad =
 'sunhours'}.
}

\section{Soil input parameters}{
 Each row of \code{soil} represents one layer,
 containing the layers' boundaries and soil hydraulic parameters. The column
 names for the upper and lower layer boundaries are 'upper' and 'lower' (m,
 negative downwards). When using \code{options_b90$imodel = 'MvG'}, the
 hydraulic parameters are 'ths', 'thr', 'alpha' (1/m), 'npar', 'ksat' (mm/d)
 and 'tort'.  With \code{options_b90$imodel = 'CH'}, the parameters are
 'thsat', 'thetaf', 'psif' (kPa), 'bexp', 'kf' (mm/d), and 'wetinf'. For both
 parameterizations, the volume fraction of stones has to be named 'gravel'.
 If the \code{soil} argument is not provided, list items \code{soil_nodes}
 and \code{soil_materials} of \code{param_b90} are used for the simulation.
 These have to be set up in advance, see \code{\link{soil_to_param}}.
}

\section{Outputs}{

\tabular{llcl}{
\strong{Name} \tab \strong{Description} \tab \strong{Unit} \cr
yr \tab year \tab - \cr
mo \tab month \tab - \cr
da \tab day of month \tab - \cr
doy \tab day of year \tab - \cr
aa \tab average available energy above canopy \tab W/m2 \cr
adef \tab available water deficit in root zone \tab mm \cr
asubs \tab average available energy below canopy \tab W/m2 \cr
awat \tab total available soil water in layers with roots between -6.18 kPa and \code{param_b90$psicr} \tab mm \cr
balerr \tab error in water balance (daily value, output at the day's last precipitation interval) \tab mm \cr
byfl \tab total bypass flow \tab mm/d \cr
dsfl \tab downslope flow \tab mm/d \cr
evap \tab evapotranspiration \tab mm/d \cr
flow \tab total streamflow \tab mm/d \cr
gwat \tab groundwater storage below soil layers \tab mm \cr
gwfl \tab groundwater flow \tab mm/d \cr
intr \tab intercepted rain \tab mm \cr
ints \tab intercepted snow \tab mm \cr
irvp \tab evaporation of intercepted rain \tab mm/d \cr
isvp \tab evaporation of intercepted snow \tab mm/d \cr
lngnet \tab net longwave radiation \tab W/m2 \cr
nits \tab total number of iterations \tab - \cr
pint \tab potential interception for a canopy always wet \tab mm/d \cr
pslvp \tab potential soil evaporation \tab mm/d \cr
ptran \tab potential transpiration \tab mm/d \cr
relawat \tab relative available soil water in layers with roots \tab - \cr
rfal \tab rainfall \tab mm/d \cr
rint \tab rain interception catch rate \tab mm/d \cr
rnet \tab rainfall to soil surface \tab mm/d \cr
rsno \tab rain on snow \tab mm/d \cr
rthr \tab rain throughfall rate \tab mm/d \cr
sthr \tab snow throughfall rate \tab mm/d \cr
safrac \tab source area fraction \tab - \cr
seep \tab seepage loss \tab mm/d \cr
sfal \tab snowfall \tab mm/d \cr
sint \tab snow interception catch rate \tab mm/d \cr
slfl \tab input to soil surface \tab mm/d \cr
slvp \tab evaporation rate from soil \tab mm/d \cr
slrad \tab average solar radiation on slope over daytime \tab W/m2 \cr
solnet \tab net solar radiation on slope over daytime \tab W/m2 \cr
smlt \tab snowmelt \tab mm/d \cr
snow \tab snowpack water equivalent \tab mm \cr
snvp \tab evaporation from snowpack \tab mm/d \cr
srfl \tab source area flow \tab mm/d \cr
stres \tab tran / ptran (daily value, output at the day's last precipitation interval) \tab - \cr
swat \tab total soil water in all layers\tab mm \cr
tran \tab transpiration \tab mm/d \cr
vrfln \tab vertical matrix drainage from lowest layer \tab mm/d \cr
}
}

\section{Layer outputs}{

\tabular{llcl}{
\strong{Name} \tab \strong{Description} \tab \strong{Unit} \cr
yr \tab year \tab - \cr
mo \tab month \tab - \cr
da \tab day of month \tab - \cr
doy \tab day of year \tab - \cr
nl \tab index of soil layer \tab \cr
swati \tab soil water volume in layer \tab mm \cr
theta \tab water content of soil layer, mm water / mm soil matrix \tab - \cr
wetnes \tab wetness of soil layer, fraction of saturation \tab - \cr
psimi \tab matric soil water potential for soil layer \tab kPa \cr
infl \tab infiltration to soil water in soil layer \tab mm/d \cr
byfl \tab bypass flow from soil layer \tab mm/d \cr
tran \tab transpiration from soil layer \tab mm/d \cr
vrfl \tab vertical matrix drainage from soil layer \tab mm/d \cr
dsfl \tab downslope drainage from layer \tab mm/d \cr
ntfl \tab net flow into soil layer \tab mm/d \cr
}
}

\examples{
# Set up lists containing model control options and model parameters:
param_b90 <- set_paramLWFB90()
options_b90 <- set_optionsLWFB90()

# Set start and end Dates for the simulation
options_b90$startdate <- as.Date("2003-06-01")
options_b90$enddate <- as.Date("2003-06-30")

# Derive soil hydraulic properties from soil physical properties
# using pedotransfer functions
soil <- cbind(slb1_soil, hydpar_wessolek_tab(slb1_soil$texture))

# Run LWF-Brook90
b90.result <- run_LWFB90(options_b90 = options_b90,
                        param_b90 = param_b90,
                        climate = slb1_meteo,
                        soil = soil)

# use a function to be performed on the output:
# aggregate soil water storage down to a specific layer
agg_swat <- function(x, layer) {
  out <- aggregate(swati~yr+doy,
                   x$SWATDAY.ASC,
                   FUN = sum,
                   subset = nl <= layer)
  out[order(out$yr, out$doy),]}

# run model without returning the selected output.
b90.aggswat <- run_LWFB90(options_b90 = options_b90,
                         param_b90 = param_b90,
                         climate = slb1_meteo,
                         soil = soil,
                         output_fun = list(swat = agg_swat),
                         rtrn_output = FALSE,
                         layer = 10)  # passed to output_fun
str(b90.aggswat$output_fun$swat)
}
