#' @title Delete repeated channel data generated by LC-MS
#' @description
#'channel.delete select one of the channel with largest mean value out of multiple replicated LC-MS cahnnels,
#'    and it returns an edited data frame with no repeated channel and simplified row name.
#'
#'
#' @param data Primary lipidomic data in .csv  format.
#' @param delete.pattern Pattern of characters that needs to be removed.
#'
#'
#' @return Edited data with no repeat channels and simplified row name.
#' @export
#'
#' @examples
#' x1 <- c(1,1,1,1)
#' x2 <- c(1.1,1.2,1.3,1.4)
#' x3 <- c(1,1,1,1)
#' x4 <- c(2,2,2,2)
#' rowNames <- c('PC(14:0/14:1)+AcO_1','PC(14:0/14:1)+AcO_2','PC(16:0/16:1)_1','PC(16:0/16:1)_2')
#' ExampleData <- data.frame(x1,x2,x3,x4, row.names =rowNames)
#' replicate.delete(ExampleData,delete.pattern= c('_\\d','(\\+)AcO'))
#' @export
channel.delete <- function(data, delete.pattern= c('_\\d','(\\+)AcO','_n')) {
  data<-na.omit(data)
  lipid<-rownames(data)
  for (pattern in delete.pattern) {
    lipid<-gsub(pattern,'',lipid)
  }
  rn<-lipid
  data$Sample.NAme<-rn
  x<-duplicated(rn)
  rre<-unique(rn[x])
  ids<-c()
  for (everyrep in rre) {
    current_lipid<-data[which(data$Sample.NAme==everyrep),]
    current_lipid<-t(current_lipid[,colnames(current_lipid) != "Sample.NAme"])
    current_lipid<-data.frame(current_lipid,stringsAsFactors = FALSE,check.names = FALSE)
    current_lipid2<-as.data.frame(lapply(current_lipid,as.numeric),row.names = rownames(current_lipid),check.names = FALSE)
    avg<-apply(current_lipid2[which(rownames(current_lipid2)!='Sample.NAme'),], 2, 'mean')
    Max<-names(which(avg==max(avg)))
    id<-ifelse(length(Max)>1,Max[1],Max)
    ids<-c(ids,setdiff(colnames(current_lipid),id))
  }
  data_NR<-data[setdiff(rownames(data),ids),]
  rownames(data_NR)<-data_NR$Sample.NAme
  data_NR <- data_NR[,colnames(data_NR) != "Sample.NAme"]
  data_NR <- na.omit(data_NR)
  return(data_NR)
}

#' @title importer()
#'
#' @description
#' Internal function to import data in lipidomicR, in order to unify data format.
#'
#' @param path Path of file loaded. The file should be in '.csv' format
#' @param header Logical. Whether to use the first row as header.
#' @param sep Character. The seperator of the file.
#'
#' @return A dataframe, with the first row set as header and the first column set as row name.
#' The data is unified to numeric.
#'
#' @export
importer <- function(path, header,sep) {
 data<-read.csv(path,header = header,row.names = 1,sep = sep)
 data[is.na(data)] <- 0
 data[data=='N/A' | data == 'na' | data == 'NA'] <- NA
 data2<- as.data.frame(lapply(data, as.numeric), row.names = rownames(data),check.names = FALSE)
 return(data2)
}


#' @title lEr
#' @description
#' Internal function to extract label data in lipidomicR
#'
#' @param data Data frame. Row lipidomics data. Should be imported with importer().
#'
#' @param Inlabel Vector. Name of Internal label.
#'
#' @param relative_as_default Logical, default as TRUE for automatically searching for internal label data.
#'
#' @param relative.mannual Vector, the exact channel name.
#'
#' @return Data of internal label.
lEr  <- function(data,Inlabel=c('PE(17:0/17:0)','PC(17:0/17:0)','PS(14:0/14:0)'),
                 relative_as_default = TRUE, relative.mannual= NULL) {
  if (relative_as_default == TRUE) {
    data_cleaned <- channel.delete(data)
    lb<-data_cleaned[Inlabel,]
    lb<-as.data.frame(lapply(lb,as.numeric),row.names = rownames(lb),check.names = FALSE)
  }else {
    lb <- data[relative.mannual,]
  }
  return(lb)
}


#' @title lEa()
#'
#' @description
#' Internal function to extract label data in lipidomicR
#'
#' @param data Data frame. Row lipidomics data. Should be imported with importer().
#' @param Inchannel Vector. Exact channel name for radio label data.
#' @param qc Vector. Column of quantity control. Default as c(1:5).
#' @param qc.omit Vector. Column of quantity control that need to be omitted, default as 1.
#'
#' @return Data of internal label.
lEa<- function(data,Inchannel = c('PC(15:0/18:1(d7))+AcO_1',
                                  'PE(15:0/18:1(d7))_2',
                                  'PS(15:0/18:1(d7))_2',
                                  'PG(15:0/18:1(d7))_2',
                                  'PI(15:0/18:1(d7))_2',
                                  'PA(15:0/18:1(d7))_2',
                                  'LPC18:1(d7)+AcO',
                                  'LPE18:1(d7)'), qc=1:5,qc.omit=1){
  qc <- setdiff(qc,qc.omit)
  lb<-data[Inchannel,qc]
  lb<-as.data.frame(lapply(lb,as.numeric),row.names = rownames(lb),check.names = FALSE)
  return(lb)
}


#' @title nor.absolute()
#' @description
#' A function to calculate parameters for absolute normalization.
#' @param data Data frame, row lipidomics data.
#' @param radio.data Data frame. Characteristic of the radio label data.
#' The row name must be the exact channel name of the label. Molecular mass should be
#' provided in a column named "Mass". Concentration should be provided in a column named "Concentration(mg/ml)".
#' @param qc Vector. Column of quantity control. Default as c(1:5).
#' @param qc.omit Vector. Column of quantity control that need to be omitted, default as 1.
#' @return Parameters for absolute normalization
nor.absolute  <- function(data, radio.data=NULL, qc = 1:5, qc.omit=1) {
  qc <- setdiff(qc,qc.omit)
  rad<-radio.data
  cM<-radio.data$Concentration.mg.ml./radio.data$Mass
  avg_rad<- apply(lEa(data, Inchannel = rownames(radio.data),qc=qc),1,'mean')
  unit<-cM/avg_rad
  rad$Concentration.mol.l <- cM
  rad$mol.peak_intensity <- unit
  return(rad)
}

#' @title nor.relative()
#' @description
#' A function to calculate parameters for relative normalization.
#'
#' @param data Data frame, row lipidomics data.
#' @param Inlabel Inlabel Vector. Name of Internal label. Default as c('PE(17:0/17:0)','PC(17:0/17:0)','PS(14:0/14:0)')
#' @param qc Numeric value indicating the columns of quality control input.
#' @param relative_as_default Logical, default as TRUE for automatically searching for internal label data.
#' @param relative.mannual Vector, the exact channel name (if you want to define the channel of internal label mannually).
#'
#' @return Parameters for relative normalization
nor.relative <- function(data,
                         Inlabel=c('PE(17:0/17:0)','PC(17:0/17:0)','PS(14:0/14:0)'),
                         qc = 1:5,
                         relative_as_default = TRUE,
                         relative.mannual = NULL) {
  rel <- lEr(data, Inlabel = Inlabel,
             relative_as_default = relative_as_default,relative.mannual = relative.mannual)
  rel2 <- rel[,setdiff(1:ncol(rel),qc)]
  rel_prop <- apply(rel[,qc], 1, 'mean')/sum(apply(rel[,qc], 1, 'mean'))
  idx<-0
  for (i in 1:length(rel_prop)) {
    ind<- rel2[i,]*rel_prop[i]
    idx<-idx+ind
  }
  rownames(idx) <- c('normalization_index')
  return(idx)
}

#' @title noridx()
#'
#' @description
#' An integrated function that call nor.relative() and nor.absolute(), for simplifying.
#'
#' @usage
#' noridx(data, normalization.mode='both', radio.data=NULL, qc= 1:5 ,qc.omit = 1,
#' Inlabel=c('PE(17:0/17:0)','PC(17:0/17:0)','PS(14:0/14:0)'),
#' relative_as_default = TRUE,relative.mannual = NULL)
#'
#' @param data Data frame, row lipidomics data.
#' @param normalization.mode Character. "absolute" tp output absolute normalization index
#' 'relative' to output relative normalization index. "both" to output both of them. Default as "both".
#' @param radio.data Data frame. Characteristic of the radio label data.
#' The row name must be the exact channel name of the label. Molecular mass should be
#' provided in a column named "Mass". Concentration should be provided in a column named "Concentration(mg/ml)".
#' @param qc Vector. Column of quantity control. Default as c(1:5).
#' @param qc.omit Vector. Column of quantity control that need to be omitted, default as 1.
#' @param Inlabel Inlabel Vector. Name of Internal label. Default as c('PE(17:0/17:0)','PC(17:0/17:0)','PS(14:0/14:0)')
#' @param relative_as_default Logical, default as TRUE for automatically searching for internal label data.
#' @param relative.mannual Vector, the exact channel name (if you want to define the channel of internal label mannually).
#'
#' @return
#' 1. normalization.mode='both'. A list of data frames of normalization indexes of the two modes.
#' 2. normalization.mode='absolute' or 'relative'. A data frame of the respective normlaization index.
#' @export
noridx <- function(data, normalization.mode='both', radio.data=NULL,
                           qc= 1:5, qc.omit = 1,
                           Inlabel=c('PE(17:0/17:0)','PC(17:0/17:0)','PS(14:0/14:0)'),
                           relative_as_default = TRUE,
                           relative.mannual = NULL) {
  if (normalization.mode == 'absolute') {
    return(list(absolute=nor.absolute(data = data, radio.data = radio.data, qc = qc, qc.omit=qc.omit)))}
  else if (normalization.mode =='relative'){
    return(list(relative=nor.relative(data = data,Inlabel = Inlabel, qc = qc, relative_as_default = relative_as_default,
                        relative.mannual = relative.mannual)))}
  else if (normalization.mode == "both") {
    x1 <- nor.absolute(data = data, radio.data = radio.data, qc = qc, qc.omit=qc.omit)
    x2 <- nor.relative(data = data,Inlabel = Inlabel,qc = qc, relative_as_default = relative_as_default,
                       relative.mannual = relative.mannual)
    return(list(absolute=x1,relative=x2))
  }else {
    warning('Error: unsupported normalization mode\n')
  }
}

#' @title absolute.calculator()
#'
#' @description
#' Internal function of lipidmicR::normalization_calculator()
#'
#' @param data Row data of lipidomics
#' @param absolute.dataset Data frame. Normalization index.
#' @param qc Numeric vector. The column of quality control.
#'
#' @return
#' Return a data frame of absolute normalized lipidomic data.
absolute.calculator <- function(data,absolute.dataset,qc=1:5){
  data<-data[,-qc]
  sep_data<-sepclass(data,'lipid')
  sep_index<-sepclass(absolute.dataset,'lipid')
  cnx<-data.frame()
  for (lipid in sep_index$lipid_type){
    cdt<-subset(data, subset = sep_data$`lipid_type`==lipid)
    cix<-sep_index[sep_index$`lipid_type`==lipid, 'mol.peak_intensity']
    cnx2 <- cdt*cix
    cnx<- rbind(cnx,cnx2)
  }
  return(cnx)
  }

#' @title relative.calculator()
#'
#' @description
#' Internal function of lipidmicR::normalization_calculator()
#'
#' @param data Row data of lipidomics
#' @param relative.dataset Data frame. Normalization index.
#' @param qc Numeric vector. The column of quality control.
#'
#' @return
#' Return a data frame of relative normalized lipidomic data.
relative_calculator <- function(data, relative.dataset,qc=1:5) {
  data<-data[,-c(qc)]
  cno<-data.frame(matrix(nrow = nrow(data),ncol = ncol(data)))
  for (i in 1:ncol(relative.dataset)) {
    cdt<-data[,i]
    cix<- relative.dataset[,i]
    cno2<-cdt/cix
    cno[,i]<-cno2
  }
  rownames(cno)<-rownames(data)
  colnames(cno)<-colnames(data)
  return(cno)
  }

#' @title percent.calculator()
#' @description
#' A function to calculate the proportion of each lipid content.
#'
#' @param data Data frame. The row lipidomic data.
#' @param absolute.dataset Data frame. Normalization parameters for absolute
#' normalization. suggested to be generated by noridx().
#' @param internal.label Vector. Contains either the row channel name or lipid name of any external lipid labels.
#' @param qc Numeric. The column of quality control.
#'
#' @return
#' Return a data frame of normalized lipidomic data in the percentage of lipid content.
percent.calculator <- function(data, absolute.dataset, internal.label, qc=1:5) {
  data <- data[!internal.label %in% data,]
  data <- na.omit(data)
  abs<-absolute.calculator(data = data,absolute.dataset = absolute.dataset,qc = qc)
  abs_cleaned<- channel.delete(abs)
  abs_cleaned <- abs_cleaned[!internal.label %in% abs_cleaned,]
  sum <- colSums(abs_cleaned)
  for (i in 1:ncol(abs_cleaned)) {
    abs_cleaned[,i] <- abs_cleaned[,i]/sum[i]
  }
  return(abs_cleaned)
}

#' @title normalization_calculator()
#'
#' @description
#' A function uses normalization parameters to calculate normalized lipidomic data.
#'
#' @usage
#' normalization_calculator(data, normalization.mode, normalization.index, qc=1:5)
#'
#' @param data Data frame. Row lipidomic data.
#' @param normalization.mode Vector. 'relative' for relative normalization. 'absolute' for absolute normalization.
#' @param normalization.index Data frame. normalization parameters, Suggested to be calculated by noridx.ouput()
#' @param qc Numeric vector. column of quality control. Default as 1:5.
#'
#' @return
#' A data frame of normalized data if either 'relative' or 'absolute' mode is used. A list if both of them are used.
#'
#' @export
normalization_calculator <- function(data,
                                     normalization.mode,
                                     normalization.index,
                                     qc=1:5){
  ncl <- list()
  if ('absolute' %in% normalization.mode) {
    outputa <- absolute.calculator(data = data, absolute.dataset = normalization.index$absolute, qc= qc)
    ncl$absolute <- outputa
  }
  if ('relative' %in% normalization.mode) {
    outputr <- relative_calculator(data = data,relative.dataset = normalization.index$relative, qc = qc)
    ncl$relative <- outputr
  }
  if ('percent' %in% normalization.mode) {
    outputp <- percent.calculator(data = data,absolute.dataset = normalization.index$absolute,
                                  internal.label =
                                    c(rownames(normalization.index$absolute),c('PE(17:0/17:0)','PC(17:0/17:0)','PS(14:0/14:0)')))
    ncl$percent <- outputp
    }
  return(ncl)
}

#' @title paramWizard()
#'
#' @description
#' A user-friendly interaction interface to input parameters for cleanXpert()
#'
#' @details
#' In this function, you should provide the following information:
#' 1. Path of row data.
#' 2. Path of the paramters for your isotope label.
#' 3. Whether default internal label should be used.
#' 4. (optional) Name of internal label.
#' 5. qc column and omitted qc column.
#' 6. Whether replicate with the largest shifting should be removed.
#' 7. (optional) The number of repeats of each group and repeats that should be removed within each group.
#' The parameters could either be import via key board typing or via external .txt file.
#' For first usage, key board typing input is suggested, and a .txt file will be automatically generated.
#'
#' @return
#' A list of parameters.
paramWizard <- function(){
    Data.path <- readline('Please type in the file path of your rowdata: ')
    RadioLabel.path <- readline('Please type in the file path of your parameters of radiolabelling: ')
    Normalization_as_default <- menu(c('(Default: use built-in labels) PE(17:0/17:0), PC(17:0/17:0), PS(14:0/14:0)', "Mannual"),title = "Please choose the label for normalization")
    if (Normalization_as_default == 1) {
      Normalization.label <- c('PE(17:0/17:0)','PC(17:0/17:0)','PS(14:0/14:0)')
    }else{
      Normalization.label <- scan(what = 'character')
    }
    qc <- readline('Please define the column of qc. E.g. type in 1:5 means qc data is presented in column 1 to 5')
    qc.omit <- as.numeric(readline('Please type in the column of qc that need to be omitted (type in 0 to skip this step).'))
    Replicate.delete <- menu(c('Yes','No'),title = 'Do you want to remove the data point with the largest shifting?')

    if (Replicate.delete == 1) {
      Repeats <- readline('How many replicates are there within one group?')
      Remove <- readline('How many points needed removal within one group?')
    }else {
      Repeats <- ''
      Remove <- ''
    }
    index <- list(Data.path = Data.path,
                  RadioLabel.path = RadioLabel.path,
                  Normalization_as_default = Normalization_as_default,
                  Normalization.label = Normalization.label,
                  qc = qc, qc.omit = qc.omit,
                  Replicate.delete = Replicate.delete, Repeats = Repeats, Remove = Remove)
  return(index)
}

#' @title cleanXpert()
#'
#' @description
#' Integrated function for lipidomic data cleaning, processing and normalization.
#'
#' @usage cleanXpert(interactive = FALSE, parameters = NULL)
#'
#' @param interactive Logical input indicating whether to use the paramWidzard interface.
#' @param parameters List input including the parameters for data cleaning. Only useful when interactive == FALSE.
#' @return
#' A list containing cleaned data with or without annotation.
#'
#' @export
cleanXpert  <- function(interactive = FALSE, parameters = NULL){
  if (interactive == TRUE){
  parameters <- paramWizard()
  }else{}
  rowdata<- importer(parameters$Data.path)
  radiolabel <- importer(parameters$RadioLabel.path)
  nor.label <- parameters$Normalization.label
  qc<-parameters$qc
  qc <- c(as.numeric(strsplit(qc,split = ':')[[1]][1]):
            as.numeric(strsplit(qc,split = ':')[[1]][2]))
  qc.omit<- as.numeric(parameters$qc.omit)
  Repdel <- ifelse(parameters$Replicate.delete==1, TRUE, FALSE)
  Repeats <- as.numeric(parameters$Repeats)
  Remove <- as.numeric(parameters$Remove)
  radiolabel.simple <- rownames(channel.delete(radiolabel))

  nix <- noridx(data = rowdata,radio.data = radiolabel,
                       qc = qc,qc.omit = qc.omit,Inlabel = nor.label)
  ndt <- normalization_calculator(data = rowdata,normalization.index = nix,qc = qc)
  abs <- channel.delete(ndt$absolute)
  abs <- abs[!rownames(abs) %in% c(radiolabel.simple, nor.label),]
  rel <- channel.delete(ndt$relative)
  rel <- rel[!rownames(rel) %in% c(radiolabel.simple, nor.label),]
  per <- ndt$percent
  row_data <- channel.delete(rowdata)
  row_data <- row_data[!rownames(row_data) %in% c(radiolabel.simple, nor.label),]

  if (Repdel == TRUE) {
    abs <- delRep(abs,n = Repeats,m = Remove)
    rel <- delRep(abs,n = Repeats,m = Remove)
    row_data <- delRep(row_data, n = Repeats, m = Remove)
    per <- delRep(per, n = Repeats, m = Remove)
    abs_add <- sepclass(data = abs, pattern = 'all')
    rel_add <- sepclass(data = rel, pattern = 'all')
    row_add <- sepclass(data = row_data, pattern = 'all')
    per_add <- sepclass(data = per, pattern = 'all')
    cleaned_list <- list(absolute_Channeldel_low_repeats = abs,
                         relative_Channeldel_low_repeats = rel,
                         row_Channeldel_low_repeats = row_data,
                         per_channeldel_low_repeats = per,
                         absolute_Channeldel_low_repeats_add = abs_add,
                         relative_Channeldel_low_repeats_add = rel_add,
                         row_Channeldel_low_repeats_add = row_add,
                         per_Channeldel_low_repeats_add = per_add)
  }else{
  abs_add <- sepclass(data = abs, pattern = 'all')
  rel_add <- sepclass(data = rel, pattern = 'all')
  row_add <- sepclass(data = row_data, pattern = 'all')
  per_add <- sepclass(data = per, pattern = 'all')
  cleaned_list <- list(absolute_Channeldel_full_repeats = abs,
                       relative_Channeldel_full_repeats = rel,
                       row_Channeldel_full_repeats = row_data,
                       per_channeldel_full_repeats = per,
                       absolute_Channeldel_full_repeats_add = abs_add,
                       relative_Channeldel_full_repeats_add = rel_add,
                       row_Channeldel_full_repeats_add = row_add,
                       per_Channeldel_full_repeats_add = per_add)
  }
  message('\nData analysis finished!\n')
  return(cleaned_list)
}
