\name{calc_CosmicDoseRate}
\alias{calc_CosmicDoseRate}
\title{Calculate the cosmic dose rate}
\description{This function calculates the cosmic dose rate taking into account the
soft- and hard-component of the cosmic ray flux and allows
corrections for geomagnetic latitude, altitude above sea-level and 
geomagnetic field changes.}
\usage{calc_CosmicDoseRate(depth, density, latitude, longitude, altitude, 
    corr.fieldChanges = FALSE, est.age = NA, half.depth = FALSE, 
    error = 10)}
\arguments{
  \item{depth}{\code{\link{numeric}} (\bold{required}): depth of overburden (m). 
For more than one absorber use \cr
\code{c(depth_1, depth_2, ..., depth_n)}}
  \item{density}{\code{\link{numeric}} (\bold{required}): average overburden 
density (g/cm^3). For more than one absorber use \cr
\code{c(density_1, density_2, ..., density_n)} }
  \item{latitude}{\code{\link{numeric}} (\bold{required}): latitude (decimal 
degree), N positive}
  \item{longitude}{\code{\link{numeric}} (\bold{required}): longitude (decimal 
degree), E positive}
  \item{altitude}{\code{\link{numeric}} (\bold{required}): altitude (m above 
sea-level)}
  \item{corr.fieldChanges}{\code{\link{logical}} (with default): correct for 
geomagnetic field changes after Prescott & Hutton (1994). Apply only when
justified by the data.}
  \item{est.age}{\code{\link{numeric}} (with default): estimated age range (ka)
for geomagnetic field change correction (0-80 ka allowed)}
  \item{half.depth}{\code{\link{logical}} (with default): How to overcome with 
varying overburden thickness. If \code{TRUE} only half the depth is used 
for calculation. Apply only when justified, i.e. when a constant 
sedimentation rate can safely be assumed.}
  \item{error}{\code{\link{numeric}} (with default): general error (percentage) 
to be implemented on corrected cosmic dose rate estimate}
}
\details{This function calculates the total cosmic dose rate considering both the 
soft- and hard-component of the cosmic ray flux.\cr

\bold{Internal calculation steps}

(1) Calculate total depth of all absorber in hg/cm^2 
(1 hg/cm^2 = 100 g/cm^2)

\deqn{absorber = depth_1*density_1 + depth_2*density_2 + ... + depth_n*
density_n}

(2) If \code{half.depth = TRUE}

\deqn{absorber = absorber/2}

(3) Calculate cosmic dose rate at sea-level and 55 deg. latitude

a) If absorber is > 167 g/cm^2 (only hard-component; Allkofer et al. 
1975): apply equation given by Prescott & Hutton (1994) (c.f. Barbouti & 
Rastin 1983)

\deqn{D0 = C/(((absorber+d)^\alpha+a)*(absober+H))*exp(-B*absorber)}

b) If absorber is < 167 g/cm^2 (soft- and hard-component): derive D0 
from Fig. 1 in Prescott & Hutton (1988).

(4) Calculate geomagnetic latitude (Prescott & Stephan 1982, Prescott & 
Hutton 1994)

\deqn{\lambda = arcsin(0.203*cos(latitude)*cos(longitude-291)+0.979*
sin(latitude))}

(5) Apply correction for geomagnetic latitude and altitude above 
sea-level. Values for F, J and H were read from Fig. 3 shown in 
Prescott & Stephan (1982) and fitted with 3-degree polynomials for 
lambda < 35 degree and a linear fit for lambda > 35 degree.

\deqn{Dc = D0*(F+J*exp((altitude/1000)/H))}

(6) Optional: Apply correction for geomagnetic field changes in the last 
0-80 ka (Prescott & Hutton 1994). Correction and altitude factors are 
given in Table 1 and Fig. 1 in Prescott & Hutton (1994). Values for 
altitude factor were fitted with a 2-degree polynomial. The altitude 
factor is operated on the decimal part of the correction factor.

\deqn{Dc' = Dc*correctionFactor}

\bold{Usage of \code{depth} and \code{density}}

(1) If only one value for depth and density is provided, the cosmic dose 
rate is calculated for exactly one sample and one absorber as overburden 
(i.e. \code{depth*density}).

(2) In some cases it might be useful to calculate the cosmic dose rate 
for a sample that is overlain by more than one absorber, e.g. in a 
profile with soil layers of different thickness and a distinct difference
in density. This can be calculated by providing a matching number of 
values for \code{depth} and \code{density} (e.g. \code{depth = c(1, 2), 
density = c(1.7, 2.4)})  

(3) Another possibility is to calculate the cosmic dose rate for more 
than one sample of the same profile. This is done by providing more than
one values for \code{depth} and only one for \code{density}. For example,
\code{depth = c(1, 2, 3), density = 1.7} will calculate the cosmic dose 
rate for three samples in 1, 2 and 3 m depth in a sediment of density 
1.7 g/cm^3.}
\value{Returns terminal output. In addition a list is returned containing 
the following element:

\item{results}{data frame with results of cosmic dose rate calculation.}}
\references{Allkofer, O.C., Carstensen, K., Dau, W.D., Jokisch, H., 1975. Letter to
the editor. The absolute cosmic ray flux at sea level. Journal of 
Physics G: Nuclear and Particle Physics, 1, pp. L51-L52. \cr\cr
Barbouti, A.I., Rastin, B.C., 1983. A study of the absolute intensity 
of muons at sea level and under various thicknesses of absorber. Journal 
of Physics G: Nuclear and Particle Physics, 9, pp. 1577-1595. \cr\cr
Crookes, J.N., Rastin, B.C., 1972. An investigation of the absolute 
intensity of muons at sea-level. Nuclear Physics B, 39, pp. 493-508. 
\cr\cr
Gruen, R., 2009. The "AGE" program for the calculation of luminescence 
age estimates. Ancient TL, 27, pp. 45-46. \cr\cr
Prescott, J.R., Hutton, J.T., 1988. Cosmic ray and gamma ray dosimetry
for TL and ESR. Nuclear Tracks and Radiation Measurements, 14, pp. \cr\cr
223-227.
Prescott, J.R., Hutton, J.T., 1994. Cosmic ray contributions to dose 
rates for luminescence and ESR dating: large depths and long-term time
variations. Radiation Measurements, 23, pp. 497-500. \cr\cr
Prescott, J.R., Stephan, L.G., 1982. The contribution of cosmic radiation
to the environmental dose for thermoluminescence dating. Latitude, 
altitude and depth dependences. PACT, 6, pp. 17-25.}
\author{Christoph Burow, University of Cologne (Germany) \cr
R Luminescence Package Team}
\note{Despite its universal use the equation to calculate the cosmic dose rate
provided by Prescott & Hutton (1994) is falsely stated to be valid from
the surface to 10^4 hg/cm^2 of standard rock. The original expression by 
Barbouti & Rastin (1983) only considers the muon flux 
(i.e. hard-component) and is by their own definition only valid for 
depths between 10-10^4 hg/cm^2.

Thus, for near-surface samples (i.e. for depths < 167 g/cm^2) the 
equation of Prescott & Hutton (1994) underestimates the total cosmic 
dose rate, as it neglects the influence of the soft-component of the 
cosmic ray flux. For samples at zero depth and at sea-level the
underestimation can be as large as ~0.1 Gy/ka. In a previous article, 
Prescott & Hutton (1988) give another approximation of Barbouti & Rastins
equation in the form of

\deqn{D = 0.21*exp(-0.070*absorber+0.0005*absorber^2)}

which is valid for depths between 150-5000 g/cm^2. For shallower depths 
(< 150 g/cm^2) they provided a graph (Fig. 1) from which the dose rate 
can be read.

As a result, this function employs the equation of Prescott & Hutton
(1994) only for depths > 167 g/cm^2, i.e. only for the hard-component 
of the cosmic ray flux. Cosmic dose rate values for depths < 167 g/cm^2
were obtained from the "AGE" programm (Gruen 2009) and fitted with a 
6-degree polynomial curve (and hence reproduces the graph shown in
Prescott & Hutton 1988). However, these values assume an average 
overburden density of 2 g/cm^3. 

It is currently not possible to obtain more precise cosmic dose rate
values for near-surface samples as there is no equation known to the
author of this function at the time of writing.}


\seealso{\code{\link{BaseDataSet.CosmicDoseRate}}}
\examples{
##(1) calculate cosmic dose rate (one absorber)
calc_CosmicDoseRate(depth = 2.78, density = 1.7,
                    latitude = 38.06451, longitude = 1.49646, 
                    altitude = 364, error = 10)

##(2a) calculate cosmic dose rate (two absorber)
calc_CosmicDoseRate(depth = c(5.0, 2.78), density = c(2.65, 1.7),
                    latitude = 38.06451, longitude = 1.49646, 
                    altitude = 364, error = 10)

##(2b) calculate cosmic dose rate (two absorber) and
##correct for geomagnetic field changes
calc_CosmicDoseRate(depth = c(5.0, 2.78), density = c(2.65, 1.7),
                    latitude = 12.04332, longitude = 4.43243, 
                    altitude = 364, corr.fieldChanges = TRUE,
                    est.age = 67, error = 15)


##(3) calculate cosmic dose rate and export results to .csv file
#calculate cosmic dose rate and save to variable
results<- calc_CosmicDoseRate(depth = 2.78, density = 1.7,
                              latitude = 38.06451, longitude = 1.49646, 
                              altitude = 364, error = 10)$results

#export results to .csv file - uncomment for usage
#write.csv(results, file = "c:/users/public/results.csv")

##(4) calculate cosmic dose rate for 6 samples from the same profile
##    and save to .csv file
#calculate cosmic dose rate and save to variable
results<- calc_CosmicDoseRate(depth = c(0.1, 0.5 , 2.1, 2.7, 4.2, 6.3), 
                              density = 1.7, latitude = 38.06451, 
                              longitude = 1.49646, altitude = 364, 
                              error = 10)$results

#export results to .csv file - uncomment for usage
#write.csv(results, file = "c:/users/public/results_profile.csv")

}
