% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calc_FadingCorr.R
\name{calc_FadingCorr}
\alias{calc_FadingCorr}
\title{Apply a fading correction according to Huntley & Lamothe (2001) for a given
g-value and a given tc}
\usage{
calc_FadingCorr(age.faded, g_value, tc = NULL, tc.g_value = tc,
  n.MC = 10000, seed = NULL, txtProgressBar = TRUE, verbose = TRUE)
}
\arguments{
\item{age.faded}{\code{\link{numeric}} \code{\link{vector}} (\bold{required}): uncorrected
age with error in ka (see example)}

\item{g_value}{\code{\link{vector}} (\bold{required}): g-value and error obtained
from separate fading measurements (see example). Alternatively an \code{\linkS4class{RLum.Results}} object
can be provided produced by the function \code{analyse_FadingMeasurement}, in this case tc is set
automatically}

\item{tc}{\code{\link{numeric}} (\bold{required}): time in seconds between
irradiation and the prompt measurement (cf. Huntley & Lamothe 2001). Argument will be ignored
if \code{g_value} was an \code{RLum.Results} object}

\item{tc.g_value}{\code{\link{numeric}} (with default): the time in seconds between irradiation
and the prompt measurement used for estimating the g-value. If the g-value was normalised
to, e.g., 2 days, this time in seconds (i.e., 172800) should be given here. If nothing is provided
the time is set to tc, which is usual case for g-values obtained using the SAR method and g-values
that had been not normalised to 2 days.}

\item{n.MC}{\code{\link{integer}} (with default): number of Monte Carlo
simulation runs for error estimation. If \code{n.MC = 'auto'} is used the function
tries to find a 'stable' error for the age. Note: This may take a while!}

\item{seed}{\code{\link{integer}} (optional): sets the seed for the random number generator
in R using \code{\link{set.seed}}}

\item{txtProgressBar}{\link{logical} (with default): enables or disables
\code{\link{txtProgressBar}}}

\item{verbose}{\code{\link{logical}} (with default): enables or disables terminal output}
}
\value{
Returns an S4 object of type \code{\linkS4class{RLum.Results}}.\cr

Slot: \bold{@data}\cr
\tabular{lll}{
\bold{Object} \tab \bold{Type} \tab \bold{Comment}\cr
 \code{age.corr} \tab \code{data.frame} \tab Corrected age \cr
 \code{age.corr.MC} \tab \code{numeric} \tab MC simulation results with all possible ages from
 that simulation\cr
}

Slot: \bold{@info}\cr

\tabular{lll}{
\bold{Object} \tab \bold{Type} \tab \bold{Comment}\cr
 \code{info} \tab \code{character} \tab the original function call

}
}
\description{
This function solves the equation used for correcting the fading affected age
including the error for a given g-value according to Huntley & Lamothe (2001).
}
\details{
As the g-value sligthly depends on the time between irradiation and the prompt measurement,
this is tc, always a tc value needs to be provided. If the g-value was normalised to a distinct
time or evaluated with a different tc value (e.g., external irradiation), also the tc value
for the g-value needs to be provided (argument \code{tc.g_value} and then the g-value is recalcualted
to tc of the measurement used for estimating the age applying the following equation:

\deqn{\kappa_{tc} = \kappa_{tc.g} / (1 - \kappa_{tc.g} * log(tc/tc.g))}

where

\deqn{\kappa_{tc.g} = g / 100 / log(10)}

with \eqn{log} the natural logarithm.

The error of the fading-corrected age is determined using a Monte Carlo
simulation approach. Solving of the equation is realised using
\code{\link{uniroot}}. Large values for \code{n.MC} will significantly
increase the computation time.\cr

\bold{\code{n.MC = 'auto'}}

The error estimation based on a stochastic process, i.e. for a small number of MC runs the calculated
error varies considerably every time the function is called, even with the same input values.
The argument option \code{n.MC = 'auto'} tries to find a stable value for the standard error, i.e.
the standard deviation of values calculated during the MC runs (\code{age.corr.MC}),
within a given precision (2 digits) by increasing the number of MC runs stepwise and
calculating the corresponding error.

If the determined error does not differ from the 9 values calculated previously
within a precision of (here) 3 digits the calculation is stopped as it is assumed that the error
is stable. Please note that (a) the duration depends on the input values as well as on
the provided computation ressources and it may take a while, (b) the length (size) of the output
vector \code{age.corr.MC}, where all the single values produced during the MC runs are stored,
equals the number of MC runs (here termed observations).

To avoid an endless loop the calculation is stopped if the number of observations exceeds 10^7.
This limitation can be overwritten by setting the number of MC runs manually,
e.g. \code{n.MC = 10000001}. Note: For this case the function is not checking whether the calculated
error is stable.\cr

\bold{\code{seed}}

This option allows to recreate previously calculated results by setting the seed
for the R random number generator (see \code{\link{set.seed}} for details). This option
should not be mixed up with the option \bold{\code{n.MC = 'auto'}}. The results may
appear similar, but they are not comparable!\cr

\bold{FAQ}\cr
Q: Which tc value is expected?\cr
A: tc is the time in seconds between irradiation and the prompt measurement applied during your
De measurement. However, this tc might differ from the tc used for estimating the g-value. In the
case of an SAR measurement tc should be similar, however, if it differs, you have to provide this
tc value (the one used for estimating the g-value) using the argument \code{tc.g_value}.\cr
}
\note{
The upper age limit is set to 500 ka! \cr
Special thanks to Sebastien Huot for his support and clarification via e-mail.
}
\section{Function version}{
 0.4.1 (2016-07-21 10:36:31)
}
\examples{

##run the examples given in the appendix of Huntley and Lamothe, 2001

##(1) faded age: 100 a
results <- calc_FadingCorr(
   age.faded = c(0.1,0),
   g_value = c(5.0, 1.0),
   tc = 2592000,
   tc.g_value = 172800,
   n.MC = 100)

##(2) faded age: 1 ka
results <- calc_FadingCorr(
   age.faded = c(1,0),
   g_value = c(5.0, 1.0),
   tc = 2592000,
   tc.g_value = 172800,
   n.MC = 100)

##(3) faded age: 10.0 ka
results <- calc_FadingCorr(
   age.faded = c(10,0),
   g_value = c(5.0, 1.0),
   tc = 2592000,
   tc.g_value = 172800,
   n.MC = 100)

##access the last output
get_RLum(results)

}
\author{
Sebastian Kreutzer, IRAMAT-CRP2A, Universite Bordeaux Montaigne (France)
\cr R Luminescence Package Team}
\references{
Huntley, D.J., Lamothe, M., 2001. Ubiquity of anomalous fading
in K-feldspars and the measurement and correction for it in optical dating.
Canadian Journal of Earth Sciences, 38, 1093-1106.
}
\seealso{
\code{\linkS4class{RLum.Results}}, \code{\link{get_RLum}},
\code{\link{uniroot}}
}
\keyword{datagen}

