% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/report_RLum.R
\name{report_RLum}
\alias{report_RLum}
\title{Create a HTML-report for (RLum) objects}
\usage{
report_RLum(
  object,
  file = tempfile(),
  title = "RLum.Report",
  compact = TRUE,
  timestamp = TRUE,
  show_report = TRUE,
  launch.browser = FALSE,
  css.file = NULL,
  quiet = TRUE,
  clean = TRUE,
  ...
)
}
\arguments{
\item{object}{(\strong{required}):
The object to be reported on, preferably of any \code{RLum}-class.}

\item{file}{\link{character} (\emph{with default}):
A character string naming the output file. If no filename is provided a
temporary file is created.}

\item{title}{\link{character} (\emph{with default}):
A character string specifying the title of the document.}

\item{compact}{\link{logical} (\emph{with default}):
When \code{TRUE} the following report components are hidden:
\verb{@.pid}, \verb{@.uid}, \code{'Object structure'}, \code{'Session Info'}
and only the first and last 5 rows of long matrices and data frames are shown.
See details.}

\item{timestamp}{\link{logical} (\emph{with default}):
\code{TRUE} to add a timestamp to the filename (suffix).}

\item{show_report}{\link{logical} (\emph{with default}): If set to \code{TRUE} the function tries to display
the report output in the local viewer, e.g., within \emph{RStudio} after rendering.}

\item{launch.browser}{\link{logical} (\emph{with default}):
\code{TRUE} to open the HTML file in the system's default web browser after
it has been rendered.}

\item{css.file}{\link{character} (\emph{optional}):
Path to a CSS file to change the default styling of the HTML document.}

\item{quiet}{\link{logical} (\emph{with default}):
\code{TRUE} to suppress printing of the pandoc command line.}

\item{clean}{\link{logical} (\emph{with default}):
\code{TRUE} to clean intermediate files created during rendering.}

\item{...}{further arguments passed to or from other methods and to control
the document's structure (see details).}
}
\value{
Writes a HTML and .Rds file.
}
\description{
Create a HTML-report for (RLum) objects
}
\details{
This function creates a HTML-report for a given object, listing its complete
structure and content. The object itself is saved as a serialised .Rds file.
The report file serves both as a convenient way of browsing through objects with
complex data structures as well as a mean of properly documenting and saving
objects.

The HTML report is created with \link[rmarkdown:render]{rmarkdown::render} and has the
following structure:

\tabular{ll}{
\strong{Section} \tab \strong{Description} \cr
\code{Header} \tab A summary of general characteristics of the object \cr
\verb{Object content} \tab A comprehensive list of the complete structure and content of the provided object. \cr
\verb{Object structure} \tab Summary of the objects structure given as a table \cr
\code{File} \tab Information on the saved RDS file \cr
\verb{Session Info} \tab Captured output from \code{sessionInfo()} \cr
\code{Plots} \tab (\emph{optional}) For \code{RLum-class} objects a variable number of plots \cr
}

The structure of the report can be controlled individually by providing one or more of the
following arguments (all \code{logical}):

\tabular{ll}{
\strong{Argument} \tab \strong{Description} \cr
\code{header} \tab Hide or show general information on the object \cr
\code{main} \tab Hide or show the object's content \cr
\code{structure} \tab Hide or show object's structure \cr
\code{rds} \tab Hide or show information on the saved RDS file \cr
\code{session} \tab Hide or show the session info \cr
\code{plot} \tab Hide or show the plots (depending on object) \cr
}

Note that these arguments have higher precedence than \code{compact}.

Further options that can be provided via the \code{...} argument:

\tabular{ll}{
\strong{Argument} \tab \strong{Description} \cr
\code{short_table} \tab If \code{TRUE} only show the first and last 5 rows of long tables. \cr
\code{theme} \tab Specifies the Bootstrap
theme to use for the report. Valid themes include \code{"default"}, \code{"cerulean"}, \code{"journal"}, \code{"flatly"},
\code{"readable"}, \code{"spacelab"}, \code{"united"}, \code{"cosmo"}, \code{"lumen"}, \code{"paper"}, \code{"sandstone"},
\code{"simplex"}, and \code{"yeti"}. \cr
\code{highlight} \tab Specifies the syntax highlighting style.
Supported styles include \code{"default"}, \code{"tango"}, \code{"pygments"}, \code{"kate"}, \code{"monochrome"},
\code{"espresso"}, \code{"zenburn"}, \code{"haddock"}, and \code{"textmate"}. \cr
\code{css} \tab \code{TRUE} or \code{FALSE} to enable/disable custom CSS styling \cr
}

The following arguments can be used to customise the report via CSS (Cascading Style Sheets):

\tabular{ll}{
\strong{Argument} \tab \strong{Description} \cr
\code{font_family} \tab Define the font family of the HTML document (default: \code{"arial"}) \cr
\code{headings_size} \tab Size of the \verb{<h1>} to \verb{<h6>} tags used to define HTML headings (default: 166\%). \cr
\code{content_color} \tab Colour of the object's content (default: #a72925). \cr
}

Note that these arguments must all be of class \link{character} and follow standard CSS syntax.
For exhaustive CSS styling you can provide a custom CSS file for argument \code{css.file}.
CSS styling can be turned of using \code{css = FALSE}.
}
\note{
This function requires the R packages 'rmarkdown', 'pander' and 'rstudioapi'.
}
\section{Function version}{
 0.1.4
}

\examples{

\dontrun{
## Example: RLum.Results ----

# load example data
data("ExampleData.DeValues")

# apply the MAM-3 age model and save results
mam <- calc_MinDose(ExampleData.DeValues$CA1, sigmab = 0.2)

# create the HTML report
report_RLum(object = mam, file = "~/CA1_MAM.Rmd",
            timestamp = FALSE,
            title = "MAM-3 for sample CA1")

# when creating a report the input file is automatically saved to a
# .Rds file (see saveRDS()).
mam_report <- readRDS("~/CA1_MAM.Rds")
all.equal(mam, mam_report)


## Example: Temporary file & Viewer/Browser ----

# (a)
# Specifying a filename is not necessarily required. If no filename is provided,
# the report is rendered in a temporary file. If you use the RStudio IDE, the
# temporary report is shown in the interactive Viewer pane.
report_RLum(object = mam)

# (b)
# Additionally, you can view the HTML report in your system's default web browser.
report_RLum(object = mam, launch.browser = TRUE)


## Example: RLum.Analysis ----

data("ExampleData.RLum.Analysis")

# create the HTML report (note that specifying a file
# extension is not necessary)
report_RLum(object = IRSAR.RF.Data, file = "~/IRSAR_RF")


## Example: RLum.Data.Curve ----

data.curve <- get_RLum(IRSAR.RF.Data)[[1]]

# create the HTML report
report_RLum(object = data.curve, file = "~/Data_Curve")

## Example: Any other object ----
x <- list(x = 1:10,
          y = runif(10, -5, 5),
          z = data.frame(a = LETTERS[1:20], b = dnorm(0:9)),
          NA)

report_RLum(object = x, file = "~/arbitray_list")
}

}
\seealso{
\link[rmarkdown:render]{rmarkdown::render}, \link[pander:pander_return]{pander::pander_return},
\link[pander:openFileInOS]{pander::openFileInOS}, \link[rstudioapi:viewer]{rstudioapi::viewer},
\link{browseURL}
}
\author{
Christoph Burow, University of Cologne (Germany),
Sebastian Kreutzer, Geography & Earth Sciences, Aberystwyth University (United Kingdom) \cr
, RLum Developer Team} 

\section{How to cite}{
Burow, C., Kreutzer, S., 2021. report_RLum(): Create a HTML-report for (RLum) objects. Function version 0.1.4. In: Kreutzer, S., Burow, C., Dietze, M., Fuchs, M.C., Schmidt, C., Fischer, M., Friedrich, J., Mercier, N., Philippe, A., Riedesel, S., Autzen, M., Mittelstrass, D., Gray, H.J., Galharret, J., 2021. Luminescence: Comprehensive Luminescence Dating Data Analysis. R package version 0.9.16. https://CRAN.R-project.org/package=Luminescence
}

