% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/run.functions.R
\name{mb.run}
\alias{mb.run}
\title{Run MBNMA time-course models}
\usage{
mb.run(
  network,
  fun = tpoly(degree = 1),
  positive.scale = FALSE,
  intercept = NULL,
  link = "identity",
  parameters.to.save = NULL,
  rho = 0,
  covar = "varadj",
  omega = NULL,
  corparam = TRUE,
  class.effect = list(),
  UME = FALSE,
  pd = "pv",
  parallel = FALSE,
  priors = NULL,
  n.iter = 20000,
  n.chains = 3,
  n.burnin = floor(n.iter/2),
  n.thin = max(1, floor((n.iter - n.burnin)/1000)),
  model.file = NULL,
  jagsdata = NULL,
  ...
)
}
\arguments{
\item{network}{An object of class \code{"mb.network"}.}

\item{fun}{An object of class \code{"timefun"} generated (see Details) using any of
\code{tloglin()}, \code{tpoly()}, \code{texp()}, \code{temax()}, \code{tfpoly()}, \code{tspline()} or \code{tuser()}}

\item{positive.scale}{A boolean object that indicates whether all continuous
mean responses (y) are positive and therefore whether the baseline response
should be given a prior that constrains it to be positive (e.g. for scales that cannot be <0).}

\item{intercept}{A boolean object that indicates whether an intercept (written
as \code{alpha} in the model) is to be included. If left as \code{NULL} (the default), an intercept will
be included only for studies reporting absolute means, and will be excluded for
studies reporting change from baseline (as indicated in \code{network$cfb}).}

\item{link}{Can take either \code{"identity"} (the default),
\code{"log"} (for modelling Ratios of Means \insertCite{friedrich2011}{MBNMAtime}) or
\code{"smd"} (for modelling Standardised Mean Differences - although this also corresponds to an identity link function).}

\item{parameters.to.save}{A character vector containing names of parameters
to monitor in JAGS}

\item{rho}{The correlation coefficient when modelling within-study correlation between time points. The default is a string representing a
prior distribution in JAGS, indicating that it be estimated from the data (e.g. \code{rho="dunif(0,1)"}). \code{rho} also be assigned a
numeric value (e.g. \code{rho=0.7}), which fixes \code{rho} in the model to this value (e.g. for use in a deterministic sensitivity analysis).
If set to \code{rho=0} (the default) then this implies modelling no correlation between time points.}

\item{covar}{A character specifying the covariance structure to use for modelling within-study correlation between time-points. This can
be done by specifying one of the following:
\itemize{
\item \code{"varadj"} - a univariate likelihood with a variance adjustment to assume a constant correlation between subsequent
time points \insertCite{jansen2015}{MBNMAtime}. This is the default.
\item \code{"CS"} - a multivariate normal likelihood with a
\href{https://support.sas.com/resources/papers/proceedings/proceedings/sugi30/198-30.pdf}{compound symmetry} structure
\item \code{"AR1"} - a multivariate normal likelihood with an
\href{https://support.sas.com/resources/papers/proceedings/proceedings/sugi30/198-30.pdf}{autoregressive AR1} structure
}}

\item{omega}{A scale matrix for the inverse-Wishart prior for the covariance matrix used
to model the correlation between time-course parameters (see Details for time-course functions). \code{omega} must
be a symmetric positive definite matrix with dimensions equal to the number of time-course parameters modelled using
relative effects (\code{pool="rel"}). If left as \code{NULL} (the default) a diagonal matrix with elements equal to 1
is used.}

\item{corparam}{A boolean object that indicates whether correlation should be modelled
between relative effect time-course parameters. This is automatically set to \code{FALSE} if class effects are modelled.
It can also be useful for providing informative priors more easily to the model.}

\item{class.effect}{A list of named strings that determines which time-course
parameters to model with a class effect and what that effect should be
(\code{"common"} or \code{"random"}). For example: \code{list(emax="common", et50="random")}.}

\item{UME}{Can take either \code{TRUE} or \code{FALSE} (for an unrelated mean effects
model on all or no time-course parameters respectively) or can be a vector
of parameter name strings to model as UME. For example: \code{c("beta.1", "beta.2")}.}

\item{pd}{Can take either:
\itemize{
\item \code{pv} only pV will be reported (as automatically outputted by R2jags).
\item \code{plugin} calculates pD by the plug-in
method \insertCite{spiegelhalter2002}{MBNMAtime}. It is faster, but may output negative
non-sensical values, due to skewed deviances that can arise with non-linear models.
\item \code{pd.kl} (the default) calculates pD by the Kullback–Leibler divergence \insertCite{plummer2008}{MBNMAtime}. This
will require running the model for additional iterations but
will always produce a sensical result.
\item \code{popt} calculates pD using an optimism adjustment which allows for calculation
of the penalized expected deviance \insertCite{plummer2008}{MBNMAtime}
}}

\item{parallel}{A boolean value that indicates whether JAGS should be run in
parallel (\code{TRUE}) or not (\code{FALSE}). If \code{TRUE} then the number of cores to
use is automatically calculated. Functions that involve updating the model (e.g. \code{devplot()}, \code{fitplot()})
cannot be used with models implemented in parallel.}

\item{priors}{A named list of parameter values (without indices) and
replacement prior distribution values given as strings
\strong{using distributions as specified in JAGS syntax} (see \insertCite{jagsmanual;textual}{MBNMAtime}).}

\item{n.iter}{number of total iterations per chain (including burn in; default: 20000)}

\item{n.chains}{number of Markov chains (default: 3)}

\item{n.burnin}{length of burn in, i.e. number of iterations to discard at the
beginning. Default is \verb{n.iter/2``, that is, discarding the first half of the simulations. If }n.burnin` is 0, jags() will run 100 iterations for adaption.}

\item{n.thin}{thinning rate. Must be a positive integer. Set \verb{n.thin > 1`` to save memory and computation time if }n.iter\verb{is large. Default is}max(1, floor(n.chains * (n.iter-n.burnin) / 1000))`` which will only thin if there are at least 2000
simulations.}

\item{model.file}{The file path to a JAGS model (.jags file) that can be used
to overwrite the JAGS model that is automatically written based on the
specified options in \code{MBNMAtime}. Useful for adding further model flexibility.}

\item{jagsdata}{A named list of the data objects to be used in the JAGS model. Only
required if users are defining their own JAGS model using \code{model.file}. Format
should match that of standard models fitted in \code{MBNMAtime}
(see \code{mbnma$model.arg$jagsdata})}

\item{...}{Arguments to be sent to R2jags.}
}
\value{
An object of S3 class \verb{c("mbnma", "rjags")`` containing parameter results from the model. Can be summarized by }print()\verb{and can check traceplots using}R2jags::traceplot()\verb{or various functions from the package}mcmcplots`.#'

If there are errors in the JAGS model code then the object will be a list
consisting of two elements - an error message from JAGS that can help with
debugging and \code{model.arg}, a list of arguments provided to \code{mb.run()}
which includes \code{jagscode}, the JAGS code for the model that can help
users identify the source of the error.
}
\description{
Fits a Bayesian time-course model for model-based network meta-analysis
(MBNMA) that can account for repeated measures over time within studies by
applying a desired time-course function. Follows the methods of \insertCite{pedder2019;textual}{MBNMAtime}.
}
\section{Time-course parameters}{

Nodes that are automatically monitored (if present in the model) have the
same name as in the time-course function for named time-course parameters (e.g. \code{emax}).
However, for named only as \code{beta.1}, \code{beta.2}, \code{beta.3} or \code{beta.4} parameters
may have an alternative interpretation.

Details of the interpretation and model specification of different parameters can be shown by using the
\code{summary()} method on an \code{"mbnma"} object generated by \code{mb.run()}.

\emph{Parameters modelled using relative effects}
\itemize{
\item If pooling is relative (e.g. \code{pool.1="rel"}) for a given parameter then the named parameter (e.g. \code{emax}) or a
numbered \code{d} parameter (e.g. \code{d.1}) corresponds to the pooled relative effect for a given
treatment compared to the network reference treatment for this time-course parameter.
\item \code{sd.} followed by a named (e.g. \code{emax}, \code{beta.1}) is the between-study SD (heterogeneity)
for relative effects, reported if pooling for a time-course parameter is relative (e.g. \code{pool.1="rel"}) \emph{and} the
method for synthesis is random (e.g. \verb{method.1="random}).
\item If class effects are modelled, parameters for classes are represented by the upper case name of the time-course
parameter they correspond to. For example if \code{class.effect=list(emax="random")}, relative class effects will be
represented by \code{EMAX}. The SD of the class effect (e.g. \code{sd.EMAX}, \code{sd.BETA.1}) is the SD of treatments within a class for the
time-course parameter they correspond to.
}

\emph{Parameters modelled using absolute effects}
\itemize{
\item If pooling is absolute (e.g. \code{pool.1="abs"}) for a given parameter then the named parameter (e.g. \code{emax}) or a
numbered \code{beta} parameter (e.g. \code{beta.1}) corresponds to the estimated absolute effect for this time-course parameter.
\item For an absolute time-course parameter if the corresponding method is common (e.g. \code{method.1="common"}) the parameter
corresponds to a single common parameter estimated across all studies and treatments. If the corresponding method is
random (e.g. \code{method.1="random"}) then parameter is a mean effect around which the study-level absolute effects vary
with SD corresponding to \code{sd.} followed by the named parameter (e.g. \code{sd.emax}, \code{sd.beta.1}) .
}

\emph{Other model parameters}
\itemize{
\item \code{rho} The correlation coefficient for correlation between time-points. Its
interpretation will differ depending on the covariance structure specified in \code{covar}
\item \code{totresdev} The residual deviance of the model
\item \code{deviance} The deviance of the model
}
}

\section{Time-course function}{

Several general time-course functions with up to 4 time-course parameters are provided, but a
user-defined time-course relationship can instead be used. Details can be found in the respective
help files for each function.

Available time-course functions are:
\itemize{
\item Log-linear: \code{tloglin()}
\item Polynomial: \code{tpoly()}
\item Exponential: \code{texp()}
\item Emax: \code{temax()}
\item Fractional polynomial: \code{tfpoly()}
\item Splines (various spline types can be used): \code{tspline()}
\item User-defined: \code{tuser()}
}
}

\section{Correlation between observations}{

When modelling correlation between observations using \code{rho}, values for \code{rho} must imply a
positive semidefinite covariance matrix.
}

\section{Advanced options}{

\code{model.file} and \code{jagsdata} can be used to run an edited JAGS model and dataset. This allows
users considerably more modelling flexibility than is possible using the basic \code{MBNMAtime} syntax,
though requires strong understanding of JAGS and the MBNMA modelling framework. Treatment-specific
priors, meta-regression and bias-adjustment are all possible in this way, and it allows users to
make use of the subsequent functions in \code{MBNMAtime} (plotting, prediction, ranking) whilst fitting
these more complex models.
}

\examples{
\donttest{
# Create mb.network object
network <- mb.network(osteopain)

# Fit a linear time-course MBNMA with:
# random relative treatment effects on the slope
mb.run(network, fun=tpoly(degree=1, pool.1="rel", method.1="random"))

# Fit an emax time-course MBNMA with:
# fixed relative treatment effects on emax
# a common parameter estimated independently of treatment
# a common Hill parameter estimated independently of treatment
# a prior for the Hill parameter (normal with mean 0 and precision 0.1)
# data reported as change from baseline
result <- mb.run(network, fun=temax(pool.emax="rel", method.emax="common",
                                    pool.et50="abs", method.et50="common",
                                    pool.hill="abs", method.hill="common"),
                 priors=list(hill="dnorm(0, 0.1)"),
                 intercept=TRUE)


#### commented out to prevent errors from JAGS version in github actions build ####
# Fit a log-linear MBNMA with:
# random relative treatment effects on the rate
# an autoregressive AR1 covariance structure
# modelled as standardised mean differences
# copdnet <- mb.network(copd)
# result <- mb.run(copdnet, fun=tloglin(pool.rate="rel", method.rate="random"),
#                covar="AR1", rho="dunif(0,1)", link="smd")



####### Examine MCMC diagnostics (using mcmcplots package) #######

# Traceplots
# mcmcplots::traplot(result)

# Plots for assessing convergence
# mcmcplots::mcmcplot(result, c("rate", "sd.rate", "rho"))

########## Output ###########

# Print R2jags output and summary
print(result)
summary(result)

# Plot forest plot of results
plot(result)


###### Additional model arguments ######

# Use gout dataset
goutnet <- mb.network(goutSUA_CFBcomb)

# Define a user-defined time-course relationship for use in mb.run
timecourse <- ~ exp(beta.1 * time) + (time^beta.2)

# Run model with:
# user-defined time-course function
# random relative effects on beta.1
# default common effects on beta.2
# default relative pooling on beta.1 and beta.2
# common class effect on beta.2
mb.run(goutnet, fun=tuser(fun=timecourse, method.1="random"),
       class.effect=list(beta.1="common"))

# Fit a log-linear MBNMA
# with variance adjustment for correlation between time-points
result <- mb.run(network, fun=tloglin(),
                 rho="dunif(0,1)", covar="varadj")
}
}
\references{
\insertAllCited
}
