% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/KRV.R
\name{KRV}
\alias{KRV}
\title{Kernel RV Coefficient Test (KRV)}
\usage{
KRV(
  y = NULL,
  X = NULL,
  kernels.otu,
  kernel.y,
  returnKRV = FALSE,
  returnR2 = FALSE
)
}
\arguments{
\item{y}{A numeric n by p matrix of p continuous phenotype variables and sample size n (default = NULL). If it is NULL, a 
phenotype kernel matrix must be entered for "kernel.y". Defaults to NULL.}

\item{X}{A numeric n by q matrix, containing q additional covariates (default = NULL). If NULL, 
an intercept only model is used. No covariate adjustment is possible if a matrix is provided in kernel.y.}

\item{kernels.otu}{A numeric OTU n by n kernel matrix or a list of matrices, where n is the sample size. It can be 
constructed from microbiome data, such as by transforming from a distance metric.}

\item{kernel.y}{Either a numerical n by n kernel matrix for phenotypes or a method to compute the kernel of phenotype. Methods are "Gaussian" or "linear". 
A Gaussian kernel (kernel.y="Gaussian") can capture the general relationship between microbiome and phenotypes; a linear kernel (kernel.y="linear") 
may be preferred if the underlying relationship is close to linear.}

\item{returnKRV}{A logical indicating whether to return the KRV statistic. Defaults to FALSE.}

\item{returnR2}{A logical indicating whether to return the R-squared coefficient. Defaults to FALSE.}
}
\value{
If only one candidate kernel matrix is considered, returns a list containing the p-value for the candidate kernel matrix. 
If more than one candidate kernel matrix is considered, returns a list of two elements: 
    \item{p_values}{P-value for each candidate kernel matrix}
    \item{omnibus_p}{Omnibus p-value} 
    \item{KRV}{A vector of kernel RV statistics (a measure of effect size), one for each candidate kernel matrix. Only returned if returnKRV = TRUE}
    \item{R2}{A vector of R-squared statistics, one for each candidate kernel matrix. Only returned if returnR2 = TRUE}
}
\description{
Kernel RV coefficient test to evaluate the overall association between microbiome composition and high-dimensional or
structured phenotype.
}
\details{
kernels.otu should be a list of numerical n by n kernel matrices, or a single n by n kernel matrix, where n is sample size.

When kernel.y is a method ("Gaussian" or "linear") to compute the kernel of phenotype, y should be a numerical phenotype matrix, 
and X (if not NULL) should be a numeric matrix of covariates. Both y and X should have n rows.

When kernel.y is a kernel matrix for the phenotype, there is no need to provide X and y, and they will be ignored if provided.
In this case, kernel.y and kernel.otu should both be numeric matrices with the same number of rows and columns.

Missing data is not permitted. Please remove all individuals with missing kernel.otu, y (if not NULL), X (if not NULL), and 
kernel.y (if a matrix is entered) prior to analysis.
}
\examples{
        
library(GUniFrac)
library(MASS)

data(throat.tree)
data(throat.otu.tab)
data(throat.meta)


set.seed(123)
n = nrow(throat.otu.tab)
Sex <- throat.meta$Sex
Smoker <- throat.meta$SmokingStatus
anti <- throat.meta$AntibioticUsePast3Months_TimeFromAntibioticUsage
Male = (Sex == "Male")**2
Smoker =(Smoker == "Smoker") **2
anti =  (anti != "None")^2
cova = cbind(Male,  anti)

otu.tab.rff <- Rarefy(throat.otu.tab)$otu.tab.rff
unifracs <- GUniFrac(otu.tab.rff, throat.tree, alpha=c(0, 0.5, 1))$unifracs

D.weighted = unifracs[,,"d_1"]
D.unweighted = unifracs[,,"d_UW"]
D.BC= as.matrix(vegdist(otu.tab.rff , method="bray"))


K.weighted = D2K(D.weighted)
K.unweighted = D2K(D.unweighted)
K.BC = D2K(D.BC)

rho = 0.2
Va = matrix(rep(rho, (2*n)^2), 2*n, 2*n)+diag(1-rho, 2*n)
G = mvrnorm(n, rep(0, 2*n), Va)

KRV(kernels.otu = K.weighted, kernel.y = G \%*\% t(G)) 



}
\references{
Zheng, Haotian, Zhan, X., Plantinga, A., Zhao, N., and Wu, M.C. A Fast Small-Sample Kernel Independence Test for Microbiome 
Community-Level Association Analysis. Biometrics. 2017 Mar 10. doi: 10.1111/biom.12684.
}
\author{
Nehemiah Wilson, Haotian Zheng, Xiang Zhan, Ni Zhao
}
