% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/utilityFunctions.R
\name{mT.translate}
\alias{mT.translate}
\title{Translate base table of a cohort mortality table to a different observation year}
\usage{
mT.translate(table, baseYear, name = NULL)
}
\arguments{
\item{table}{A life table object (instance of a \code{mortalityTable} class)
or a list, table or array of mortalityTable objects}

\item{baseYear}{Target base year. The underlying period life table of the
cohort life table is translated to the desired target base
year by applying the trend factors of the table, resulting
in a consistent shift of the internal representation without
changing the resulting probabilities.}

\item{name}{(optional) new name for the mortality table}
}
\description{
Translate the base table of a cohort life table to a different observation period,
using the existing base table and the trend functions. This only has an effect on
cohort life tables (e.g. objects of class \code{mortalityTable.trendProjection}).
For all other life tables (period life tables, observed, etc.), this function has no effect.
}
\details{
This function also does not modify the resulting death probabilities of the life table
object, it just reparameterizes the internal representation of a life table
with trend projection factors.

This functionality is often needed when publisheing life thables. Typically,
the table is derived from a certain observation period, so the resulting base
table describes the middle of the observation period. Projetion into the future
is then done using trend projection factors starting from that base table.
On the other hand, for the published table it is often desired to tabulate
not the middle of the observation period, but rather the current year as base
year for the extrapolation.
For the resulting period or cohort death probabilities, it is irrelevant, which
base year is used, as long as the shift to another base year (which includes
translating the base mortalities of the base year) is done consistenly with the
trend functions. The function \code{mT.translate} ensures this.
}
\examples{
mortalityTables.load("Austria_Annuities_AVOe2005R")
# The AVOe2005R.male.nodamping has 2001 as the base year. Move its base year
# to 2020 without modifying cohort probabilities
avoe05r.shifted = mT.translate(AVOe2005R.male.nodamping, 2020, "AVÖ 2005-R, translated to 2020")
plotMortalityTables(
    getPeriodTable(AVOe2005R.male.nodamping),
    getPeriodTable(avoe05r.shifted),
    title = "Base tables of the AVÖ 2005R a translated version to 2020")
# Even though the base tables are shifted, the resulting probabilities are
# unchanged (except for numeric artefacts)
abs(periodDeathProbabilities(AVOe2005R.male.nodamping, Period = 2050) -
    periodDeathProbabilities(avoe05r.shifted, Period = 2050)) < 0.00000001
abs(deathProbabilities(AVOe2005R.male.nodamping, YOB = 2050) -
    deathProbabilities(avoe05r.shifted, YOB = 2050)) < 0.00000001
}
