%
%   Copyright 2007-2016 The OpenMx Project
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
% 
%        http://www.apache.org/licenses/LICENSE-2.0
% 
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

\name{mxFitFunctionMultigroup}
\alias{MxFitFunctionMultigroup-class}
\alias{mxFitFunctionMultigroup}

\title{Create MxFitFunctionMultigroup object}

\description{The fit function used to fit a multiple group model}

\usage{
mxFitFunctionMultigroup(groups, ..., verbose = 0L)
}

\arguments{
\item{groups}{vector of fit function names (strings)}

\item{...}{Not used. Forces subsequent arguments to be specified by name.}

\item{verbose}{the level of debugging output}
}

\details{
The mxFitFunctionMultigroup creates a fit function consisting of the sum of the fit
statistics from a list of submodels provided.  Thus, it aggregates fit statistics from multiple submodels.

This is conceptually similar to creating an \link{mxAlgebra} consisting of the sum of 
the subModel objectives and also creating an \link{mxFitFunctionAlgebra} to optimize the
model based on this aggregate value.

This call to mxFitFunctionMultigroup:

    \code{mxFitFunctionMultigroup(c("model1", "model2"))}

then, is almost equivalent to the following pair of statements:

    \code{mxAlgebra(model1.objective + model2.objective, name="myAlgebra")}

    \code{mxFitFunctionAlgebra("myAlgebra")}

The preferred method of specifying such a fit function is with this multigroup fit function, not the algebra fit function.

In addition to being more compact and readable, using mxFitFunctionMultigroup has
additional side effects which are valuable for multi-group modeling.

Firstly, it aggregates analytic derivative calculations. Secondly, it allows \code{\link{mxRefModels}} 
to compute saturated models for raw data, as this function can learn which are the constituent submodels.  Thirdly, it allows \code{\link{mxCheckIdentification}} to evaluate the local identification of the multigroup model.

Note: You can refer to the algebra generated by mxFitFunctionMultigroup when used in a group "modelName"
as:

    \code{modelName.fitfunction}

}

\seealso{
Other fit functions:
\code{\link{mxFitFunctionML}}, 
\code{\link{mxFitFunctionWLS}}, \code{\link{mxFitFunctionAlgebra}}, 
\code{\link{mxFitFunctionGREML}}, \code{\link{mxFitFunctionR}}, 
\code{\link{mxFitFunctionRow}}

More information about the OpenMx package may be found \link[=OpenMx]{here}. 
}


\examples{
#------------------------------------------------
# Brief non-running example
require("OpenMx")
mxFitFunctionMultigroup(c("model1", "model2")) # names of sub-models to be jointly optimised


#------------------------------------------------
# Longer, fully featured, running example
# 
# 
# Create and fit a model using mxMatrix, mxExpectationRAM, mxFitFunctionML,
# and mxFitFunctionMultigroup.
# The model is multiple group regression.
# Only the residual variances are allowed to differ across groups.


library(OpenMx)

# Simulate some data

# Group 1
N1=100
x=rnorm(N1, mean=0, sd=1)
y= 0.5*x + rnorm(N1, mean=0, sd=1)
ds1 <- data.frame(x, y)
dsNames <- names(ds1)

# Group 2
N2=150
x=rnorm(N2, mean=0, sd=1)
y= 0.5*x + rnorm(N2, mean=0, sd=sqrt(1.5))
ds2 <- data.frame(x, y)


# Define the matrices
M <- mxMatrix(type = "Full", nrow = 1, ncol = 2, values=0,
              free=TRUE, labels=c("Mx", "My"), name = "M")
S1 <- mxMatrix(type = "Diag", nrow = 2, ncol = 2, values=1, 
              free=TRUE, labels=c("Vx", "ResidVy1"), name = "S")
S2 <- mxMatrix(type = "Diag", nrow = 2, ncol = 2, values=1, 
              free=TRUE, labels=c("Vx", "ResidVy2"), name = "S")
A <- mxMatrix(type = "Full", nrow = 2, ncol = 2, values=c(0,1,0,0), 
              free=c(FALSE,TRUE,FALSE,FALSE), labels=c(NA, "b", NA, NA),
              name = "A")
I <- mxMatrix(type="Iden", nrow=2, ncol=2, name="I")


# Define the expectation
expect <- mxExpectationRAM('A', 'S', 'I', 'M', dimnames=dsNames)


# Choose a fit function
fitFunction <- mxFitFunctionML(rowDiagnostics=TRUE)
# also return row likelihoods, even though the fit function
#  value is still 1x1

# Multiple groupd fit function sums the model likelihoods
#  from its component models
mgFitFun <- mxFitFunctionMultigroup(c('g1model', 'g2model'))


# Define the models
m1 <- mxModel(model="g1model", M, S1, A, I, expect, fitFunction, 
              mxData(observed=cov(ds1), type="cov", numObs=nrow(ds1),
                     means=colMeans(ds1)))
m2 <- mxModel(model="g2model", M, S2, A, I, expect, fitFunction, 
              mxData(observed=cov(ds2), type="cov", numObs=nrow(ds2),
                     means=colMeans(ds2)))
mg <- mxModel(model='multipleGroup', m1, m2, mgFitFun)

# Fit the model and print a summary

mgOut <- mxRun(mg)

# Look at summary of model
summary(mgOut)

# Examine fit function results
fitResOnly <- mxEval(fitfunction, mgOut)
( g1Fit <- mxEval(g1model.fitfunction, mgOut) )
( g2Fit <- mxEval(g2model.fitfunction, mgOut) )


}
