% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plsim.vs.soft.r
\name{plsim.vs.soft}
\alias{plsim.vs.soft}
\alias{plsim.vs.soft.formula}
\alias{plsim.vs.soft.default}
\title{Penalized Profile Least Squares Estimator}
\usage{
plsim.vs.soft(\dots)

\method{plsim.vs.soft}{formula}(formula, data, \dots)

\method{plsim.vs.soft}{default}(xdat=NULL, zdat, ydat, h=NULL, zetaini=NULL, 
lambda=0.01, l1_ratio=NULL, MaxStep = 1L, penalty = "SCAD", verbose=TRUE, 
ParmaSelMethod="SimpleValidation", TestRatio=0.1, K = 3, seed=0, \dots)
}
\arguments{
\item{\dots}{additional arguments.}

\item{formula}{a symbolic description of the model to be fitted.}

\item{data}{an optional data frame, list or environment containing the variables in the model.}

\item{xdat}{input matrix (linear covariates). The model reduces to a single index model when \code{x} is NULL.}

\item{zdat}{input matrix (nonlinear covariates). \code{z} should not be NULL.}

\item{ydat}{input vector (response variable).}

\item{h}{a value or a vector for bandwidth. If \code{h} is NULL, a default vector c(0.01,0.02,0.05,0.1,0.5)
will be set for it. \link{plsim.bw} is employed to select the optimal bandwidth when \code{h} is a vector or NULL.}

\item{zetaini}{initial coefficients, optional (default: NULL). It could be obtained by the function \code{\link{plsim.ini}}.
\code{zetaini[1:ncol(z)]} is the initial coefficient vector \eqn{\alpha_0},
and \code{zetaini[(ncol(z)+1):(ncol(z)+ncol(x))]} is the initial coefficient vector \eqn{{\beta}_0}.}

\item{MaxStep}{int, optional (default=1). Hard limit on iterations within solver.}

\item{lambda}{double. Constant that multiplies the penalty term.}

\item{l1_ratio}{double, default=NULL. It should be set with a value from the range \eqn{[0,1]}
when you choose "ElasticNet" for the parameter \code{penalty}.}

\item{penalty}{string, optional (default="SCAD"). It could be "SCAD", "LASSO" and "ElasticNet".}

\item{verbose}{bool, default: TRUE. Enable verbose output.}

\item{ParmaSelMethod}{the parameter for the function \link{plsim.bw}.}

\item{TestRatio}{the parameter for the function \link{plsim.bw}.}

\item{K}{the parameter for the function \link{plsim.vs.soft}.}

\item{seed}{int, default: 0.}
}
\value{
\item{eta}{estimated non-parametric part \eqn{\hat{\eta}(Z^T{\hat{\alpha} })}.}
\item{zeta}{estimated coefficients. \code{zeta[1:ncol(z)]} is \eqn{\hat{\alpha}}, 
and \code{zeta[(ncol(z)+1):(ncol(z)+ncol(x))]} is \eqn{\hat{\beta}}.}
\item{y_hat}{ \code{y}'s estimates.}
\item{mse}{mean squared errors between y and \code{y_hat}.}
\item{data}{data information including \code{x}, \code{z}, \code{y}, bandwidth \code{h}, 
initial coefficients \code{zetaini}, iteration step \code{MaxStep}, flag \code{SiMflag}, 
\code{penalty}, \code{lambda} and \code{l1_ratio}.
\code{SiMflag} is TRUE when \code{x} is NULL, otherwise \code{SiMflag} is FALSE.}
\item{Z_alpha}{\eqn{Z^T{\hat{\alpha}}}.}
\item{r_square}{multiple correlation coefficient.}
\item{variance}{variance of \code{y_hat}.}
\item{stdzeta}{standard error of \code{zeta}.}
}
\description{
PPLS along with introducing penalty terms so as to simultaneously
estimate parameters and select important variables in PLSiM 
\deqn{Y = \eta(Z^T\alpha) + X^T\beta + \epsilon}.
}
\examples{
 
# EXAMPLE 1 (INTERFACE=FORMULA)
# To estimate parameters of partially linear single-index model and select 
# variables using different penalization methods such as SCAD, LASSO, ElasticNet.

n = 50
sigma = 0.1

alpha = matrix(1,2,1)
alpha = alpha/norm(alpha,"2")

beta = matrix(4,1,1)

# Case 1: Matrix Input
x = matrix(1,n,1)
z = matrix(runif(n*2),n,2)
y = 4*((z\%*\%alpha-1/sqrt(2))^2) + x\%*\%beta + sigma*matrix(rnorm(n),n,1)

# Compute the penalized profile least-squares estimator with the SCAD penalty
fit_scad = plsim.vs.soft(y~x|z,lambda = 0.01)
summary(fit_scad)

# Compute the penalized profile least-squares estimator with the LASSO penalty
fit_lasso = plsim.vs.soft(y~x|z,lambda = 1e-3, penalty = "LASSO")
summary(fit_lasso)

# Compute the penalized profile least-squares estimator with the ElasticNet penalty
fit_enet = plsim.vs.soft(y~x|z,lambda = 1e-3, penalty = "ElasticNet")
summary(fit_enet)

# Case 2: Vector Input
x = rep(1,n)
z1 = runif(n)
z2 = runif(n) 
y = 4*((z\%*\%alpha-1/sqrt(2))^2) + x\%*\%beta + sigma*matrix(rnorm(n),n,1)

# Compute the penalized profile least-squares estimator with the SCAD penalty
fit_scad = plsim.vs.soft(y~x|z1+z2,lambda = 0.01)
summary(fit_scad)

# Compute the penalized profile least-squares estimator with the LASSO penalty
fit_lasso = plsim.vs.soft(y~x|z1+z2,lambda = 1e-3, penalty = "LASSO")
summary(fit_lasso)

# Compute the penalized profile least-squares estimator with the ElasticNet penalty
fit_enet = plsim.vs.soft(y~x|z1+z2,lambda = 1e-3, penalty = "ElasticNet")
summary(fit_enet)

# EXAMPLE 2 (INTERFACE=DATA FRAME)
# To estimate parameters of partially linear single-index model and select 
# variables using different penalization methods such as SCAD, LASSO, ElasticNet.

n = 50
sigma = 0.1

alpha = matrix(1,2,1)
alpha = alpha/norm(alpha,"2")

beta = matrix(4,1,1)

x = rep(1,n)
z1 = runif(n)
z2 = runif(n) 
X = data.frame(x)
Z = data.frame(z1,z2)

x = data.matrix(X)
z = data.matrix(Z)
y = 4*((z\%*\%alpha-1/sqrt(2))^2) + x\%*\%beta + sigma*matrix(rnorm(n),n,1)

# Compute the penalized profile least-squares estimator with the SCAD penalty
fit_scad = plsim.vs.soft(xdat=X,zdat=Z,ydat=y,lambda = 0.01)
summary(fit_scad)

# Compute the penalized profile least-squares estimator with the LASSO penalty
fit_lasso = plsim.vs.soft(xdat=X,zdat=Z,ydat=y,lambda = 1e-3, penalty = "LASSO")
summary(fit_lasso)

# Compute the penalized profile least-squares estimator with the ElasticNet penalty
fit_enet = plsim.vs.soft(xdat=X,zdat=Z,ydat=y,lambda = 1e-3, penalty = "ElasticNet")
summary(fit_enet)

}
\references{
H. Liang, X. Liu, R. Li, C. L. Tsai. \emph{Estimation and testing for partially linear single-index models}.
Annals of statistics, 2010, 38(6): 3811.
}
