% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Return.portfolio.R
\name{Return.portfolio}
\alias{Return.portfolio}
\alias{Return.rebalancing}
\title{Calculate weighted returns for a portfolio of assets}
\usage{
Return.portfolio(R, weights = NULL, wealth.index = FALSE,
  contribution = FALSE, geometric = TRUE, rebalance_on = c(NA,
  "years", "quarters", "months", "weeks", "days"), value = 1,
  verbose = FALSE, ...)
}
\arguments{
\item{R}{An xts, vector, matrix, data frame, timeSeries or zoo object of
asset returns}

\item{weights}{A time series or single-row matrix/vector containing asset
weights, as decimal percentages, treated as beginning of period weights.  
See Details below.}

\item{wealth.index}{TRUE/FALSE whether to return a wealth index. Default FALSE}

\item{contribution}{if contribution is TRUE, add the weighted return 
contributed by the asset in a given period. Default FALSE}

\item{geometric}{utilize geometric chaining (TRUE) or simple/arithmetic (FALSE)
to aggregate returns. Default TRUE.}

\item{rebalance_on}{Default "none"; alternatively "daily" "weekly" "monthly" 
"annual"  to specify calendar-period rebalancing supported by 
\code{\link[xts]{endpoints}}. Ignored if \code{weights} is an xts object
that specifies the rebalancing dates.}

\item{value}{The beginning of period total portfolio value. This is used for calculating position value.}

\item{verbose}{If verbose is TRUE, return a list of intermediary calculations. 
See Details below.}

\item{\dots}{any other passthru parameters. Not currently used.}
}
\value{
returns a time series of returns weighted by the \code{weights}
parameter, or a list that includes intermediate calculations
}
\description{
Using a time series of returns and any regular or irregular time series of weights
for each asset, this function calculates the returns of a portfolio with the same 
periodicity of the returns data.
}
\details{
By default, this function calculates the time series of portfolio returns given asset
returns and weights. In verbose mode, the function returns a list of intermediary 
calculations that users may find helpful, including both asset contribution and  
asset value through time.

When asset return and weights are matched by period, contribution is simply the 
weighted return of the asset.  c_i = w_i * R_i Contributions are summable across the 
portfolio to calculate the total portfolio return.

Contribution cannot be aggregated through time.  For example, say we have an equal 
weighted portfolio of five assets with monthly returns.  The geometric return of the 
portfolio over several months won't match any aggregation of the individual 
contributions of the assets, particularly if any rebalancing was done during the 
period.

To aggregate contributions through time such that they are summable to the geometric 
returns of the portfolio, the calculation must track changes in the notional value of 
the assets and portfolio.  For example, contribution during a quarter will be 
calculated as the change in value of the position through those three months, divided 
by the original value of the portfolio.  Approaching it this way makes the 
calculation robust to weight changes as well. c_pi = V_(t-p)i - V_t)/V_ti

If the user does not specify weights, an equal weight portfolio is assumed.  
Alternatively, a vector or single-row matrix of weights that matches the length 
of the asset columns may be specified.  In either case, if no rebalancing period is
specified, the weights will be applied at the beginning of the asset time series
and no further rebalancing will take place. If a rebalancing period is specified, 
the portfolio will be rebalanced to the starting weights at the interval specified.

Note that if \code{weights} is an xts object, then any value passed to 
\code{rebalance_on} is ignored. The \code{weights} object specifies the 
rebalancing dates, therefore a regular rebalancing frequency provided via
\code{rebalance_on} is not needed and ignored.

\code{Return.portfolio} will work only on daily or lower frequencies. If you are 
rebalancing intraday, you should be using a trades/prices framework like 
the \code{blotter} package, not a weights/returns framework.

Irregular rebalancing can be done by specifying a time series of weights.  The 
function uses the date index of the weights for xts-style subsetting of rebalancing 
periods.

Weights specified for rebalancing should be thought of as "end-of-period" weights. 
Rebalancing periods can be thought of as taking effect immediately after the close 
of the bar. So, a March 31 rebalancing date will actually be in effect for April 1. 
A December 31 rebalancing date will be in effect on Jan 1, and so forth. This 
convention was chosen because it fits with common usage, and because it simplifies 
xts Date subsetting via endpoints.

In verbose mode, the function returns a list of data and intermediary calculations.
\itemize{
  \item{\code{returns}:}{ The portfolio returns.}
  \item{\code{contribution}:}{ The per period contribution to portfolio 
  return of each asset. Contribution is calculated as BOP weight times the 
  period's return divided by BOP value. Period contributions are summed 
  across the individual assets to calculate portfolio return}
  \item{\code{BOP.Weight}:}{ Beginning of Period (BOP) Weight for each 
  asset. An asset's BOP weight is calculated using the input weights 
  (or assumed weights, see below) and rebalancing parameters given. The next 
  period's BOP weight is either the EOP weights from the prior period or 
  input weights given on a rebalance period.}
  \item{\code{EOP.Weight:}}{ End of Period (BOP) Weight for each asset. 
  An asset's EOP weight is the sum of the asset's BOP weight and 
  contribution for the period divided by the sum of the contributions and 
  initial weights for the portfolio.}
  \item{\code{BOP.Value:}}{ BOP Value for each asset. The BOP value for each 
  asset is the asset's EOP value from the prior period, unless there is a 
  rebalance event.  If there is a rebalance event, the BOP value of the 
  asset is the rebalance weight times the EOP value of the portfolio. That 
  effectively provides a zero-transaction cost change to the position values 
  as of that date to reflect the rebalance.  Note that the sum of the BOP 
  values of the assets is the same as the prior period's EOP portfolio value.}
  \item{\code{EOP.Value:}}{ EOP Value for each asset. The EOP value is for 
  each asset is calculated as (1 + asset return) times the asset's BOP value. 
  The EOP portfolio value is the sum of EOP value across assets.}
}

To calculate BOP and EOP position value, we create an index for each position.  The 
sum of that value across assets represents an indexed value of the total portfolio.  
Note that BOP and EOP position values are only computed when \code{geometric = TRUE}.

From the value calculations, we can calculate different aggregations through time 
for the asset contributions.  Those are calculated as the EOP asset value less the 
BOP asset value; that quantity is divided by the BOP portfolio value.  
Across assets, those will sum to equal the geometric chained returns of the 
portfolio for that same time period.  The function does not do this directly, however.
}
\note{
This function was previously two functions: \code{Return.portfolio} and 
\code{Return.rebalancing}.  Both function names are still exported,
but the code is now common, and \code{Return.portfolio} is probably to be preferred.
}
\examples{

data(edhec)
Return.portfolio(edhec["1997",1:5], rebalance_on="quarters") # returns time series
Return.portfolio(edhec["1997",1:5], rebalance_on="quarters", verbose=TRUE) # returns list
# with a weights object
data(weights) # rebalance at the beginning of the year to various weights through time
chart.StackedBar(weights)
x <- Return.portfolio(edhec["2000::",1:11], weights=weights,verbose=TRUE)
chart.CumReturns(x$returns)
chart.StackedBar(x$BOP.Weight)
chart.StackedBar(x$BOP.Value)

}
\references{
Bacon, C. \emph{Practical Portfolio Performance Measurement and
Attribution}. Wiley. 2004. Chapter 2\cr
}
\seealso{
\code{\link{Return.calculate}} \code{\link[xts]{endpoints}} \cr
}
\author{
Peter Carl, Ross Bennett, Brian Peterson
}
