#' @include Connection.R
NULL

AthenaResult <- function(conn,
                         statement = NULL,
                         s3_staging_dir = NULL){

  stopifnot(is.character(statement))
  Athena <- client_athena(conn)
  Request <- request(conn, statement)
  
  tryCatch(response <- do.call(Athena$start_query_execution, Request, quote = T),
           error = function(e) py_error(e))
  on.exit(if(!is.null(conn@info$expiration)) time_check(conn@info$expiration))
  new("AthenaResult", connection = conn, athena = Athena, info = response)
}

#' @rdname AthenaConnection
#' @export
setClass(
  "AthenaResult",
  contains = "DBIResult",
  slots = list(
    connection = "AthenaConnection",
    athena = "ANY",
    info = "list"
  )
)

#' Clear Results
#' 
#' Frees all resources (local and Athena) associated with result set. It does this by removing query output in AWS S3 Bucket,
#' stopping query execution if still running and removed the connection resource locally.
#' @name dbClearResult
#' @inheritParams DBI::dbClearResult
#' @return \code{dbClearResult()} returns \code{TRUE}, invisibly.
#' @seealso \code{\link[DBI]{dbIsValid}}
#' @examples
#' \donttest{
#' # Note: 
#' # - Require AWS Account to run below example.
#' # - Different connection methods can be used please see `RAthena::dbConnect` documnentation
#' 
#' library(DBI)
#' 
#' # Demo connection to Athena using profile name 
#' con <- dbConnect(RAthena::athena())
#' 
#' res <- dbSendQuery(con, "show databases")
#' dbClearResult(res)
#' 
#' # Check if connection if valid after closing connection
#' dbDisconnect(con)
#' }
#' @docType methods
NULL

#' @rdname dbClearResult
#' @export
setMethod(
  "dbClearResult", "AthenaResult",
  function(res, ...){
    if (!dbIsValid(res)) {
      warning("Result already cleared", call. = FALSE)
      } else {
        
      # checks status of query
      tryCatch(query_execution <- res@athena$get_query_execution(QueryExecutionId = res@info$QueryExecutionId),
               error = function(e) py_error(e))
      
      # stops resource if query is still running
      if (!(query_execution$QueryExecution$Status$State %in% c("SUCCEEDED", "FAILED", "CANCELLED"))){
        tryCatch(res@athena$stop_query_execution(QueryExecutionId = res@info$QueryExecutionId),
                 error = function(e) py_error(e))}
      
      # clear s3 athena output
      tryCatch(s3 <- res@connection@ptr$resource("s3"),
               error = function(e) py_error(e))
        
      # split s3_uri
      s3_info <- split_s3_uri(res@connection@info$s3_staging)
      result_info <- split_s3_uri(query_execution$QueryExecution$ResultConfiguration$OutputLocation)
      
      tryCatch(bucket <- s3$Bucket(s3_info$bucket),
               error = function(e) py_error(e))
      
      # remove class pointers
      eval.parent(substitute(res@connection@ptr <- NULL))
      eval.parent(substitute(res@athena <- NULL))
      
      # Out put Python error as warning if s3 resource can't be dropped
      tryCatch(s3$Object(s3_info$bucket, paste0(result_info$key, ".metadata"))$delete(),
               error = function(e) py_warning(e))
      tryCatch(s3$Object(s3_info$bucket, result_info$key)$delete(),
               error = function(e) cat(""))
      }
    invisible(TRUE)
  })

#' Fetch records from previously executed query
#' 
#' Currently returns the top n elements (rows) from result set or returns entire table from Athena.
#' @name dbFetch
#' @param n maximum number of records to retrieve per fetch. Use \code{n = -1} or \code{n = Inf} to retrieve all pending records.
#'          Some implementations may recognize other special values. Currently chunk sizes range from 0 to 999, 
#'          if entire dataframe is required use \code{n = -1} or \code{n = Inf}.
#' @inheritParams DBI::dbFetch
#' @return \code{dbFetch()} returns a data frame.
#' @seealso \code{\link[DBI]{dbFetch}}
#' @examples
#' \donttest{
#' # Note: 
#' # - Require AWS Account to run below example.
#' # - Different connection methods can be used please see `RAthena::dbConnect` documnentation
#' 
#' library(DBI)
#' 
#' # Demo connection to Athena using profile name 
#' con <- dbConnect(RAthena::athena())
#' 
#' res <- dbSendQuery(con, "show databases")
#' dbFetch(res)
#' dbClearResult(res)
#' 
#' # Disconnect from Athena
#' dbDisconnect(con)
#' }
#' @docType methods
NULL

#' @rdname dbFetch
#' @export
setMethod(
  "dbFetch", "AthenaResult",
  function(res, n = -1, ...){
    if (!dbIsValid(res)) {stop("Result already cleared", call. = FALSE)}
    # check status of query
    result <- poll(res)
    
    result_info <- split_s3_uri(result$QueryExecution$ResultConfiguration$OutputLocation)
    
    # if query failed stop
    if(result$QueryExecution$Status$State == "FAILED") {
      stop(result$QueryExecution$Status$StateChangeReason, call. = FALSE)
    }
  
    if(n >= 0 && n !=Inf){
      n = as.integer(n + 1)
      tryCatch(result <- res@athena$get_query_results(QueryExecutionId = res@info$QueryExecutionId, MaxResults = n),
               error = function(e) py_error(e))
      
      output <- lapply(result$ResultSet$Rows, function(x) (sapply(x$Data, function(x) if(length(x) == 0 ) NA else x)))
      
      dt <- rbindlist(output)
      colnames(dt) <- as.character(unname(dt[1,]))
      rownames(dt) <- NULL
      return(dt[-1,])
    }

    #create temp file
    File <- tempfile()
    on.exit(unlink(File))
    
    # connect to s3 and create a bucket object
    tryCatch({s3 <- res@connection@ptr$resource("s3")},
             error = function(e) py_error(e))
    
    # download athena output
    tryCatch(s3$Bucket(result_info$bucket)$download_file(result_info$key, File),
             error = function(e) py_error(e))
    
    # return metadata of athena data types
    tryCatch(result_class <- res@athena$get_query_results(QueryExecutionId = res@info$QueryExecutionId, MaxResults = as.integer(1)),
             error = function(e) py_error(e))
    
    Type2 <- Type <- AthenaToRDataType(result_class$ResultSet$ResultSetMetadata$ColumnInfo)
    # Type2 is to handle issue with data.table fread 
    Type2[Type2 %in% "POSIXct"] <- "character"
    
    if(grepl("\\.csv$",result_info$key)){
      # currently parameter data.table is left as default. If users require data.frame to be returned then parameter will be updated
      output <- data.table::fread(File, col.names = names(Type2), colClasses = unname(Type2), sep = ",", showProgress = F, na.strings="")
      # formatting POSIXct: from string to POSIXct
      for (col in names(Type[Type %in% "POSIXct"])) set(output, j=col, value=as.POSIXct(output[[col]]))
      # AWS Athena returns " values as "". Due to this "" will be reformatted back to "
      for (col in names(Type[Type %in% "character"])) set(output, j=col, value=gsub('""' , '"', output[[col]]))
    } else{
      file_con <- file(File)
      output <- suppressWarnings(readLines(file_con))
      close(file_con)
      if(any(grepl("create|table", output, ignore.case = T))){
        output <- data.frame("TABLE_DDL" = paste0(output, collapse = "\n"), stringsAsFactors = FALSE)
      } else (output <- data.frame(var1 = trimws(output), stringsAsFactors = FALSE))
    }
    return(output)
  })

#' Completion status
#' 
#' This method returns if the query has completed. 
#' @name dbHasCompleted
#' @inheritParams DBI::dbHasCompleted
#' @return \code{dbHasCompleted()} returns a logical scalar. \code{TRUE} if the query has completed, \code{FALSE} otherwise.
#' @seealso \code{\link[DBI]{dbHasCompleted}}
#' @examples
#' \donttest{
#' # Note: 
#' # - Require AWS Account to run below example.
#' # - Different connection methods can be used please see `RAthena::dbConnect` documnentation
#' 
#' library(DBI)
#' 
#' # Demo connection to Athena using profile name 
#' con <- dbConnect(RAthena::athena())
#' 
#' # Check if query has completed
#' res <- dbSendQuery(con, "show databases")
#' dbHasCompleted(res)
#'
#' dbClearResult(res)
#' 
#' # Disconnect from Athena
#' dbDisconnect(con)
#' }
#' @docType methods
NULL

#' @rdname dbHasCompleted
#' @export
setMethod(
  "dbHasCompleted", "AthenaResult",
  function(res, ...) {
    if (!dbIsValid(res)) {stop("Result already cleared", call. = FALSE)}
    tryCatch(query_execution <- res@athena$get_query_execution(QueryExecutionId = res@info$QueryExecutionId),
             error = function(e) py_error(e))
    
    if(query_execution$QueryExecution$Status$State %in% c("SUCCEEDED", "FAILED", "CANCELLED")) TRUE
    else if (query_execution$QueryExecution$Status$State == "RUNNING") FALSE
  })

#' @rdname dbIsValid
#' @export
setMethod(
  "dbIsValid", "AthenaResult",
  function(dbObj, ...){
    resource_active(dbObj)
  }
)

#' @rdname dbGetInfo
#' @inheritParams DBI::dbGetInfo
#' @export
setMethod(
  "dbGetInfo", "AthenaResult",
  function(dbObj, ...) {
    if (!dbIsValid(dbObj)) {stop("Result already cleared", call. = FALSE)}
    info <- dbObj@info
    info
  })


#' Information about result types
#' 
#' Produces a data.frame that describes the output of a query. 
#' @name dbColumnInfo
#' @inheritParams DBI::dbColumnInfo
#' @return \code{dbColumnInfo()} returns a data.frame with as many rows as there are output fields in the result.
#'         The data.frame has two columns (field_name, type).
#' @seealso \code{\link[DBI]{dbHasCompleted}}
#' @examples
#' \donttest{
#' # Note: 
#' # - Require AWS Account to run below example.
#' # - Different connection methods can be used please see `RAthena::dbConnect` documnentation
#' 
#' library(DBI)
#' 
#' # Demo connection to Athena using profile name 
#' con <- dbConnect(RAthena::athena())
#' 
#' # Get Column information from query
#' res <- dbSendQuery(con, "select * from information_schema.tables")
#' dbColumnInfo(res)
#' dbClearResult(res)
#'  
#' # Disconnect from Athena
#' dbDisconnect(con)
#' }
#' @docType methods
NULL

#' @rdname dbColumnInfo
#'@export
setMethod(
  "dbColumnInfo", "AthenaResult",
  function(res, ...){
    if (!dbIsValid(res)) {stop("Result already cleared", call. = FALSE)}
    result <- poll(res)
    if(result$QueryExecution$Status$State == "FAILED") {
      stop(result$QueryExecution$Status$StateChangeReason, call. = FALSE)
    }
    
    tryCatch(result <- res@athena$get_query_results(QueryExecutionId = res@info$QueryExecutionId, MaxResults = as.integer(1)),
             error = function(e) py_error(e))
    
    Name <- sapply(result$ResultSet$ResultSetMetadata$ColumnInfo, function(x) x$Name)
    Type <- sapply(result$ResultSet$ResultSetMetadata$ColumnInfo, function(x) x$Type)
    data.frame(field_name = Name,
               type = Type, stringsAsFactors = F)
  }
)
