\name{NCEP.vis.points}
\alias{NCEP.vis.points}

\title{ Visualize Weather Data Interpolated to a Point on a Map }

\description{ This function creates a map with points.  The color of the points indicates the value of some variable at that point.  These values can e.g. be obtained by applying the function \code{\link{NCEP.interp}}. } 

\usage{
NCEP.vis.points(wx, lats, lons, cols=heat.colors(64),
    transparency=.5, connect=TRUE, axis.args=NULL, 
    points.args=NULL, map.args=NULL, grid.args=NULL, 
    title.args=NULL, image.plot.args=NULL, lines.args=NULL)
}

\arguments{
  \item{wx}{ A vector of weather data as returned by \code{\link{NCEP.interp}} }
  \item{lats}{ A vector of latitudes in decimal degrees indicating the locations of the points }
  \item{lons}{ A vector of longitudes in decimal degrees indicating the locations of the points }
  \item{cols}{ A vector of colors such as that generated by \code{\link[grDevices]{rainbow}}, \code{\link[grDevices]{heat.colors}}, \code{\link[grDevices]{topo.colors}}, \code{\link[grDevices]{terrain.colors}}, or similar functions }  
  \item{transparency}{ A numeric value between 0 and 1 indicating the transparency of the filled points on the map.}
  \item{connect}{ Logical.  Should a line be drawn connecting the points?}
  \item{axis.args}{ A list of arguments controlling the drawing of axes. See \code{\link{axis}} for acceptable arguments and the examples below for a demonstration. }
  \item{points.args}{A list of arguments controlling the drawing of points. See \code{\link{points}} for acceptable arguments and the examples below for a demonstration. }  
  \item{map.args}{ A list of arguments controlling the drawing of the map. See \code{\link[maps]{map}} for acceptable arguments and the examples below for a demonstration.}
  \item{grid.args}{ A list of arguments controlling the drawing of the lat/long grid lines. See \code{\link{abline}} for acceptable arguments and the examples below for a demonstration. }
  \item{title.args}{ A list of arguments controlling the how titles and axis lables are written. See \code{\link{title}} for acceptable arguments and the examples below for a demonstration. }
  \item{image.plot.args}{ A list of arguments controlling the plotting of the color-bar legend and the legend axis and labels. See \code{\link[fields]{image.plot}} for acceptable arguments and the examples below for a demonstration. }
  \item{lines.args}{ A list of arguments controlling the drawing of the line connecting the points. See \code{\link{lines}} for acceptable arguments and the examples below for a demonstration. }

  }

\details{
Most of the components of a plot produced by this function can be controlled by supplying a list of arguments to the embedded function that produces the particular component of the plot.
For example, the text and size of the plot's title can be controlled by specifying a list of acceptable arguments to \code{title.args}.
Similarly, the axes, map, and grid lines are controlled by specifying a list of acceptable arguements to \code{axis.args}, \code{map.args}, and \code{grid.args}, respectively.
Through the argument \code{image.plot.args} the user can control the plotting of the color-bar legend and the color-bar's title and axis labels.
See the examples below for a demonstration of how to apply these different arguments.
}

\value{ A plot is produced.  No data are returned. }

\references{ Kemp, M. U., van Loon, E. E., Shamoun-Baranes, J., and Bouten, W. 2011. RNCEP:global weather and climate data at your fingertips. -- Ecography submitted. }

\author{ Michael U. Kemp \email{M.U.Kemp@UvA.nl} }

\examples{
library(RNCEP)
## In this example, we use datetime and locational data
## obtained from a GPS device attached to a lesser 
## black-backed gull. 
data(gull, package='RNCEP')

## First, visualize the entire track representing altitude
## with the point colors ##
## Note the specification of the title
## Also, note the specification of the legend label
## and adjustment of its placement
NCEP.vis.points(wx=gull$altitude, lats=gull$latitude, 
    lons=gull$longitude, cols=topo.colors(64),
    title.args=list(main='Lesser black-backed gull'),
    image.plot.args=list(legend.args=list(text='Altitude',
    adj=-1, cex=1.25)))

## Take a subset of the data based on the datetime of 
## the measurement ##
ss <- subset(gull, format(gull$datetime, "\%Y-\%m-\%d \%H:\%M:\%S") >=
    "2008-09-19 16:00:00" & format(gull$datetime, 
    "\%Y-\%m-\%d \%H:\%M:\%S")	<= "2008-09-19 19:30:00")


## Now collect cloud cover, temperature, and wind
## information for each point in the subset ##
cloud <- NCEP.interp(variable='tcdc.eatm', level='gaussian', 
    lat=ss$latitude, lon=ss$longitude, dt=ss$datetime, 
    reanalysis2=TRUE, keep.unpacking.info=TRUE)
temp <- NCEP.interp(variable='air.sig995', level='surface', 
    lat=ss$latitude, lon=ss$longitude, dt=ss$datetime,
    reanalysis2=FALSE, keep.unpacking.info=TRUE)
uwind <- NCEP.interp(variable='uwnd', level=925, 
    lat=ss$latitude, lon=ss$longitude, dt=ss$datetime,
    reanalysis2=TRUE, keep.unpacking.info=TRUE)
vwind <- NCEP.interp(variable='vwnd', level=925, 
    lat=ss$latitude, lon=ss$longitude, dt=ss$datetime, 
    reanalysis2=TRUE, keep.unpacking.info=TRUE)	

## Now visualize the subset of the GPS track using color
## to indicate the cloud cover ##
## Note the adjustment to the color of the basemap
## And the setting of the map range ##
## And the explicit placement of the colorbar legend
## using the smallplot argument
NCEP.vis.points(wx=cloud, lats=ss$latitude, lons=ss$longitude,
    cols=rev(heat.colors(64)),
    map.args=list(col='darkgreen',xlim=c(-7,4), ylim=c(40,50)),
    title.args=list(main='Lesser black-backed gull'),
    image.plot.args=list(legend.args=list(text='Cloud Cover \%',
        adj=-.1, padj=-.5, cex=1),
    smallplot=c(.83,.86,.15,.85)))

## Now visualize the subset of the GPS track using color
## to indicate the temperature ##
## Note the adjustment of point size
NCEP.vis.points(wx=temp, lats=ss$latitude, lons=ss$longitude,
    cols=rev(heat.colors(64)),
    points.args=list(cex=1.25),
    title.args=list(main='Lesser black-backed gull'),
    image.plot.args=list(legend.args=list(text='Kelvin',
        adj=-.4, padj=-.5, cex=1.15)),
    map.args=list(xlim=c(-7,4), ylim=c(40,50)))

## Now calculate the tailwind component from the U and V
## wind components assuming that the bird's preferred 
## direction is 225 degrees
tailwind <- (sqrt(uwind^2 + vwind^2)*cos(((atan2(uwind,vwind)*
    (180/pi))-225)*(pi/180)))

## Now visualize the subset of the GPS track using color
## to indicate the tailwind speed ##
## Note the adjustment of grid and axis properties
NCEP.vis.points(wx=tailwind, lats=ss$latitude, lons=ss$longitude,
    cols=rev(heat.colors(64)),
    axis.args=list(las=2), grid.args=list(lty=1),
    title.args=list(main='Lesser black-backed gull'),
    image.plot.args=list(legend.args=list(text='Tailwind m/s',
        adj=0, padj=-2, cex=1.15)),
    map.args=list(xlim=c(-7,4), ylim=c(40,50)))
}