% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/longevity.R
\name{longevity}
\alias{longevity}
\title{Calculate longevity from a matrix population model}
\usage{
longevity(matU, start = 1L, x_max = 1000, lx_crit = 0.01)
}
\arguments{
\item{matU}{The survival component of a matrix population model (i.e. a
square projection matrix reflecting survival-related transitions; e.g.
progression, stasis, and retrogression). Optionally with named rows and
columns indicating the corresponding life stage names.}

\item{start}{The index (or stage name) of the first stage at which the author
considers the beginning of life. Defaults to 1. Alternately, a numeric vector
giving the starting population vector (in which case \code{length(start)}
must match \code{ncol(matU))}. See section \emph{Starting from multiple stages}.}

\item{x_max}{The maximum age, in units of the MPM projection interval, to
which survivorship will be calculated. Defaults to 1000.}

\item{lx_crit}{Proportion of initial cohort remaining before all are considered
dead (a value between 0 and 1). Defaults to 0.01.}
}
\value{
Returns longevity, the integer age at which expected survivorship
  falls below \code{lx_crit}. If survivorship doesn't reach \code{lx_crit} by
  \code{x_max}, returns \code{NA} and prints a warning message.
}
\description{
Calculate longevity (the age \emph{x} at which survivorship for a synthetic cohort falls below some 
critical proportion) from a matrix population model
}
\section{Starting from multiple stages}{

Rather than specifying argument \code{start} as a single stage class from
which all individuals start life, it may sometimes be desirable to allow for
multiple starting stage classes. For example, if we want to start our
calculation of longevity from reproductive maturity (i.e. first
reproduction), we should account for the possibility that there may be
multiple stage classes in which an individual could first reproduce.

To specify multiple starting stage classes, specify argument \code{start} as
the desired starting population vector, giving the proportion
of individuals starting in each stage class (the length of \code{start}
should match the number of columns in the relevant MPM).
}

\examples{
data(mpm1)

longevity(mpm1$matU, start = 2)
longevity(mpm1$matU, start = "small")  # equivalent using named life stages
longevity(mpm1$matU, start = 2, lx_crit = 0.05)

# starting from first reproduction
repstages <- repro_stages(mpm1$matF)
n1 <- mature_distrib(mpm1$matU, start = 2, repro_stages = repstages)
longevity(mpm1$matU, start = n1)

}
\references{
Caswell, H. 2001. Matrix Population Models: Construction,
  Analysis, and Interpretation. Sinauer Associates; 2nd edition. ISBN:
  978-0878930968

  Morris, W. F. & Doak, D. F. 2003. Quantitative Conservation Biology:
  Theory and Practice of Population Viability Analysis. Sinauer Associates,
  Sunderland, Massachusetts, USA
}
\seealso{
\code{\link{mature_distrib}} for calculating the proportion of
individuals achieving reproductive maturity in each stage class.

Other life history traits: 
\code{\link{entropy_d}()},
\code{\link{entropy_k}()},
\code{\link{gen_time}()},
\code{\link{life_expect_mean}()},
\code{\link{net_repro_rate}()},
\code{\link{repro_maturity}},
\code{\link{shape_rep}()},
\code{\link{shape_surv}()}
}
\author{
Roberto Salguero-Gomez <rob.salguero@zoo.ox.ac.uk>

Hal Caswell <hcaswell@whoi.edu>
}
\concept{life history traits}
