% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/project.R
\name{project}
\alias{project}
\title{Project a linear factor model}
\usage{
project(A, w, nonneg = TRUE, L1 = 0, threads = 0, ...)
}
\arguments{
\item{A}{sparse matrix of features x samples, of or coercible to class \code{Matrix::dgCMatrix}}

\item{w}{dense matrix of features x factors giving the linear model to be projected}

\item{nonneg}{apply non-negativity constraints}

\item{L1}{L1/LASSO penalty to be applied to \eqn{h}. Generally should be scaled to \code{max(b)} where \eqn{b = WA_j} for all columns \eqn{j} in \eqn{A}}

\item{threads}{number of CPU threads for parallelization, default \code{0} for all available threads.}

\item{...}{advanced parameters, see details}
}
\value{
matrix \eqn{h}
}
\description{
Solves the equation \eqn{A = wh} for \eqn{h}
}
\details{
For the classical alternating matrix factorization update problem \eqn{A = wh}, the updates
(or projection) of \eqn{h} is given by the equations: \deqn{w^Twh = wA_j}
in the form \eqn{ax = b} where \eqn{a = w^Tw} \eqn{x = h} and \eqn{b = wA_j} for all columns \eqn{j} in \eqn{A}.

Given \eqn{A} and \eqn{w}, \code{RcppML::project} solves for \eqn{h} using the above formulas and
\code{RcppML::nnls}.

The corresponding equation for updating \eqn{w} in block-pivoting is: \deqn{hh^Tw^T = hA^T_j}

Thus, one may also solve for \eqn{w} by inputting the transpose of \eqn{A} and \eqn{h} in place of \eqn{w}.
}
\section{Advanced parameters}{

Several parameters hidden in the \code{...} argument may be adjusted (although defaults should entirely satisfy) in addition to those documented explicitly:
\itemize{
\item \code{cd_maxit}, default 1000. Maximum number of coordinate descent iterations for solution refinement after initialization with solution from previous iteration. Only used as stopping criterion if \code{cd_tol} is not satisfied previously. See \code{\link{nnls}}.
\item \code{fast_maxit}, default 10. Maximum number of FAST iterations for finding an approximate solution to initialize coordinate descent. See \code{\link{nnls}}.
\item \code{cd_tol}, default 1e-8. Stopping criterion for coordinate descent iterations given by the maximum relative change in any coefficient between consecutive solutions. See \code{\link{nnls}}.
}
}

\examples{
library(Matrix)
w <- matrix(runif(1000 * 10), 1000, 10)
h_true <- matrix(runif(10 * 100), 10, 100)
A <- (w \%*\% h_true) * (rsparsematrix(1000, 100, 0.5) > 0)
h <- project(A, w)
cor(as.vector(h_true), as.vector(h))

# alternating projections refine solution (like NMF)
mse(A, w, h = h) # mse before alternating updates
h <- project(A, w)
w <- project(t(A), t(h))
h <- project(A, w)
w <- project(t(A), t(h))
h <- project(A, w)
w <- t(project(t(A), t(h)))
mse(A, w, h = h) # mse after alternating updates

}
\seealso{
\code{\link{nnls}}, \code{\link{nmf}}
}
\author{
Zach DeBruine
}
