% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/exceedance.R
\name{exceedance}
\alias{exceedance}
\title{Detect consecutive days in exceedance of a given threshold.}
\usage{
exceedance(data, threshold = 20, below = FALSE, min_duration = 5,
  join_across_gaps = TRUE, max_gap = 2, max_pad_length = 3)
}
\arguments{
\item{data}{A data frame with at least the two following columns:
a \code{date} column which is a vector of dates of class \code{Date},
and a \code{temp} column, which is the temperature on those given
dates. The function will not accurately detect consecutive days of
temperatures in exceedance of the \code{threshold} if missing days of
data are not filled in with \code{NA}. Data of the appropriate format are created
by the function \code{\link{make_whole}}, but your own data may be used
directly if they meet the given criteria.}

\item{threshold}{The static threshold used to determine how many consecutive
days are in exceedance of the temperature of interest. Default is
\code{20} degrees.}

\item{below}{Default is \code{FALSE}. When set to TRUE, consecutive days of temperature
below the \code{threshold} variable are calculated. When set to FALSE,
consecutive days above the \code{threshold} variable are calculated.}

\item{min_duration}{Minimum duration that temperatures must be in exceedance
of the \code{threshold} variable. Default is \code{5} days.}

\item{join_across_gaps}{A TRUE/ FALSE statement that indicates whether
or not to join consecutive days of temperatures in exceedance of the
\code{threshold} across a small gap between groups before/after a short
gap as specified by \code{max_gap}. Default is \code{TRUE}.}

\item{max_gap}{The maximum length of the gap across which to connect
consecutive days in exceedance of the \code{threshold} when
\code{join_across_gaps} is \code{TRUE}.}

\item{max_pad_length}{Specifies the maximum length of days over which to
interpolate (pad) missing data (specified as \code{NA}) in the input
temperature time series; i.e., any consecutive blocks of NAs with length
greater than \code{max_pad_length} will be left as \code{NA}. Set as an
integer. Default is \code{3} days.}
}
\value{
The function will return a list of two components. The first being
\code{threshold}, which shows the daily temperatures and on which specific days
the given \code{threshold} was exceeded. The second component of the list is
\code{exceedance}, which shows a medley of statistics for each discrete group
of days in exceedance of the given \code{threshold}. Note that any additional
columns left in the data frame given to this function will be output in the
\code{threshold} component of the output. For example, if one uses
\code{\link{make_whole}} to prepare a time series for analysis and leaves
in the \code{doy} column, this column will appear in the output.

The information shown in the \code{threshold} component is:
  \item{date}{The date of the temperature measurement.}
  \item{temp}{Temperature on the specified date [deg. C].}
  \item{thresh}{The static \code{threshold} chosen by the user [deg. C].}
  \item{thresh_criterion}{Boolean indicating if \code{temp} exceeds
  \code{threshold}.}
  \item{duration_criterion}{Boolean indicating whether periods of consecutive
  \code{thresh_criterion} are >= \code{min_duration}.}
  \item{exceedance}{Boolean indicting if all criteria that define a discrete
  group in exceedance of the \code{threshold} are met.}
  \item{exceedance_no}{A sequential number indicating the ID and order of
  occurence of exceedances.}

The individual exceedances are summarised using the following metrics:
  \item{index_start}{Row number on which exceedance starts.}
  \item{index_stop}{Row number on which exceedance stops.}
  \item{exceedance_no}{The same sequential number indicating the ID and
  order of the exceedance as found in the \code{threshold} component of the
  output list.}
  \item{duration}{Duration of exceedance [days].}
  \item{date_start}{Start date of exceedance [date].}
  \item{date_stop}{Stop date of exceedance [date].}
  \item{date_peak}{Date of exceedance peak [date].}
  \item{int_mean}{Mean intensity [deg. C].}
  \item{int_max}{Maximum (peak) intensity [deg. C].}
  \item{int_var}{Intensity variability (standard deviation) [deg. C].}
  \item{int_cum}{Cumulative intensity [deg. C x days].}
  \item{rate_onset}{Onset rate of exceedance [deg. C / day].}
  \item{rate_decline}{Decline rate of exceedance [deg. C / day].}

\code{int_max_abs}, \code{int_mean_abs}, \code{int_var_abs}, and
\code{int_cum_abs} are as above except as absolute magnitudes
rather than relative to the threshold.
}
\description{
Detect consecutive days in exceedance of a given threshold.
}
\details{
\enumerate{
\item This function assumes that the input time series consists of continuous
daily temperatures, with few missing values. The accompanying function
\code{\link{make_whole}} aids in the preparation of a time series that is
suitable for use with \code{exceedance}, although this may also be accomplished
'by hand' as long as the criteria are met as discussed in the documentation
to \code{\link{make_whole}}.
\item Future versions seek to accomodate monthly and annual time series, too.
\item The calculation of onset and decline rates assumes that exceedance of the
\code{threshold} started a half-day before the start day and ended a half-day
after the end-day. This is consistent with the duration definition as implemented,
which assumes duration = end day - start day + 1.
\item For the purposes of exceedance detection, any missing temperature values not
interpolated over (through optional \code{max_pad_length}) will remain as
\code{NA}. This means they will trigger the end of an exceedance if the adjacent
temperature values are in exceedance of the \code{threshold}.
\item If the function is used to detect consecutive days of temperature under
the given \code{theshold}, these temperatures are then taken as being in
exceedance below the \code{threshold} as there is no antonym in the English
language for 'exceedance'.
}
This function is based largely on the \code{detect} function found in this
package, which was ported from the Python algorithm that was written by Eric
Oliver, Institute for Marine and Antarctic Studies, University of Tasmania,
Feb 2015, and is documented by Hobday et al. (2016).
}
\examples{
t_dat <- make_whole(sst_WA)
res <- exceedance(t_dat, threshold = 25)
# show first ten days of daily data:
res$threshold[1:10, ]
# show first five exceedances:
res$exceedance[1:5, ]
}
\references{
Hobday, A.J. et al. (2016). A hierarchical approach to defining
marine heatwaves, Progress in Oceanography, 141, pp. 227-238,
doi:10.1016/j.pocean.2015.12.014
}
\author{
Robert W. Schlegel, Albertus J. Smit, Eric C. J. Oliver
}
