#############################################################################
#
#  This file is a part of the R package "RoughSets".
#
#  Author: Lala Septem Riza and Andrzej Janusz
#  Supervisors: Chris Cornelis, Francisco Herrera, Dominik Slezak and Jose Manuel Benitez
#  Copyright (c):
#       DiCITS Lab, Sci2s group, DECSAI, University of Granada and
#       Institute of Mathematics, University of Warsaw
#
#  This package is free software: you can redistribute it and/or modify it under
#  the terms of the GNU General Public License as published by the Free Software
#  Foundation, either version 2 of the License, or (at your option) any later version.
#
#  This package is distributed in the hope that it will be useful, but WITHOUT
#  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
#  A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
#############################################################################
#' It is a additional function used to import dataset from files and then construct them into the standard decision table. 
#'
#' It is used to read a dataset from files with the following format: ".csv", ".txt", ".dat", and ".xls", into decision table format. 
#' The data should be in table format containing rows and columns without header, where every row represents 
#' every object/instance while columns represent attributes of the objects. 
#' 
#' The output of this function is a decision table which fulfills a standard format of RoughSets package 
#' (See \code{\link{SF.asDecisionTable}}). 
#' In order to construct the decision table, especially in determining description of attributes, this function uses simple heuristic techniques as follows:
#' \itemize{
#' \item string values: they will be recognized as nominal/symbolic values. 
#' \item integer values: they will be recognized as continuous/real values.
#' \item decimal/real values: they will be recognized as continuous/real values. 
#' \item indx.nominal: the attributes that are set in this parameter will be assigned as nominal values. 
#' }
#'
#' @title The importing function
#' @param filename a file name that contains objects/data. 
#' @param decision.attr a index position of decision attribute. When we ignore this parameter,  
#'        the function will treat the data as an information system or newdata/testing data. In other words,
#'        we must define the index of decision attribute if we have the decision attribute in the dataset. 
#' @param indx.nominal a indexes of nominal values. It is used to define specific indexes as nominal attributes.
#' @param ... other parameters which are involved in \code{read.table} function. See \code{\link{read.table}}.
#' @return A \code{"DecisionTable"} class which is the standard decision table. See \code{\link{SF.asDecisionTable}}.
#' @examples
#' #############################################################
#' ## Example 1: dataset saved in a file
#' #############################################################
#' ## Let us assume we have the following data which has been already saved to the file "tes.dat"
#' data <- data.frame(c(0.12, 0.23, 0.24), c(1,3,2), c(10, 12, 18), c("a", "a", "b"), c(1, 1, 2))
#' \dontrun{write.table(data, file = "tes.dat", row.names = FALSE, col.names = FALSE, 
#'                     fileEncoding ="")}
#' 
#' ## Then we would generate decision table from tes.dat file.
#' ## in this case, we want to define that second and third attributes are nominal and continuous,
#' ## respectively.
#' \dontrun{decision.table <- SF.read.DecisionTable(filename = "tes.dat", decision.attr = 4, 
#'                   indx.nominal = 2, sep= " ", col.names = c("v1", "v2", "v3", "v4", "o1"))}
#' 
SF.read.DecisionTable <- function(filename, decision.attr = NULL, indx.nominal = NULL, ...) {
  
  if(is.null(decision.attr)) {
    warning("A decision attribute is not indicated - the data will be treated as an information system.")
  }
  
  dataset <- read.table(file = filename, ...)
  decision.table = SF.asDecisionTable(dataset, decision.attr = decision.attr, indx.nominal = indx.nominal)
  return(decision.table)
}

#' It is a function used to construct a standard decision table in the \code{RoughSets} package. So, users are strongly recommended to 
#' call this function first before any other functions in this package. 
#'
#' The \code{RoughSets} package requires a standard decision table to calculate a particular task. 
#' The standard decision table is a \code{"DecisionTable"} class which constitutes the set of all objects/instances (universe of discourse) and other
#' attributes. The objects are built by a table or data frame where the number of rows and columns represent the number of objects/instances and attributes, respectively. 
#' Three important attributes that describe the decision table are as follows.
#' \itemize{
#'   \item \code{desc.attrs}: a list containing the names of attributes and range of the data. There are two kinds of representation 
#'                            in this parameters which depend on whether the attributes are 
#'                            in symbolic/nominal or real values, for example:
#'                            \itemize{
#'                              \item symbolic/nominal attribute: \code{a = c(1,2,3)} means the attribute \code{a} has values 1, 2, and 3.
#'                              \item real/continuous attribute: \code{a = c(10, 20)} means the attribute \code{a} has values between 10 and 20.
#'                            }
#'   \item \code{nominal.attrs}: a vector containing boolean values that show whether 
#'                            the related attributes is a nominal value or not. 
#'                            For example: 
#'
#'                            \code{nominal.attrs = c(FALSE, TRUE, FALSE)} means that the first and third attributes
#'                            are continuous attributes and otherwise for the second one.
#'  \item \code{decision.attr}: a numeric value representing an index of the decision attribute. Users must define the index of the decision attribute
#'                            when they want to construct a decision table or training dataset. But, it is a null value when constructing a testing dataset/newdata. 
#'                            It is strongly recommended that the index of decision attribute is 
#'         					  put on the last column of the dataset.                         
#' }  
#' It also has been provided the function \code{\link{SF.read.DecisionTable}} which is used to import from a file and then construct into the decision table. 
#'
#' @title The construction function 
#' @param dataset a data frame representing the dataset that contains objects/instances and attributes/features in its rows and columns, respectively. 
#'        See in Section \code{Details}.
#' @param decision.attr a numeric value representing the index position of the decision attribute. If we ignore this parameter, then 
#'        the function will treat the data as an information system or newdata/testing data. In other words,
#'        we must define the index of decision attribute if we have the decision attribute in the dataset. 
#' @param indx.nominal a vector representing indexes of nominal values. It is used to define specific indexes as nominal attributes.
#' @return A \code{"DecisionTable"} class which is the standard decision table.
#' @examples
#' ################################################################
#' ## Example : converting from datasets in data.frame 
#' ##            into decision table
#' ################################################################
#' ## Let use iris which is available in R be dataset
#' decision.table <- SF.asDecisionTable(dataset = iris, decision.attr = 5,
#'                   indx.nominal = 5)
#' @export
SF.asDecisionTable <- function(dataset, decision.attr = NULL, indx.nominal = NULL) {
  
  nominal.attrs = rep(FALSE, ncol(dataset))
  if (length(indx.nominal) > 0) {
		nominal.attrs[indx.nominal] = TRUE
  }
  
  ## assign nominal.attrs showing the nominal of attributes
  class.vector = sapply(dataset, class)
  nominal.attrs[class.vector %in% c("factor", "character")] = TRUE
  
  ## construct desc.attrs as describe of attributes
  desc.attrs = list()
  desc.attrs[1:ncol(dataset)] = numeric(1)
  indx.nominal = which(nominal.attrs)
  if(length(indx.nominal) > 0) {
		desc.attrs[indx.nominal] = lapply(dataset[indx.nominal], function(x) {tmp = unique(x);
                                                                       tmp = tmp[order(tmp)];
                                                                       as.character(tmp)})
    if(sum(nominal.attrs) != ncol(dataset)) {
      desc.attrs[which(!nominal.attrs)] = lapply(dataset[which(!nominal.attrs)], range)
    }
  } else {
    desc.attrs = lapply(dataset, range)
  }
  
  ## construct the class "DecisionTable"
  names(desc.attrs) <- colnames(dataset)	
  decision.table = dataset
  attr(decision.table, "nominal.attrs") = nominal.attrs
  attr(decision.table, "desc.attrs") = desc.attrs
  attr(decision.table, "decision.attr") = decision.attr 
  decision.table = ObjectFactory(decision.table, "DecisionTable")
  
  return(decision.table)
}

#' This function enables the output of a summary of the rule induction methods. 
#'
#' @title The summary function of rules based on FRST
#' 
#' @param object a \code{"RuleSetFRST"} object. See \code{\link{RI.hybridFS.FRST}} and \code{\link{RI.GFRS.FRST}}.
#' @param ... the other parameters.
#' @return a description that contains the following information. For FRST model: 
#' \itemize{
#' \item The type of the considered model.
#' \item The type of the considered method.
#' \item The type of the considered task.
#' \item The type of similarity.
#' \item The type of triangular norm.
#' \item The names of attributes and their type (whether nominal or not).
#' \item The interval of the data.
#' \item the variance values of the data.
#' \item The rules. Every rule constitutes two parts which are IF and THEN parts. 
#'       For example, \code{"IF pres is around 90 and preg is around 8 THEN class is 2"}.
#'       See \code{\link{RI.GFRS.FRST}}.
#' }
#' @examples
#' ###########################################################
#' ## Example 1: Regression problem
#' ###########################################################
#' data(RoughSetData)
#' decision.table <- RoughSetData$housing7.dt
#'
#' control <- list(type.aggregation = c("t.tnorm", "lukasiewicz"), type.relation = 
#'                 c("tolerance", "eq.3"), t.implicator = "lukasiewicz")
#' res.1 <- RI.hybridFS.FRST(decision.table, control)
#'
#' summary(res.1)
#' ###########################################################
#' ## Example 2: Classification problem
#' ##############################################################
#' data(RoughSetData)
#' decision.table <- RoughSetData$pima7.dt
#' 
#' control <- list(type.aggregation = c("t.tnorm", "lukasiewicz"), type.relation =  
#'                 c("tolerance", "eq.3"), t.implicator = "lukasiewicz")
#' res.2 <- RI.hybridFS.FRST(decision.table, control)
#'
#' summary(res.2)
#' @export  
#' @method summary RuleSetFRST
#' @S3method summary RuleSetFRST
summary.RuleSetFRST <- function(object, ...){
  
 if(!inherits(object, "RuleSetFRST")) stop("not a legitimate object in this package")
 cat("The type of the considered model: ", "\n")
 print(object$type.model)
 cat("The type of the considered method: ", "\n")
 print(object$type.method)
 cat("The type of the considered task: ", "\n")
 print(object$type.task)
 cat("The type of similarity: ", "\n")
 print(object$t.similarity)
 cat("The type of triangular norm: ", "\n")
 print(object$t.tnorm)
 cat("The names of attributes and their type (whether nominal or not): ", "\n")
 temp = matrix(object$nominal.att, nrow = 1)
 colnames(temp) <- matrix(object$antecedent.attr, nrow = 1)
 print(temp)
 cat("The interval of the data: ", "\n")
 temp <- colnames(object$variance.data)
 colnames(object$range.data) <- temp
 print(object$range.data)
 cat("The variance values of the data: ", "\n")
 print(object$variance.data)
 cat("The rules : ", "\n")
 rules <- toStr.rules(object$rules$rules, object$type.task, object$nominal.att)
 print(rules)
   
 invisible(object)	
}

#' This function enables the output of a summary of the indiscernibility relation functions. 
#'
#' @title The summary function of indiscernibility relation based on RST and FRST
#' 
#' @param object a \code{"IndiscernibilityRelation"} object. See \code{\link{BC.IND.relation.FRST}} 
#'
#'        and \code{\link{BC.IND.relation.RST}}.
#' @param ... the other parameters.
#' @examples
#' ###########################################################
#' ## Example 1: Dataset containing nominal values for 
#' ## all attributes.
#' ###########################################################
#' ## Decision table is represented as data frame
#' dt.ex1 <- data.frame(c(1,0,2,1,1,2,2,0), c(0, 1,0, 1,0,2,1,1), 
#'                         c(2,1,0,0,2,0,1,1), c(2,1,1,2,0,1,1,0), c(0,2,1,2,1,1,2,1))
#' colnames(dt.ex1) <- c("aa", "bb", "cc", "dd", "ee")
#' decision.table <- SF.asDecisionTable(dataset = dt.ex1, decision.attr = 5)
#'
#' ## In this case, we only consider the second and third attributes.
#' attributes <- c(2, 3)
#' 
#' #### calculate fuzzy indiscernibility relation ####
#' ## in this case, we are using "crisp" as a type of relation and type of aggregation
#' control.ind <- list(type.relation = c("crisp"), type.aggregation = c("crisp"))
#' IND <- BC.IND.relation.FRST(decision.table, attribute = attributes, control = control.ind)
#'
#' summary(IND)
#' @export  
#' @method summary IndiscernibilityRelation
#' @S3method summary IndiscernibilityRelation
summary.IndiscernibilityRelation <- function(object, ...){
  
 if(!inherits(object, "IndiscernibilityRelation")) stop("not a legitimate object in this package")
 cat("The name of model: ", object$type.model, "\n")
 if (object$type.model == "FRST"){
	cat("The type of aggregation: ", "\n")
	print(object$type.aggregation)
 }
 cat("The name of relation: ", "\n")
 print(object$type.relation)
 cat("The matrix of indiscernibility relation: ", "\n")
 print(object$IND.relation)
 
  invisible(object)	
}


#' This function enables the output of a summary of the lower and upper approximations. 
#'
#' @title The summary function of lower and upper approximations based on RST and FRST
#' 
#' @param object a \code{"LowerUpperApproximation"} object. See \code{\link{BC.LU.approximation.FRST}} and \code{\link{BC.LU.approximation.RST}}.
#' @param ... the other parameters.
#' @examples
#' #######################################
#' ## Example: Using simple data set
#' #######################################
#' dt.ex1 <- data.frame(c(1,0,2,1,1,2,2,0), c(0, 1,0, 1,0,2,1,1), 
#'                         c(2,1,0,0,2,0,1,1), c(2,1,1,2,0,1,1,0), c(0,2,1,2,1,1,2,1))
#' colnames(dt.ex1) <- c("aa", "bb", "cc", "dd", "ee")
#' decision.table <- SF.asDecisionTable(dataset = dt.ex1, decision.attr = 5, 
#'                                      indx.nominal = c(1:5))
#'
#' P <- c(2,3)
#' 
#' ####### Compute indiscernibility relation #######
#' IND <- BC.IND.relation.RST(decision.table, attribute = P)
#'
#' ####### Compute lower and upper approximation #####
#' decision.attr <- c(5)
#' roughset <- BC.LU.approximation.RST(decision.table, IND, decision.attr)
#'
#' summary(roughset)
#' @export  
#' @method summary LowerUpperApproximation
#' @S3method summary LowerUpperApproximation
summary.LowerUpperApproximation <- function(object, ...){
  
 if(!inherits(object, "LowerUpperApproximation")) stop("not a legitimate object in this package")
 cat("The name of model: ", object$type.model, "\n")
 cat("The model of lower/upper approximations: ", object$type.LU, "\n")
 cat("The lower approximation: ", "\n")
 if (object$type.model == c("FRST")){
 print(object$fuzzy.lower)
 }
 else print(object$lower.approximation)
 
 cat("The upper approximation: ", "\n")
 if (object$type.model == c("FRST")){
 print(object$fuzzy.upper)
 }
 else print(object$upper.approximation)
 
  invisible(object)	
}

#' This function enables the output of a summary of the positive region and degree of dependency. 
#'
#' @title The summary function of positive region based on RST and FRST
#' 
#' @param object a \code{"PositiveRegion"} object. See \code{\link{BC.positive.reg.FRST}} and \code{\link{BC.positive.reg.RST}}.
#' @param ... the other parameters.
#' @examples
#' dt.ex1 <- data.frame(c(1,0,2,1,1,2,2,0), c(0, 1,0, 1,0,2,1,1), 
#'                         c(2,1,0,0,2,0,1,1), c(2,1,1,2,0,1,1,0), c(0,2,1,2,1,1,2,1))
#' colnames(dt.ex1) <- c("aa", "bb", "cc", "dd", "ee")
#' decision.table <- SF.asDecisionTable(dataset = dt.ex1, decision.attr = 5, 
#'                                     indx.nominal = c(1:5))
#'
#' ## in this case, we consider second and third attributes only
#' P <- c(2,3)
#' 
#' ####### Perform indiscernibility relation #######
#' IND <- BC.IND.relation.RST(decision.table, attribute = P)
#'
#' ####### Perform lower and upper approximations #####
#' decision.attr <- c(5)
#' roughset <- BC.LU.approximation.RST(decision.table, IND, decision.attr)
#' 
#' ####### Determine the positive region ######
#' region <- BC.positive.reg.RST(decision.table, roughset)
#'
#' summary(region)
#' @export  
#' @method summary PositiveRegion
#' @S3method summary PositiveRegion
summary.PositiveRegion <- function(object, ...){
  
 if(!inherits(object, "PositiveRegion")) stop("not a legitimate object in this package")
 cat("The name of model: ", object$type.model, "\n")
 cat("The positive region: ", "\n")
 if (object$type.model == "FRST"){
	print(object$positive.freg)
 }
 else print(object$positive.reg)
 cat("The degree of dependency: ", "\n")
 print(object$degree.dependency)
 
  invisible(object)
}

# It is used to build class of rough set and fuzzy rough set theories. Currently, its implementation is very basic and 
# does no argument checking, as it is only used internally.
#
# @title The object factory for RoughSets objects
# @param mod a list containing all the attributes for the object
# @param classname a class name
# @return an object of type \code{RoughSet}
# @aliases RoughSets-object
ObjectFactory <- function(mod, classname){	
	class(mod) <- unique(c(classname, class(mod)))
	return(mod)
}

#' It is used to apply a particular object/model for obtaining a new decision table. The models have been calculated previously by feature selection, instance selection, and
#' discretization.
#'
#' @title Apply for obtaining a new decision table
#' @param decision.table a \code{"DecisionTable"} class representing a decision table. See \code{\link{SF.asDecisionTable}}.
#' @param object a class resulting from feature selection (e.g. \code{\link{FS.reduct.computation}}), discretization (e.g. \code{\link{D.discretization.RST}}), 
#'               and instance selection functions 
#'
#'              (e.g. \code{\link{IS.FRIS.FRST}}). 
#' @param control a list of other parameters which are \code{indx.reduct} representing an index of the chosen decision reduct. It is only considered when 
#'               we calculate all reducts using \code{\link{FS.all.reducts.computation}}. The default value is that the first reduct will be chosen.
#' @return A new decision table. Especially for the new decision table resulting from discretization, we 
#'         obtain a different representation. Values are expressed in intervals instead of labels. For example,
#'         \eqn{a_1 = [-Inf, 1.35]} refers to the value \eqn{a_1} has a value in that range.
#' @examples
#' #############################################################
#' ## Example 1: The feature selection in RST 
#' ## using quickreduct
#' #############################################################
#' data(RoughSetData)
#' decision.table <- RoughSetData$hiring.dt 
#'
#' object.1 <- FS.quickreduct.RST(decision.table)
#'
#' new.decTable <- SF.applyDecTable(decision.table, object.1)
#'
#' #############################################################
#' ## Example 2: The feature selection in FRST
#' ## using fuzzy.QR (fuzzy quickreduct)
#' #############################################################
#' data(RoughSetData)
#' decision.table <- RoughSetData$hiring.dt 
#'
#' ## fuzzy quickreduct using fuzzy lower approximation 
#' control <- list(decision.attr = c(5), t.implicator = "lukasiewicz", 
#'                 type.relation = c("tolerance", "eq.1"), type.aggregation = 
#'                 c("t.tnorm", "lukasiewicz"))
#' object.2 <- FS.quickreduct.FRST(decision.table, type.method = "fuzzy.dependency", 
#'                             type.QR = "fuzzy.QR", control = control)
#' 
#' ## generate new decision table
#' new.decTable <- SF.applyDecTable(decision.table, object.2)
#'
#' ###################################################
#' ## Example 3: The Instance selection by IS.FRPS and
#' ## generate new decision table
#' ###################################################
#' dt.ex1 <- data.frame(c(0.5, 0.2, 0.3, 0.7, 0.2, 0.2), 
#'                   c(0.1, 0.4, 0.2, 0.8, 0.4, 0.4), c(0, 0, 0, 1, 1, 1))
#' colnames(dt.ex1) <- c("a1", "a2", "d")
#' decision.table <- SF.asDecisionTable(dataset = dt.ex1, decision.attr = 3)
#'
#' ## evaluate instances
#' res.1 <- IS.FRPS.FRST(decision.table, type.alpha = "FRPS.3")
#'
#' ## generate new decision table
#' new.decTable <- SF.applyDecTable(decision.table, res.1)
#'
#' #################################################################
#' ## Example 4: Discretization by determining cut values and 
#' ## then generate new decision table
#' #################################################################
#' dt.ex2 <- data.frame(c(1, 1.2, 1.3, 1.4, 1.4, 1.6, 1.3), c(2, 0.5, 3, 1, 2, 3, 1),
#'                              c(1, 0, 0, 1, 0, 1, 1))
#' colnames(dt.ex2) <- c("a", "b", "d")
#' decision.table <- SF.asDecisionTable(dataset = dt.ex2, decision.attr = 3,
#'                   indx.nominal = 3)
#'
#' ## get cut values using the local strategy algorithm
#' cut.values <- D.discretization.RST(decision.table, type.method = "local.disc.matrix")
#'
#' ## generate new decision table
#' new.decTable <- SF.applyDecTable(decision.table, cut.values)
#' @export
SF.applyDecTable <- function(decision.table, object, control = list()){
  
	if(!inherits(decision.table, "DecisionTable")) {
		stop("Provided data should inherit from the \'DecisionTable\' class.")
	}
  
	objects <- decision.table
	nominal.att <- attr(objects, "nominal.attrs")
	desc.attrs <- attr(objects, "desc.attrs")	
	control <- setDefaultParametersIfMissing(control, list(indx.reduct = 1))
	names.attrs <- colnames(decision.table)
	
	if (inherits(object, "FeatureSubset")) {
		tmpIdx = c(object$reduct, attr(objects, "decision.attr"))
		decision.table <- objects[, tmpIdx, drop = FALSE]
		attr(decision.table, "nominal.attrs") = nominal.att[tmpIdx]
		attr(decision.table, "desc.attrs") = desc.attrs[tmpIdx]
		if	(!is.null(attr(objects, "decision.attr"))) {
			attr(decision.table, "decision.attr") = ncol(decision.table)
		}
		else {
			attr(decision.table, "decision.attr") = NULL
		}
		decision.table = ObjectFactory(decision.table, "DecisionTable")
    
	}	
	else if (inherits(object, "ReductSet")) {
		indx.reduct <- control$indx.reduct
		reducts <- object$decision.reduct
		if (is.null(reducts[indx.reduct]) || is.na(reducts[indx.reduct])) {
			stop("there is no reducts at the given indx.reduct")
		}
    
		tmpIdx = c(reducts[[indx.reduct]], attr(objects, "decision.attr"))		
		decision.table <- objects[, tmpIdx, drop = FALSE]
		attr(decision.table, "nominal.attrs") = nominal.att[tmpIdx]
		attr(decision.table, "desc.attrs") = desc.attrs[tmpIdx]
		if (!is.null(attr(objects, "decision.attr"))) {
			attr(decision.table, "decision.attr") = ncol(decision.table)
		}
		else {
			attr(decision.table, "decision.attr") = NULL
		}
		decision.table = ObjectFactory(decision.table, "DecisionTable")

	} 
	else if (inherits(object, "InstanceSelection")){
		indx.objects <- object$indx.objects
		if (length(indx.objects) > 0) {
			decision.table <- objects[c(indx.objects), , drop = FALSE]
			attr(decision.table, "nominal.attrs") = nominal.att
			attr(decision.table, "desc.attrs") = desc.attrs
			attr(decision.table, "decision.attr") = attr(objects, "decision.attr")
			decision.table = ObjectFactory(decision.table, "DecisionTable")
		}
	}	
	else if (inherits(object, "Discretization")){
		## sort the cut values
		cut.values <- lapply(object$cut.values, sort)

		## get discrete values according to the cut values
		if (!is.null(attr(objects, "decision.attr"))) {
			decision.attr = as.character(objects[[attr(objects, "decision.attr")]])
			decision.table = mapply(applyDiscretization, 
  		                        objects[-attr(objects, "decision.attr")], cut.values, 
                              SIMPLIFY = FALSE)
			decision.table[[length(decision.table) + 1]] = decision.attr
		} 
		else {
			decision.table = mapply(applyDiscretization, objects, cut.values, SIMPLIFY = FALSE)
		}
		decision.table = data.frame(decision.table, stringsAsFactors = TRUE)
		colnames(decision.table) = colnames(objects)
		
		## generate decision table with changing in nominal.attrs attribute  
		decision.table <- SF.asDecisionTable(dataset = decision.table, decision.attr = attr(objects, "decision.attr"), 
	                                     indx.nominal = 1:ncol(decision.table))													   
	}
  
    return(decision.table)
}

## checking missing parameters
# @param control parameter values of each method
# @param defaults default parameter values of each method
setDefaultParametersIfMissing <- function(control, defaults) {
  for(i in names(defaults)) {
    if(is.null(control[[i]])) control[[i]] <- defaults[[i]]
  }
  control
}

# It is used to convert rules into string
# @param rules rules in numeric
# @param type.task a type of task
# @param nominal.att a list of types of attributes
toStr.rules <- function(rules, type.task = "classification", nominal.att){
	options(stringsAsFactors = FALSE)
	Str.rules <- list()
	
	for (h in 1 : length(rules)){
		rule <- rules[[h]]

		if (ncol(rule) > 2){
			ante <- paste(colnames(rule[1]), rule[1], sep = ifelse(nominal.att[1] == TRUE, c(" is "), c(" is around ")))			
			for (i in 2 : (ncol(rule) - 1)){
				temp <- paste(colnames(rule[i]), rule[i], sep = ifelse(nominal.att[i] == TRUE, c(" is "), c(" is around ")))
				ante <- paste(ante, temp, sep = " and ")
			}
		}
		else {
			ante <- paste(colnames(rule[1]), rule[1], sep = ifelse(nominal.att[1] == TRUE, c(" is "), c(" is around ")))
		}
		
		if (type.task == "classification"){
			cons <- paste(colnames(rule[ncol(rule)]), rule[[ncol(rule)]], sep = c(" is "))		
		}
		else {
			cons <- paste(colnames(rule[ncol(rule)]), rule[ncol(rule)], sep = c(" is around "))	
		}

		rule <- paste("IF", ante, "THEN", cons)	
		Str.rules <- append(Str.rules, rule)
	}
	
	return(Str.rules)
}
