#' Import 'Raven' batch correlator output 
#' 
#' \code{imp_corr_mat} imports the output of 'Raven' batch correlator.
#' @usage imp_corr_mat(file, path = NULL)
#' @param file A character string with the name of the output '.txt' file
#' generated by Raven.
#' @param path A character string indicating the path of the directory in which to look for the text files. 
#' If not provided (default) the function searches into the current working directory.
#' @return A list with 2 matrices. The first one contains the correlation coefficients and
#' the second one the time lags of the peak correlations.
#' @details The function imports the output of a batch correlation routine in Raven. 
#' Both the correlation and lag matrices contained in the output '
#' .txt' file are read and both waveform and spectrogram (cross-correlation) correlations can be 
#' imported.
#' @seealso \code{\link{imp_raven}}; \code{\link{exp_raven}} 
#' @export
#' @name imp_corr_mat
#' @examples
#' \dontrun{ 
#' # Load data
#' library(warbleR)
#' data(list = c("Phae.long1", "Phae.long2", "Phae.long3", "Phae.long4", "lbh_selec_table"))
#' 
#' writeWave(Phae.long1, file.path(tempdir(), "Phae.long1.wav"), extensible = FALSE) #save sound files 
#' writeWave(Phae.long2, file.path(tempdir(), "Phae.long2.wav"), extensible = FALSE)
#' writeWave(Phae.long3, file.path(tempdir(), "Phae.long3.wav"), extensible = FALSE)
#' writeWave(Phae.long4, file.path(tempdir(), "Phae.long4.wav"), extensible = FALSE)
#' 
#' #create new folder to put cuts
#' dir.create(file.path(tempdir(), "cuts"))
#' 
#' # cut files
#' cut_sels(X = lbh_selec_table, mar = 0.05, path = tempdir(),
#'  dest.path = file.path(tempdir(), "cuts"))
#' 
#' #Now run 'Raven' batch correlator un the cuts and save the output in the same folder
#' 
#' # Import output (change the name of the file if you used a different one)
#' xcorr.rav <- imp_corr_mat(file = "BatchCorrOutput.txt", 
#' path = file.path(tempdir(), "cuts"))
#'
#' # check results
#'   
#' ## correlation matrix
#' xcorr.rav[[1]]
#' 
#' ## time lag matrix
#' xcorr.rav[[2]]
#' }
#' 
#' @author Marcelo Araya-Salas (\email{marceloa27@@gmail.com})
#last modification on sep-2-2019
imp_corr_mat <- function(file, path = NULL){
  
  #check path to working directory
  if (is.null(path)) path <- getwd() else 
    if (!dir.exists(path)) 
      stop("'path' provided does not exist") 
  
  lns <- readLines(file.path(path, file))
  lbks <- which(lns == "")
  
  mat1 <- do.call(rbind, strsplit(lns[lbks[1]:lbks[2]], split = "\t"))
  
  nms <- mat1[-1, 1]
  
  mat1 <- apply(as.matrix(mat1[-1, -1]), 2, as.numeric)
  
  rownames(mat1) <- colnames(mat1) <- nms
  

  mat2 <- do.call(rbind, strsplit(lns[(lbks[3]):(lbks[3] + nrow(mat1) + 1)], split = "\t"))
  
  mat2 <- apply(as.matrix(mat2[-1, -1]), 2, as.numeric)
  
  rownames(mat2) <- colnames(mat2) <- nms
  
  out <- list(mat1, mat2)
  
  names(out) <- c("correlation", "lag (s)")
  
  return(out)
  }

