\name{use.aux.urng-method}
\docType{methods}

\alias{use.aux.urng}
\alias{use.aux.urng-method}
\alias{use.aux.urng,unuran-method}
\alias{use.aux.urng<-}
\alias{use.aux.urng<--method}
\alias{use.aux.urng<-,unuran-method}

\alias{set.aux.seed}

\title{Use auxiliary random number generator for Runuran objects}

\description{
  Some UNU.RAN methods that are based on the rejection method can make
  use of a second auxiliary uniform random number generator.
  It is only used when a rejection has occurred, see below for details.
  This allows to keep two streams of random variates (almost)
  synchronized. This is in particular necessary for variance reduction
  methods like common or antithetic random variates.

  [Advanced] -- Use auxiliary URNG for rejection method.
}

\usage{
\S4method{use.aux.urng}{unuran}(unr)
use.aux.urng(unr) <- value
set.aux.seed(seed)
}

\arguments{
  \item{unr}{a \code{unuran} generator object.}
  \item{value}{\code{TRUE} when an auxiliary URNG is used,
    \code{FALSE} when no auxiliary URNG is used (the default).}
  \item{seed}{seed for the auxiliary URNG.}
}

\details{
  Variance reduction techniques like common or antithetic random
  variates require correlated streams of non-uniform random variates.
  Such streams can be easily created by means of the inversion method
  using the same source of uniform random numbers (URNs).
  However, the quantile function (inverse CDF) or even the CDF often is
  not available in closed form and thus numerical method are required
  that are expensive or only return approximate random numbers or both.

  On the other hand two streams of non-uniformly distributed random
  variates are completely uncorrelated when the
  acceptance-rejection method is used. Then the two streams run
  \dQuote{out-of-sync} when the first rejection occurs. 

  Schmeiser and Kachitvichyanukul, however, have shown that this problem
  can be overcome when the proposal point is generated by inversion and
  a fixed number \eqn{k} of URNs is used for generating one non-uniform
  random variate. This can be accomplished by means of a second
  auxiliary stream of URNs which is used when the required number of
  URNs exceeds \eqn{k}.

  By this approach two streams of non-uniform random variates run
  synchronized except when a rejection occurs in one of the two streams.
  Therefore the two generated streams are respected mixtures of highly
  correlated streams and independent streams.
  The induced correlation thus decreases when the rejection constants of
  the acceptance-rejection algorithms used for generating the two
  streams increases.

  In UNU.RAN some of the acceptance-rejection algorithms make use of a
  second auxiliary stream of URNs. It is implemented in one of the
  following ways:
  \itemize{
    \item
    The primary uniform random number generator is used for the
    first loop of the acceptance-rejection step. When a rejection occurs
    the algorithms switches to auxiliary generator until the proposal
    point is accepted.
    Thus exactly \emph{two} URNs from the primary generator are used to
    generate one non-uniform random variate.
    \item
    The primary uniform random number generator is used just for the
    first proposal point and then switches to the auxiliary generator
    until the proposal point is accepted.
    Thus exactly \emph{one} URN from the primary generator is used to
    generate one non-uniform random variate.
  }

  The call \code{use.aux.urng(unr)} returns \code{FALSE} if this feature
  is disabled for Runuran generator object \code{unr} (the default)
  and \code{TRUE} if this feature is enabled.
  It auxiliary URNs are not supported at all for object \code{unr} then
  \code{use.aux.urng(unr)} returns \code{NA}.
  
  The replacement method
  \code{use.aux.urng(unr) <- TRUE}
  enables this feature for generator \code{unr}.
  It can be disabled by means of
  \code{use.aux.urng(unr) <- FALSE}.
  (One gets an error if this feature is not supported at all.)

  The seed of the auxiliary uniform random number generator can be set by
  means of \code{set.aux.seed(seed)}. The auxiliary generator is
  a combined multiple recursive generator (MRG31k3p) by
  L'Ecuyer and Touzin and is built into package \pkg{Runuran}.
  Currently it cannot be replaced by some other generator. 
}

\section{Methods}{
  Currently the following UNU.RAN methods support this feature.
  (Currently the last four methods are only available via
  \code{\link{unuran.new}}, see the UNU.RAN manual for more details.)
  \tabular{lll}{
    method name        \tab Runuran call          \tab URN per variate \cr
    \dQuote{tdr} (ps)  \tab \code{\link{tdr.new}} \tab 2 \cr
    \dQuote{arou} (ia) \tab --                    \tab 1 \cr
    \dQuote{tabl} (ia=false) \tab --              \tab 2 \cr
    \dQuote{tdr} (gw)  \tab --                    \tab 2 \cr
    \dQuote{tdr} (ia)  \tab --                    \tab 1 \cr
    \cr
    inversion          \tab                       \tab 1 \cr
  }
}

\value{
  \code{use.aux.urng} returns \cr
  \code{TRUE}, if using the auxiliary generator is enabled, \cr
  \code{FALSE}, it is disabled, and \cr
  \code{NA}, if this feature does not exist at all.
  
  \code{set.aux.seed} returns \code{NULL}, invisibly.
}

\note{Using an auxiliary uniform random number generator generator is
  only useful if the rejection constant is close to 1.}
  
\seealso{
  \code{\link{tdr.new}}.
}

\references{
  W. H\"ormann, J. Leydold, and G. Derflinger (2004):
  Automatic Nonuniform Random Variate Generation, Sect. 8.4.2.
  Springer-Verlag, Berlin Heidelberg

  B. W. Schmeiser and V. Kachitvichyanukul (1986):
  Correlation induction without the inverse transformation.
  In: Proc. 1986 Winter Simulation Conf.,
  J. Wilson, J. Henriksen, S. Roberts (eds.), 266-274.

  B. W. Schmeiser and V. Kachitvichyanukul (1990):
  Non-inverse correlation induction: guidelines for algorithm
  development.
  J. Comput. Appl. Math. 31, 173-180.

  W. H\"ormann and G. Derflinger (1994):
  Universal generators for correlation induction.
  In: Compstat, Proceedings in Computational Statistics,
  R. Dutter and W. Grossmann (eds.), 52-57.

  J. Leydold, E. Janka, and W. H\"ormann (2002):
  Variants of Transformed Density Rejection and Correlation Induction.
  In: Monte Carlo and Quasi-Monte Carlo Methods 2000,
  K.-T. Fang, F. Hickernell, and H. Niederreiter (eds.),
  345--356.

  P. L'Ecuyer and R. Touzin (2000):
  Fast combined multiple recursive generators with multipliers of the
  form a = +/- 2^q +/- 2^r.
  In: J.A. Jones, R.R. Barton, K. Kang, and P.A. Fishwick (eds.),
  Proc. 2000 Winter Simulation Conference, 683-689.
}

\examples{
## Create respective generators for normal and exponential distribution.
## Use method TDR
gen1 <- tdrd.new(udnorm())
gen2 <- tdrd.new(udexp())

## The two streams are independent even we use the same seed
set.seed(123); x1 <- ur(gen1,1e5)
set.seed(123); x2 <- ur(gen2,1e5)
cor(x1,x2)

## We can enable the auxiliary URNG and get correlated streams
use.aux.urng(gen1) <- TRUE
use.aux.urng(gen2) <- TRUE
set.seed(123); x1 <- ur(gen1,1e5)
set.seed(123); x2 <- ur(gen2,1e5)
cor(x1,x2)

## This feature can be disabled again
use.aux.urng(gen1)
use.aux.urng(gen1) <- FALSE
use.aux.urng(gen2) <- FALSE

## Notice that TDR cannot simply mixed with an inversion method
## as the number of URNG per random point differs
gen3 <- pinvd.new(udexp())
set.seed(123); x3 <- ur(gen3,1e5)
cor(x1,x3)

## But a trick would do this
set.seed(123); x3 <- ur(gen3,2*1e5)
x3 <- x3[seq(1,2*1e5,2)]
cor(x1,x3)
## or ...
set.seed(123); u3 <- runif(2*1e5); u3 <- u3[seq(1,2*1e5,2)]
x3 <- uq(gen3,u3)
cor(x1,x3)

## Maybe method AROU is more appropriate
gen4 <- unuran.new(udnorm(), "arou")
use.aux.urng(gen4) <- TRUE
set.seed(123); x3 <- ur(gen3,1e5)
set.seed(123); x4 <- ur(gen4,1e5)
cor(x3,x4)

}

\keyword{methods}
\keyword{datagen}
