\name{plot}
\docType{methods}
\alias{plot}
\alias{plot.strata}
\title{Plot method for stratigraphic and sedimentological data\cr
       (strata class object)}
\description{
Plot method for stratigraphic and sedimentological data. This function will produce a graphic log 
of outcrop section or borehole log in a PDF format (check the working directory for the PDF output file).
}
\usage{
\method{plot}{strata}(x,
  datum = "top",
  data.units = "feet",
  d.scale = 100,
  d.barscale = 2,
  lithology = TRUE,
  bed.number = TRUE,
  file.name = "SDAR_graphical_output",
  GR.log = NULL,
  ncore = NULL,
  samples = NULL,
  oil.stain = NULL,
  sed.structures = NULL,
  fossils = NULL,
  tracefossils = NULL, 
  other.sym = NULL,
  bioturbation = NULL,
  lithostrat = NULL,
  d.legend = TRUE,
  metadata = NULL,
  main = NULL,
  sub = NULL,
  subset.base = NULL,
  subset.top = NULL,
  plot.order = NULL,  
  d.color = 0,
  bar.type = 0,
  symbols.size = 1,
  xlim.GR=c(0, 300),
  family = "serif",   
  fontsize = 10,
  cex.main = 1,
  \dots)
}
\arguments{
  \item{x}{an object of \link{strata} class to be plotted, which holds the minimum required data to draw a lithological
    profile in SDAR. See \link{strata} class for more details about the requirements of this 'x' object.}
    \item{datum}{string; either 'base' or 'top', case sensitive. 'base' is the case when thickness is measured up from the bottom of, e.g., an outcrop section; 'top' is the case when depths are measured down from the surface, e.g., boreholes and cores.}
    \item{data.units}{string; either 'meters' or 'feet', case sensitive. This argument specifies the unit of measure of the stratigraphic thickness used for input data (thickness measured in field). Default unit 'feet'.}
    \item{d.scale}{default 100, a numerical value giving the graphic vertical scaling. It defines the vertical scale on which the graphic log is drawn; recommended scales are 1:25, 1:50, 1:100, 1:250, 1:500, and 1:1000.}
    \item{d.barscale}{default 2, a numerical value indicating how often the horizontal tick marks will be displayed on the vertical scale (thickness). The bar scale is displayed using the same units as 'data.units' parameter, and it will be plotted at the left side of the graphic log.}
    \item{lithology}{default \code{\link[base:logical]{TRUE}}; a logical value indicating whether lithology should be displayed.}
    \item{bed.number}{default \code{\link[base:logical]{TRUE}}; a logical value indicating whether bed number should be displayed.}
    \item{file.name}{string; a file name without extension. Graphic logs generated by SDAR are exported as PDF files (editable with vector graphics software). The paper size is automatically updated when the vertical scale changes, or when different sets of attributes are displayed adjacent to the lithology column. Default name "SDAR_graphical_output".}
    \item{GR.log}{a two-column data frame containing the depth, and the total natural radioactivity measured in API units; the columns must be named ('Depth' and 'GR'). See \link{read.LAS} function for details of how to parse geophysical "Log Ascii Standard" files (.las files) into R.}
    \item{ncore}{a data frame containing the depth interval and the core number. It is useful information about how the core is stored. This information should be stored into three columns named ('base', 'top', and 'core_number'), with the information of each core in a separate row containing the depth interval (base and top) and the core number.}
    \item{samples}{a data frame containing the sample dataset. This information should be stored in three columns named ('base', 'top', and 'type'); the columns 'base' and 'top' should be numeric. Each sample should be in a separate row containing the stratigraphic position (base and top) and the analysis type (e.g., palynology, petrography, rock sample). It is possible to add drawing parameters such as 'pch', and 'color', each one in a separate column in the input data.}
    \item{oil.stain}{a data frame containing the oil.stain dataset. This information should be stored in three columns named ('base', 'top', and 'intensity'); the columns 'base' and 'top' should be numeric. Each staining interval should be in a separate row containing the stratigraphic position (base and top) and the staining intensity (weak, moderate, moderate strong, strong).}
    \item{sed.structures}{a data frame containing the sedimentary structure dataset. This information should be stored in three columns named ('base', 'top', and 'sed_structure'); the columns 'base' and 'top' should be numeric. Each sedimentary structure occurrence should be in a separate row containing the stratigraphic position and the sedimentary structure type; overlapping between sedimentary structures intervals is allowed.}
    \item{fossils}{a data frame containing the fossil dataset. This information should be stored in three columns named ('base', 'top', and 'fossil'); the columns 'base' and 'top' should be numeric. Each fossil occurrence should be in a separate row containing the stratigraphic position and the fossil type. If the fossil occurrence covers a stratigraphic interval, define it using 'base' and 'top' (overlapping between fossil intervals is allowed); in a case where fossil occurrence corresponds to a specific stratigraphic position, fill variables 'base' and 'top' with the same stratigraphic position value.}
    \item{tracefossils}{same dataset structure as fossil dataset. Three columns named ('base', 'top', and 'tracefossil'), "this option is not available at current version".}
    \item{other.sym}{same dataset structure as fossil dataset. Three columns named ('base', 'top', and 'other_symbols'). This is the place to symbolize features as accessory minerals, organic matter, intraclast, etc.}
    \item{bioturbation}{a data frame containing the bioturbation dataset. This information should be stored in three columns named ('base', 'top', and 'index'); the columns 'base' and 'top' should be numeric. Each bioturbated interval should be in a separate row containing the stratigraphic interval (base and top) and the bioturbation index value. The index classifies on a scale of zero to six (Reineck, 1967; modified by Taylor and Goldring, 1993).}
    \item{lithostrat}{a data frame containing the lithostratigraphic data. This information should be stored in four columns named ('base', 'top', 'litho_unit_type' and 'name'); the columns 'base' and 'top' should be numeric. Each lithostratigraphic interval should be in a separate row containing the stratigraphic interval (base and top), the lithostratigraphic unit rank (e.g., Group, Formation, Member, Informal Member), and the name of the lithostratigraphic unit.}
    \item{d.legend}{default \code{\link[base:logical]{TRUE}}; a logical value indicating whether automatic legend should be drawn. The legend function creates a legend based on the lithological, sedimentological, and paleontological data provided for the current stratigraphic section or well. SDAR uses the standard symbols suggested by the \href{http://www.fgdc.gov}{Federal Geographic Data Committee}. The legend will be displayed at the bottom of the PDF file.}
    \item{metadata}{an object of class \link{list}, including any or all of the following named values: locality_name, locality_id, locality_type, thickness_unit, reference_datum, latitude, longitude, elevation, country, author and/or references. The objects in the list should be named using the previous names 'case sensitive', e.g., list(locality_name = "Saltarin", locality_type = "borehole core", thickness_unit = "meters", country = "Colombia", author ="Ortiz J. & Jaramillo C.").}
    \item{main}{an overall title for the graphic log, the main title (on top).}
    \item{sub}{an overall sub-title for the graphic log, the sub-title (on top).}
    \item{subset.base}{option to draw and analyse a specific interval for a given outcrop section or borehole log. This argument defines the lower limit of the stratigraphic interval of interest.}
    \item{subset.top}{This argument defines the upper limit of the stratigraphic interval of interest.}
    \item{plot.order}{a string vector. This parameter provides a user interaction to arrange (order) the layout scheme. If the user wants to change the default order, this string vector provides the desired order, e.g., c("d.samples", "d.ncore", "d.bed.number", "d.graphic_bar", "d.lithostrat", "d.lithology", "d.fossils", "d.tracefossils",\cr "d.sed.structures", "d.other.sym", "d.oil.stain", d.bioturbation", "d.GR.log").}
    \item{d.color}{numeric 0 or 3; '0 to draw SDAR default colors', or '3 for black & white option', (munsell and RGB options will be available at next version)}
    \item{bar.type}{numeric 0 or 1; '0 to draw classical black & white bar', or '1 to draw a simple "line" with tick marks'} 
    \item{symbols.size}{numeric; a number indicating the amount by which plotting symbols (fossils, tracefossils, sedimentary structures, and other symbols) should be re-scaled relative to the default. 0.5 = default, 1 is 50\% larger, 0.4 is 10\% smaller, etc.}
    \item{xlim.GR}{a range to specify axis limits \emph{c(xmin, xmax)}, its define the minimum and maximun value of the x-axis to display Gamma Ray data.}
    \item{family}{the font family to be used. The default font family is serif, in windows, serif is mapped to "TT Times New Roman".}		
    \item{fontsize}{numeric; number indicating the amount by which plotting text should be scaled relative to default. 1=default, 1.5 is 50\% larger, 0.5 is 50\% smaller, etc.}
    \item{cex.main}{number indicating the amount by which main text should be scaled relative to the default. 1=default, 1.5 is 50\% larger, 0.5 is 50\% smaller, etc.}
    \item{\dots}{ other arguments }
}
\note{
We have presented a summary of the specific types of data required by the SDAR package, 
along with the formats that should be followed when inputting data to be integrated using SDAR.
The SDAR project includes the development of a graphic user interface to connect this R package
with a database management system; for this reason the structure of the data and headers
(column names) should be followed in order to match the database structure.\cr
On the \href{https://repository.si.edu/10088/35917}{SDAR repository} there are two excel spreadsheet with the suggested format by SDAR, one to store thickness, and texture description of rock layers (beds), and other to store "intervals information" (e.g., metadata, samples, oil stain, bioturbation, sedimentary structures, fossil and trace fossil content).\cr
If you see problems with the PDF output, remember that the problem is much more likely
to be in your viewer than in R. Try another viewer if possible.
}
\author{
John Ortiz, \email{jrortizt@unal.edu.co}
}
\references{
Reineck, H.-E., 1967. \emph{Parameter von Schichtung und bioturbation. Geologischen Rundschau} \bold{56}, 420-438.

Taylor, A.M., Goldring, R., 1993. \emph{Description and analysis of bioturbation and ichnofabric. Journal of the Geological Society of London} \bold{150}, 141-148.
}

\examples{
# --------  OUTPUT  --------
#  The output of this function is a graphic log of outcrop section or borehole log 
#     in a PDF format (check the working directory for the PDF output file, deafult
#     name "SDAR_graphic_log.pdf").
# --------------------------

# ----------
# example 1: Graphic log of Saltarin 1A core using SDAR default options.  
# ----------

data(saltarin)
saltarin_val <- strata(saltarin, datum="top")
plot(saltarin_val, subset.base=200, subset.top=0,
  main="Graphic log of Saltarin-1A well", sub="Scale 1:100",  
  file.name=tempfile("saltarin_well_SDAR_demo"))

# ----------
# example 2:   
# ----------

data(log_demo)
plot(saltarin_val, d.scale=200, d.barscale=5,
  subset.base=400, subset.top=0,
  main="Graphic log of Saltarin-1A well", sub="Scale 1:200",
  GR.log=log_demo, file.name=tempfile("log_demo_SDAR"))

# ----------
# example 3:  Read beds information from "SDAR excel spreadsheet format"  
# ----------

library(readxl)
fpath <- system.file("demo_data_entry", 
  "SDAR_beds_data_entry_saltarin.xlsx", package = "SDAR")
beds_saltarin <- read_excel(fpath, trim_ws=FALSE)
beds_val <- strata(beds_saltarin, datum="top")
plot(beds_val, d.scale=50, 
  subset.base=120, subset.top=10,
  main="Graphic log of Saltarin-1A well", sub="Scale 1:200",
  file.name=tempfile("saltarin_well"))
}
