% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/SSI.R
\name{plotNet}
\alias{plotNet}
\title{Network plot}
\usage{
plotNet(fm, B, Z = NULL, K, indexK = NULL, subsetG = NULL,
           tst = NULL, U = NULL, d = NULL, group = NULL, group.shape = NULL,
           set.color = NULL, set.size = NULL, df = NULL, title, axis.labels = TRUE,
           curve = FALSE, bg.color = "gray20", unified = TRUE, ntst = 36,
           line.color = "gray90", line.tick = 0.3, legend.pos="right",
           point.color = "gray20", sets = c("Testing","Supporting","Non-active"))
}
\arguments{
\item{fm}{An object of the 'SSI' class}

\item{B}{(numeric matrix) (Optional) Regression coefficients for individuals corresponding to \code{fm$tst} (in rows) and to \code{fm$trn} (in columns)}

\item{Z}{(numeric matrix) Design matrix for random effects. When \code{Z=NULL} an identity matrix is considered (default) thus \ifelse{html}{\out{<b>G</b> = <b>K</b>}}{\eqn{\textbf{G = K}}{G = K}}; otherwise \ifelse{html}{\out{<b>G</b> = <b>Z K Z'</b>}}{\eqn{\textbf{G} = \textbf{Z K Z'}}{G = Z K Z'}} is used}

\item{K}{(numeric matrix) Kinship relationships. This can be a (character) name of a binary file where the matrix is stored}

\item{indexK}{(integer vector)  Which columns and rows will be read when \code{K} is the name of a binary file. Default \code{indexK=NULL} will read the whole matrix}

\item{subsetG}{(integer vector) Which columns (and rows) from \code{G} the vectors \code{fm$trn} and \code{fm$tst} refer to. Default \code{subsetG=NULL} considers that elements \code{fm$trn} and \code{fm$tst} refer to columns (and rows) from \code{G}; otherwise elements in training and testing in \code{G} have indices \code{subsetG[fm$trn]}  and \code{subsetG[fm$tst]}}

\item{tst}{(integer vector) Which individuals are in testing set and to plot. It must be contained in \code{fm$tst}. Default \code{tst=NULL} will consider the whole vector \code{fm$tst} to plot}

\item{U}{(numeric matrix) Eigenvectors from spectral value decomposition of \ifelse{html}{\out{<b>G</b> = <b>U D U'</b>}}{\eqn{\textbf{G} = \textbf{U D U'}}{G = U D U'}}}

\item{d}{(numeric vector) Eigenvalues from spectral value decomposition of \ifelse{html}{\out{<b>G</b> = <b>U D U'</b>}}{\eqn{\textbf{G} = \textbf{U D U'}}{G = U D U'}}}

\item{group}{(data.frame) Column grouping for the individuals. The rows must match with the rows in \code{G} matrix}

\item{df}{(numeric) Average number of training individuals contributing to the prediction (active) of testing individuals. Default \code{df=NULL} will use the \code{df} that yielded the optimal accuracy}

\item{title}{(character/expression) Title of the plot}

\item{bg.color}{(character) Plot background color}

\item{line.color}{(character) Color of lines connecting 'active' training individuals with each individual in testing set}

\item{line.tick}{(numeric) Tick of lines connecting 'active' training individuals with each individual in testing set}

\item{set.color}{(character vector) Color point of each level of 'testing', 'active', and 'non-active' elements, respectively}

\item{set.size}{(numeric vector) Size of 'testing', 'active', and 'non-active' elements, respectively}

\item{group.shape}{(integer vector) Shape of each level of the grouping column provided as \code{group}}

\item{curve}{\code{TRUE} or \code{FALSE} to whether draw curve of rect lines connecting 'active' training individuals with each individual in testing set}

\item{axis.labels}{\code{TRUE} or \code{FALSE} to whether show labels in both axes}

\item{unified}{\code{TRUE} or \code{FALSE} to whether show an unified plot or separated for each individual in 'testing'}

\item{point.color}{(character) Color of the points in the plot}

\item{ntst}{(integer) Maximum number of individuals in 'testing' that are plotted separated as indicated by \code{unified=FALSE}}

\item{legend.pos}{(character) Either "right", topright","bottomleft","bottomright","topleft", or "none" indicating where the legend is positioned in the plot}

\item{sets}{(character vector) Names of the sets: testing group, predictors with non-zero coefficient, and predictors with zero coefficient in the SSI, respectively}
}
\value{
Returns the top-2 PC's plot connecting testing (predicted) individuals with training (predictors) individuals
}
\description{
Network plot of testing and training individuals from an object of the class 'SSI'
}
\examples{
  require(SFSI)
  data(wheatHTP)
  
  X = scale(X[1:200,])/sqrt(ncol(X))    # Subset and scale markers
  G = tcrossprod(X)                     # Genomic relationship matrix
  y = as.vector(scale(Y[1:200,"YLD"]))  # Subset ans scale response variable
  
  fm = SSI(y,K=G,tst=1:20,trn=21:length(y))
  
  # Basic setting
  plotNet(fm,K=G,bg.color="white",line.color="gray25")
  plotNet(fm,K=G,unified=FALSE)
  
  \donttest{
  # Passing a matrix of coefficients
  B = as.matrix(coef(fm,df=15))
  plotNet(fm,B=B,K=G,curve=TRUE,set.size=c(3.5,1.5,1))
  
  # Using Spectral Value Decomposition and grouping
  EVD = eigen(G)
  gp = data.frame(group=kmeans(EVD$vectors[,1:3],centers=5)$cluster)
  plotNet(fm,curve=TRUE,group=gp,U=EVD$vectors,d=EVD$values)
  }
}
\author{
Marco Lopez-Cruz (\email{maraloc@gmail.com}) and Gustavo de los Campos
}

