% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/poly2poly_ap.R
\name{poly2poly_ap}
\alias{poly2poly_ap}
\title{Area and population weighted polygon-to-polygon interpolation}
\usage{
poly2poly_ap(
  poly_from,
  poly_to,
  poly_to_id,
  geo_vor = NULL,
  methodz = "aw",
  char_methodz = "aw",
  pop_raster = NULL,
  varz = NULL,
  char_varz = NULL,
  char_assign = "biggest_overlap",
  funz = function(x, w) {     stats::weighted.mean(x, w, na.rm = TRUE) },
  seed = 1
)
}
\arguments{
\item{poly_from}{Source polygon layer. \code{sf} object.}

\item{poly_to}{Destination polygon layer. Must have identical CRS to \code{poly_from}. \code{sf} object.}

\item{poly_to_id}{Name of unique ID column for destination polygon layer. Character string.}

\item{geo_vor}{Voronoi polygons object (used internally by \code{point2poly_tess}). \code{sf} object.}

\item{methodz}{Area interpolation method(s). Could be either of "aw" (areal weighting, default) and/or "pw" (population weighting). See "details". Character string or vector of character strings.}

\item{char_methodz}{Interpolation method(s) for character strings. Could be either of "aw" (areal weighting, default) or "pw" (population weighting). See "details". Character string.}

\item{pop_raster}{Population raster to be used for population weighting, Must be supplied if \code{methodz="pw"}. Must have identical CRS to \code{poly_from}. \code{raster} object.}

\item{varz}{Names of numeric variable(s) to be interpolated from source polygon layer to destination polygons. Character string or vector of character strings.}

\item{char_varz}{Names of character string variables to be interpolated from source polygon layer to destination polygons. Character string or vector of character strings.}

\item{char_assign}{Assignment rule to be used for variables specified in \code{char_varz}. Could be either "biggest_overlap" (default) or "all_overlap". See "details". Character string or vector of character strings.}

\item{funz}{Aggregation function to be applied to variables specified in \code{varz}. Must take as an input a numeric vector \code{x} and vector of weights \code{w}. Function or list of functions.}

\item{seed}{Seed for generation of random numbers. Default is 1. Numeric.}
}
\value{
\code{sf} polygon object, with variables from \code{poly_from} interpolated to the geometries of \code{poly_to}.
}
\description{
Function for interpolating values from a source polygon layer to an overlapping (but spatially misaligned) destination polygon layer, using area and/or population weights.
}
\details{
Currently supported integration methods (\code{methodz}) include:
\itemize{
 \item{Areal weighting ("aw"). }{Values from \code{poly_from} weighted in proportion to relative area of spatial overlap between source features and geometries of \code{poly_to}.}
 \item{Population weighting ("pw"). }{Values from \code{poly_from} weighted in proportion to relative population sizes in areas of spatial overlap between source features and geometries of \code{poly_to}. This routine uses a third layer (supplied in \code{pop_raster}) to calculate the weights.}
}
It is possible to pass multiple arguments to \code{methodz} (e.g. \code{methodz=c("aw","pw")}), in which case the function will calculate both sets of weights, and append the resulting columns to the output.

Interpolation procedures are handled somewhat differently for numeric and character string variables. For numeric variables supplied in \code{varz}, "aw" and/or "pw" weights are passed to the function specified in \code{funz}. If different sets of numeric variables are to be aggregated with different functions, both \code{varz} and \code{funz} should be specified as lists (see examples below).

For character string (and any other) variables supplied in \code{char_varz}, "aw" and/or "pw" weights are passed to the assignment rule(s) specified in \code{char_assign}. Note that the \code{char_varz} argument may include numerical variables, but \code{varz} cannot include character string variables.

Currently supported assignment rules for character strings (\code{char_assign}) include:
\itemize{
 \item{"biggest_overlap". }{For each variable in \code{char_varz}, the features in \code{poly_to} are assigned a single value from overlapping \code{poly_from} features, corresponding to the intersection with largest area and/or population weight.}
 \item{"all_overlap". }{For each variable in \code{char_varz}, the features in \code{poly_to} are assigned all values from overlapping \code{poly_from} features, ranked by area and/or population weights (largest-to-smallest) of intersections.}
}
It is possible to pass multiple arguments to \code{char_assign} (e.g. \code{char_assign=c("biggest_overlap","all_overlap")}), in which case the function will calculate both, and append the resulting columns to the output.
}
\examples{
# Interpolation of a single variable, with area weights
\dontrun{
data(clea_deu2009)
data(hex_05_deu)
out_1 <- poly2poly_ap(poly_from = clea_deu2009,
              poly_to = hex_05_deu,
              poly_to_id = "HEX_ID",
              varz = "to1"
             )
}

# Interpolation of multiple variables, with area weights
\dontrun{
out_2 <- poly2poly_ap(
              poly_from = clea_deu2009,
              poly_to = hex_05_deu,
              poly_to_id = "HEX_ID",
              varz = list(
                c("to1","pvs1_margin"),
                c("vv1") ),
              funz = list(
                function(x,w){stats::weighted.mean(x,w)},
                function(x,w){sum(x*w)} ),
              char_varz = c("incumb_pty_n","win1_pty_n")
             )
}

# Interpolation of a single variable, with population weights
\dontrun{
data(gpw4_deu2010)
out_3 <- poly2poly_ap(poly_from = clea_deu2009,
                         poly_to = hex_05_deu,
                         poly_to_id = "HEX_ID",
                         varz = "to1",
                         methodz = "pw",
                         pop_raster = gpw4_deu2010)
}

# Interpolation of a single variable, with area and population weights
\dontrun{
out_4 <- poly2poly_ap(poly_from = clea_deu2009,
                         poly_to = hex_05_deu,
                         poly_to_id = "HEX_ID",
                         varz = "to1",
                         methodz = c("aw","pw"),
                         pop_raster = gpw4_deu2010)
}
}
