% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mergeRaster.R, R/splitRaster.R
\name{mergeRaster}
\alias{mergeRaster}
\alias{mergeRaster,list-method}
\alias{splitRaster}
\alias{splitRaster,RasterLayer-method}
\title{Split and re-merge RasterLayer(s)}
\usage{
mergeRaster(x, fun = NULL)

\S4method{mergeRaster}{list}(x, fun = NULL)

splitRaster(
  r,
  nx = 1,
  ny = 1,
  buffer = c(0, 0),
  path = NA,
  cl,
  rType = "FLT4S"
)

\S4method{splitRaster}{RasterLayer}(
  r,
  nx = 1,
  ny = 1,
  buffer = c(0, 0),
  path = NA,
  cl,
  rType = "FLT4S"
)
}
\arguments{
\item{x}{A list of split raster tiles (i.e., from \code{splitRaster}).}

\item{fun}{Function (e.g. \code{mean}, \code{min}, or \code{max} that
accepts a \code{na.rm} argument. The default is \code{mean}.}

\item{r}{The raster to be split.}

\item{nx}{The number of tiles to make along the x-axis.}

\item{ny}{The number of tiles to make along the y-axis.}

\item{buffer}{Numeric vector of length 2 giving the size of the buffer along the x and y axes.
If these values less than or equal to \code{1} are used, this
is interpreted as the number of pixels (cells) to use as a buffer.
Values between \code{0} and \code{1} are interpreted as proportions
of the number of pixels in each tile (rounded up to an integer value).
Default is \code{c(0, 0)}, which means no buffer.}

\item{path}{Character specifying the directory to which the split tiles will be saved.
If missing, the function will write to memory.}

\item{cl}{A cluster object. Optional. This would generally be created using
parallel::makeCluster or equivalent. This is an alternative way, instead
of \code{beginCluster()}, to use parallelism for this function, allowing for
more control over cluster use.}

\item{rType}{Datatype of the split rasters. Defaults to FLT4S.}
}
\value{
\code{mergeRaster} returns a \code{RasterLayer} object.

\code{splitRaster} returns a list (length \code{nx*ny}) of cropped raster tiles.
}
\description{
\code{splitRaster} divides up a raster into an arbitrary number of pieces (tiles).
Split rasters can be recombined using \code{do.call(merge, y)} or \code{mergeRaster(y)},
where \code{y <- splitRaster(x)}.
}
\details{
\code{mergeRaster} differs from \code{merge} in how overlapping tile
regions are handled: \code{merge} retains the values of the first raster in
the list. This has the consequence of retaining the values from the buffered
region in the first tile in place of the values from the neighbouring tile.
On the other hand, \code{mergeRaster} retains the values of the tile region,
over the values in any buffered regions. This is useful for reducing edge
effects when performing raster operations involving contagious processes.
To use the average of cell values, or do another computation, use
\code{\link[raster]{mosaic}}.
\code{mergeRaster} is also faster than \code{merge}. \code{mergeRaster} also
differs from \code{\link[raster]{mosaic}} in speed and ability to take a
raster list. It can, however, use the average of cell values, or do other
computations. At last, \code{mergeRaster} can also merge tiles of a split
raster that were resampled and, therefore, could have had different changes
in the buffer sizes on each side of the raster. If the user resamples the
tiles and the new resolution is not a multiple of the original one,
\code{mergeRaster} will use mosaic with the max function to merge the tiles
with a message. The user can also pass the function to be used when mosaic
is triggered.

This function is parallel-aware, using the same mechanism as used in the \code{raster}
package. Specifically, if you start a cluster using \code{\link{beginCluster}}, then
this function will automatically use that cluster. It is always a good
idea to stop the cluster when finished, using \code{\link{endCluster}}.
}
\examples{
library(raster)

# an example with dimensions:
# nrow: 77
# ncol: 101
# nlayers: 3
b <- brick(system.file("external/rlogo.grd", package = "raster"))
r <- b[[1]] # use first layer only
nx <- 1
ny <- 2

tmpdir <- file.path(tempdir(), "splitRaster-example")
dir.create(tmpdir)

y0 <- splitRaster(r, nx, ny, path = file.path(tmpdir, "y0")) # no buffer

# buffer: 10 pixels along both axes
y1 <- splitRaster(r, nx, ny, c(10, 10), path = file.path(tmpdir, "y1"))

# buffer: half the width and length of each tile
y2 <- splitRaster(r, nx, ny, c(0.5, 0.5), path = file.path(tmpdir, "y2"))

# parallel cropping
if (interactive()) {
  n <- pmin(parallel::detectCores(), 4) # use up to 4 cores
  beginCluster(n)
  y3 <- splitRaster(r, nx, ny, c(0.7, 0.7), path = file.path(tmpdir, "y3"))
  endCluster()
}

# the original raster:
if (interactive()) plot(r) # may require a call to `dev()` if using RStudio

# the split raster:
layout(mat = matrix(seq_len(nx * ny), ncol = nx, nrow = ny))
plotOrder <- c(4, 8, 12, 3, 7, 11, 2, 6, 10, 1, 5, 9)
if (interactive()) invisible(lapply(y0[plotOrder], plot))

# can be recombined using `raster::merge`
m0 <- do.call(merge, y0)
all.equal(m0, r) ## TRUE

m1 <- do.call(merge, y1)
all.equal(m1, r) ## TRUE

m2 <- do.call(merge, y2)
all.equal(m2, r) ## TRUE

# or recombine using mergeRaster
n0 <- mergeRaster(y0)
all.equal(n0, r) ## TRUE

n1 <- mergeRaster(y1)
all.equal(n1, r) ## TRUE

n2 <- mergeRaster(y2)
all.equal(n2, r) ## TRUE

unlink(tmpdir, recursive = TRUE)
}
\seealso{
\code{\link[raster]{merge}}, \code{\link[raster]{mosaic}}

\code{\link{do.call}}, \code{\link[raster]{merge}}.
}
\author{
Yong Luo, Alex Chubaty, Tati Micheletti & Ian Eddy

Alex Chubaty and Yong Luo
}
